"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssetInventoryDataClient = exports.ASSET_INVENTORY_STATUS = void 0;
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _entity_analytics = require("../../../common/api/entity_analytics");
var _data_view = require("./saved_objects/data_view");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ASSET_INVENTORY_STATUS = exports.ASSET_INVENTORY_STATUS = {
  INACTIVE_FEATURE: 'inactive_feature',
  DISABLED: 'disabled',
  INITIALIZING: 'initializing',
  INSUFFICIENT_PRIVILEGES: 'insufficient_privileges',
  EMPTY: 'empty',
  READY: 'ready'
};

// AssetInventoryDataClient is responsible for managing the asset inventory,
// including initializing and cleaning up resources such as Elasticsearch ingest pipelines.
class AssetInventoryDataClient {
  constructor(options) {
    this.options = options;
  }

  // Initializes the asset inventory by validating experimental feature flags and triggering asynchronous setup.
  async init() {
    const {
      logger
    } = this.options;
    logger.debug(`Initializing asset inventory`);
    this.asyncSetup().catch(e => logger.error(`Error during async setup of asset inventory: ${e.message}`));
  }

  // Sets up the necessary resources for asset inventory.
  async asyncSetup() {
    const {
      logger
    } = this.options;
    try {
      logger.debug('Initializing asset inventory');
    } catch (err) {
      logger.error(`Error initializing asset inventory: ${err.message}`);
      await this.delete();
    }
  }

  // Checks if the Asset Inventory DataView exists, if not, installs it
  async installAssetInventoryDataView(secSolutionContext) {
    const {
      logger
    } = this.options;
    const dataViewService = secSolutionContext.getDataViewsService();
    const currentSpaceId = secSolutionContext.getSpaceId();
    const currentSpaceDataViewId = `${_constants.ASSET_INVENTORY_DATA_VIEW_ID_PREFIX}-${currentSpaceId}`;
    let dataViewExists = false;
    try {
      logger.debug(`Checking if data view exists: ${currentSpaceDataViewId}`);
      await dataViewService.get(currentSpaceDataViewId, false);
      dataViewExists = true;
    } catch (error) {
      logger.error(`Error getting data view: ${error}`);
      if (error && typeof error === 'object' && 'output' in error && 'statusCode' in error.output && error.output.statusCode === 404) {
        logger.info(`DataView with ID '${currentSpaceDataViewId}' not found. Proceeding with installation.`);
        dataViewExists = false; // Confirm it doesn't exist
      } else {
        logger.error('An unexpected error occurred while checking data view existence:', error);
      }
    }
    if (!dataViewExists) {
      logger.debug('Installing Asset Inventory DataView');
      return (0, _data_view.installDataView)(currentSpaceId, dataViewService, _constants.ASSET_INVENTORY_DATA_VIEW_NAME, _constants.ASSET_INVENTORY_INDEX_PATTERN, _constants.ASSET_INVENTORY_DATA_VIEW_ID_PREFIX, logger);
    } else {
      logger.debug('DataView is already installed. Skipping installation.');
    }
  }

  // Enables the asset inventory by deferring the initialization to avoid blocking the main thread.
  async enable(secSolutionContext, requestBodyOverrides) {
    const {
      logger
    } = this.options;
    logger.debug(`Enabling asset inventory`);
    try {
      if (!(await this.checkUISettingEnabled())) {
        throw new Error('uiSetting');
      }

      // Retrieve entity store status
      const entityStoreStatus = await secSolutionContext.getEntityStoreDataClient().status({
        include_components: true
      });
      const entityEngineStatus = entityStoreStatus.status;
      let entityStoreEnablementResponse;
      // If the entity store is not installed, we need to install it.
      if (entityEngineStatus === 'not_installed') {
        entityStoreEnablementResponse = await secSolutionContext.getEntityStoreDataClient().enable(requestBodyOverrides);
      } else {
        // If the entity store is already installed, we need to check if the generic engine is installed.
        const genericEntityEngine = entityStoreStatus.engines.find(this.isGenericEntityEngine);

        // If the generic engine is not installed or is stopped, we need to start it.
        if (!genericEntityEngine) {
          entityStoreEnablementResponse = await secSolutionContext.getEntityStoreDataClient()
          // @ts-ignore-next-line TS2345
          .init(_entity_analytics.EntityType.enum.generic, requestBodyOverrides);
        }
      }
      try {
        await (0, _data_view.installDataView)(secSolutionContext.getSpaceId(), secSolutionContext.getDataViewsService(), _constants.ASSET_INVENTORY_DATA_VIEW_NAME, _constants.ASSET_INVENTORY_INDEX_PATTERN, _constants.ASSET_INVENTORY_DATA_VIEW_ID_PREFIX, logger);
      } catch (error) {
        logger.error(`Error installing asset inventory data view: ${error.message}`);
      }
      logger.debug(`Enabled asset inventory`);
      return entityStoreEnablementResponse;
    } catch (err) {
      logger.error(`Error enabling asset inventory: ${err.message}`);
      throw err;
    }
  }

  // Cleans up the resources associated with the asset inventory, such as removing the ingest pipeline.
  async delete() {
    const {
      logger
    } = this.options;
    logger.debug(`Deleting asset inventory`);
    try {
      if (!(await this.checkUISettingEnabled())) {
        throw new Error('uiSetting');
      }
      logger.debug(`Deleted asset inventory`);
      return {
        deleted: true
      };
    } catch (err) {
      logger.error(`Error deleting asset inventory: ${err.message}`);
      throw err;
    }
  }
  async status(secSolutionContext, entityStorePrivileges) {
    const {
      logger
    } = this.options;
    if (!(await this.checkUISettingEnabled())) {
      return {
        status: ASSET_INVENTORY_STATUS.INACTIVE_FEATURE
      };
    }

    // Check if the user has the required privileges to access the entity store.
    if (!entityStorePrivileges.has_all_required) {
      try {
        const hasGenericDocuments = await this.hasGenericDocuments(secSolutionContext);
        // check if users doesn't have entity store privileges but generic documents are present
        if (hasGenericDocuments) {
          try {
            await this.installAssetInventoryDataView(secSolutionContext);
          } catch (error) {
            logger.error(`Error installing asset inventory data view: ${error.message}`);
          }
          return {
            status: ASSET_INVENTORY_STATUS.READY
          };
        }
      } catch (error) {
        logger.error(`Error checking for generic documents: ${error.message}`);
      }
      return {
        status: ASSET_INVENTORY_STATUS.INSUFFICIENT_PRIVILEGES,
        privileges: entityStorePrivileges
      };
    }

    // Retrieve entity store status
    const entityStoreStatus = await secSolutionContext.getEntityStoreDataClient().status({
      include_components: true
    });
    const entityEngineStatus = entityStoreStatus.status;

    // Determine the asset inventory status based on the entity engine status
    if (entityEngineStatus === 'not_installed') {
      return {
        status: ASSET_INVENTORY_STATUS.DISABLED
      };
    }
    if (entityEngineStatus === 'installing') {
      return {
        status: ASSET_INVENTORY_STATUS.INITIALIZING
      };
    }

    // Check for the Generic entity engine
    const genericEntityEngine = entityStoreStatus.engines.find(this.isGenericEntityEngine);
    // If the generic engine is not installed, the asset inventory is disabled.
    if (!genericEntityEngine) {
      return {
        status: ASSET_INVENTORY_STATUS.DISABLED
      };
    }

    // Determine final status based on transform metadata
    if (this.hasDocumentsProcessed(genericEntityEngine)) {
      await this.installAssetInventoryDataView(secSolutionContext);
      return {
        status: ASSET_INVENTORY_STATUS.READY
      };
    }
    if (this.hasTransformTriggered(genericEntityEngine)) {
      return {
        status: ASSET_INVENTORY_STATUS.EMPTY
      };
    }

    // If the engine is still initializing, return the initializing status
    return {
      status: ASSET_INVENTORY_STATUS.INITIALIZING
    };
  }
  async checkUISettingEnabled() {
    const {
      uiSettingsClient,
      logger
    } = this.options;
    const isAssetInventoryEnabled = await uiSettingsClient.get(_managementSettingsIds.SECURITY_SOLUTION_ENABLE_ASSET_INVENTORY_SETTING);
    if (!isAssetInventoryEnabled) {
      logger.debug(`${_managementSettingsIds.SECURITY_SOLUTION_ENABLE_ASSET_INVENTORY_SETTING} advanced setting is disabled`);
    }
    return isAssetInventoryEnabled;
  }

  // Type guard to check if an entity engine is a generic entity engine
  isGenericEntityEngine(engine) {
    return engine.type === 'generic';
  }

  // Type guard function to validate entity store component metadata
  isTransformMetadata(metadata) {
    return typeof metadata === 'object' && metadata !== null && 'documents_processed' in metadata && 'trigger_count' in metadata && typeof metadata.documents_processed === 'number' && typeof metadata.trigger_count === 'number';
  }
  hasDocumentsProcessed(engine) {
    var _engine$components;
    return !!((_engine$components = engine.components) !== null && _engine$components !== void 0 && _engine$components.some(component => {
      if (component.resource === 'transform' && this.isTransformMetadata(component.metadata)) {
        return component.metadata.documents_processed > 0;
      }
      return false;
    }));
  }
  hasTransformTriggered(engine) {
    var _engine$components2;
    return !!((_engine$components2 = engine.components) !== null && _engine$components2 !== void 0 && _engine$components2.some(component => {
      if (component.resource === 'transform' && this.isTransformMetadata(component.metadata)) {
        return component.metadata.trigger_count > 0;
      }
      return false;
    }));
  }
  async hasGenericDocuments(secSolutionContext) {
    const elasticsearchClient = secSolutionContext.core.elasticsearch.client;
    const spaceId = secSolutionContext.getSpaceId();
    const genericIndexCurrentSpace = `${_constants.ASSET_INVENTORY_GENERIC_INDEX_PREFIX}${spaceId}`;
    const response = await elasticsearchClient.asInternalUser.count({
      index: genericIndexCurrentSpace
    });
    return response.count > 0;
  }
}
exports.AssetInventoryDataClient = AssetInventoryDataClient;