"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EndpointFleetServicesFactory = void 0;
var _server = require("@kbn/fleet-plugin/server");
var _common = require("@kbn/fleet-plugin/common");
var _errors = require("@kbn/fleet-plugin/server/errors");
var _common2 = require("@kbn/spaces-plugin/common");
var _errors2 = require("../../../../common/endpoint/errors");
var _utils = require("../../utils");
var _stringify = require("../../utils/stringify");
var _errors3 = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The set of Fleet services used by Endpoint
 */

/**
 * Provides centralized way to get all services for Fleet and access internal saved object clients
 */
class EndpointFleetServicesFactory {
  constructor(fleetDependencies, savedObjects, logger) {
    this.fleetDependencies = fleetDependencies;
    this.savedObjects = savedObjects;
    this.logger = logger;
  }
  asInternalUser(spaceId, unscoped = false) {
    this.logger.debug(`creating set of fleet services with spaceId [${spaceId}] and unscoped [${unscoped}]`);
    if (spaceId && unscoped) {
      throw new _errors2.EndpointError(`asInternalUser(): a 'spaceId' can not be set when 'unscoped' is 'true'`);
    }
    const {
      agentPolicyService: agentPolicy,
      packagePolicyService: packagePolicy,
      agentService,
      packageService
    } = this.fleetDependencies;
    const agent = spaceId ? agentService.asInternalScopedUser(spaceId) : agentService.asInternalUser;

    // Lazily Initialized at the time it is needed
    let _soClient;
    const getSoClient = () => {
      if (!_soClient) {
        if (unscoped) {
          this.logger.debug(`getSoClient(): initializing UNSCOPED SO client`);
          _soClient = this.savedObjects.createInternalUnscopedSoClient(false);
        } else {
          this.logger.debug(`getSoClient(): initializing SO client for space [${spaceId}]`);
          _soClient = this.savedObjects.createInternalScopedSoClient({
            spaceId,
            readonly: false
          });
        }
      }
      return _soClient;
    };
    const ensureInCurrentSpace = async ({
      integrationPolicyIds = [],
      agentPolicyIds = [],
      agentIds = [],
      options
    }) => {
      this.logger.debug(`EnsureInCurrentSpace(): Checking access for space [${spaceId}]`);
      return checkInCurrentSpace({
        soClient: getSoClient(),
        agentService: agent,
        agentPolicyService: agentPolicy,
        packagePolicyService: packagePolicy,
        options,
        integrationPolicyIds,
        agentPolicyIds,
        agentIds
      });
    };
    const getPolicyNamespace = async options => {
      return fetchIntegrationPolicyNamespace({
        ...options,
        soClient: getSoClient(),
        logger: this.logger,
        packagePolicyService: packagePolicy,
        agentPolicyService: agentPolicy,
        // When using an unscoped soClient, make sure the search for policies is done across all spaces.
        spaceId: unscoped ? '*' : undefined
      });
    };
    const getIntegrationNamespaces = async integrationNames => {
      return fetchIntegrationNamespaces({
        soClient: getSoClient(),
        logger: this.logger,
        packagePolicyService: packagePolicy,
        agentPolicyService: agentPolicy,
        integrationNames
      });
    };
    const isEndpointPackageInstalled = async () => {
      const installedEndpointPackages = await packageService.asInternalUser.getInstalledPackages({
        nameQuery: 'endpoint',
        perPage: 1000,
        sortOrder: 'asc'
      });
      return installedEndpointPackages.items.some(pkg => pkg.name === 'endpoint');
    };
    return {
      spaceId: spaceId || _common2.DEFAULT_SPACE_ID,
      logger: this.logger,
      agent,
      agentPolicy,
      packages: packageService.asInternalUser,
      packagePolicy,
      savedObjects: this.savedObjects,
      endpointPolicyKuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: "endpoint"`,
      ensureInCurrentSpace,
      getPolicyNamespace,
      getIntegrationNamespaces,
      getSoClient,
      isEndpointPackageInstalled
    };
  }
}
exports.EndpointFleetServicesFactory = EndpointFleetServicesFactory;
/**
 * Checks if data provided (integration policies, agent policies and/or agentIds) are visible in
 * current space
 *
 * @param soClient
 * @param agentService
 * @param agentPolicyService
 * @param packagePolicyService
 * @param integrationPolicyIds
 * @param agentPolicyIds
 * @param agentIds
 * @param options
 *
 * @throws NotFoundError
 */
const checkInCurrentSpace = async ({
  soClient,
  agentService,
  agentPolicyService,
  packagePolicyService,
  integrationPolicyIds = [],
  agentPolicyIds = [],
  agentIds = [],
  options: {
    matchAll = true
  } = {}
}) => {
  const handlePromiseErrors = err => {
    // We wrap the error with our own Error class so that the API can property return a 404
    if (err instanceof _server.AgentNotFoundError || err instanceof _errors.AgentPolicyNotFoundError || err instanceof _errors.PackagePolicyNotFoundError) {
      throw new _errors3.NotFoundError(err.message, err);
    }
    throw err;
  };
  await Promise.all([agentIds.length ? agentService.getByIds(agentIds, {
    ignoreMissing: !matchAll
  }).catch(handlePromiseErrors).then(response => {
    // When `matchAll` is false, the results must have at least matched 1 id
    if (!matchAll && response.length === 0) {
      throw new _errors3.NotFoundError(`Agent ID(s) not found: [${agentIds.join(', ')}]`);
    }
  }) : null, agentPolicyIds.length ? agentPolicyService.getByIds(soClient, agentPolicyIds, {
    ignoreMissing: !matchAll
  }).catch(handlePromiseErrors).then(response => {
    if (!matchAll && response.length === 0) {
      throw new _errors3.NotFoundError(`Agent policy ID(s) not found: [${agentPolicyIds.join(', ')}]`);
    }
  }) : null, integrationPolicyIds.length ? packagePolicyService.getByIDs(soClient, integrationPolicyIds, {
    ignoreMissing: !matchAll
  }).catch(handlePromiseErrors).then(response => {
    if (!matchAll && response.length === 0) {
      throw new _errors3.NotFoundError(`Integration policy ID(s) not found: [${integrationPolicyIds.join(', ')}]`);
    }
  }) : null]);
};
const fetchIntegrationPolicyNamespace = async ({
  logger,
  soClient,
  packagePolicyService,
  agentPolicyService,
  integrationPolicies,
  spaceId
}) => {
  const response = {
    integrationPolicy: {}
  };
  const agentPolicyIdsToRetrieve = new Set();
  const retrievedIntegrationPolicies = {};
  const retrievedAgentPolicies = {};
  if (integrationPolicies.length > 0) {
    var _await$packagePolicyS;
    logger.debug(() => `Retrieving package policies from fleet for:\n${(0, _stringify.stringify)(integrationPolicies)}`);
    const packagePolicies = (_await$packagePolicyS = await packagePolicyService.getByIDs(soClient, integrationPolicies, {
      spaceIds: spaceId ? [spaceId] : undefined
    }).catch(_utils.catchAndWrapError)) !== null && _await$packagePolicyS !== void 0 ? _await$packagePolicyS : [];
    logger.trace(() => `Fleet package policies retrieved:\n${(0, _stringify.stringify)(packagePolicies)}`);
    for (const packagePolicy of packagePolicies) {
      retrievedIntegrationPolicies[packagePolicy.id] = packagePolicy;

      // Integration policy does not have an explicit namespace, which means it
      // inherits it from the associated agent policies, so lets retrieve those
      if (!packagePolicy.namespace) {
        packagePolicy.policy_ids.forEach(agentPolicyId => {
          agentPolicyIdsToRetrieve.add(agentPolicyId);
        });
      }
    }
  }
  if (agentPolicyIdsToRetrieve.size > 0) {
    const ids = Array.from(agentPolicyIdsToRetrieve);
    logger.debug(() => `Retrieving agent policies from fleet for:\n${(0, _stringify.stringify)(ids)}`);
    const agentPolicies = await agentPolicyService.getByIds(soClient, ids, {
      spaceId
    }).catch(_utils.catchAndWrapError);
    logger.trace(() => `Fleet agent policies retrieved:\n${(0, _stringify.stringify)(agentPolicies)}`);
    for (const agentPolicy of agentPolicies) {
      retrievedAgentPolicies[agentPolicy.id] = agentPolicy;
    }
  }
  for (const integrationPolicyId of integrationPolicies) {
    const integrationPolicyNamespace = retrievedIntegrationPolicies[integrationPolicyId].namespace;
    response.integrationPolicy[integrationPolicyId] = integrationPolicyNamespace ? [integrationPolicyNamespace] : retrievedIntegrationPolicies[integrationPolicyId].policy_ids.map(agentPolicyId => {
      return retrievedAgentPolicies[agentPolicyId].namespace;
    });
  }
  logger.debug(() => `Policy namespaces:\n${(0, _stringify.stringify)(response)}`);
  return response;
};
const fetchIntegrationNamespaces = async ({
  logger,
  soClient,
  packagePolicyService,
  agentPolicyService,
  integrationNames = []
}) => {
  const integrationToNamespaceMap = integrationNames.reduce((acc, name) => {
    acc[name] = new Set();
    return acc;
  }, {});
  const agentPolicyIdsToRetrieve = {};
  if (integrationNames.length > 0) {
    const kuery = `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: (${integrationNames.join(' OR ')})`;
    logger.debug(() => `Fetch of policies for integrations using Kuery [${kuery}]`);
    const policiesFound = await packagePolicyService.list(soClient, {
      perPage: 10_000,
      kuery
    }).catch(_utils.catchAndWrapError);
    logger.trace(() => `Fetch of policies for integrations using Kuery [${kuery}] returned:\n${(0, _stringify.stringify)(policiesFound)}`);
    for (const packagePolicy of policiesFound.items) {
      var _packagePolicy$packag;
      if ((_packagePolicy$packag = packagePolicy.package) !== null && _packagePolicy$packag !== void 0 && _packagePolicy$packag.name) {
        const integrationName = packagePolicy.package.name;
        if (packagePolicy.namespace) {
          integrationToNamespaceMap[integrationName].add(packagePolicy.namespace);
        } else {
          // Integration policy does not have an explicit namespace, which means it
          // inherits it from the associated agent policies. We'll retrieve these next
          packagePolicy.policy_ids.forEach(agentPolicyId => {
            if (!agentPolicyIdsToRetrieve[agentPolicyId]) {
              agentPolicyIdsToRetrieve[agentPolicyId] = new Set();
            }
            agentPolicyIdsToRetrieve[agentPolicyId].add(integrationToNamespaceMap[integrationName]);
          });
        }
      }
    }
  }
  const agentPolicyIds = Object.keys(agentPolicyIdsToRetrieve);
  if (agentPolicyIds.length > 0) {
    logger.debug(() => `Retrieving agent policies from fleet for:\n${(0, _stringify.stringify)(agentPolicyIds)}`);
    const agentPolicies = await agentPolicyService.getByIds(soClient, agentPolicyIds).catch(_utils.catchAndWrapError);
    logger.trace(() => `Fleet agent policies retrieved:\n${(0, _stringify.stringify)(agentPolicies)}`);
    for (const agentPolicy of agentPolicies) {
      for (const nameSpaceSet of agentPolicyIdsToRetrieve[agentPolicy.id]) {
        nameSpaceSet.add(agentPolicy.namespace);
      }
    }
  }
  const response = Object.entries(integrationToNamespaceMap).reduce((acc, [integrationName, namespaceSet]) => {
    acc[integrationName] = Array.from(namespaceSet.values());
    return acc;
  }, {});
  logger.debug(() => `Integration namespaces in use:\n${(0, _stringify.stringify)(response)}`);
  return response;
};