"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ensureUserHasAuthzToFilesForAction = void 0;
var _std = require("@kbn/std");
var _lodash = require("lodash");
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _is_response_action_supported = require("../../../../common/endpoint/service/response_actions/is_response_action_supported");
var _errors = require("../../errors");
var _fetch_action_request_by_id = require("../../services/actions/utils/fetch_action_request_by_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// FYI: this object here should help to quickly catch instances where we might forget to update the
//      authz on the file info/download apis when a response action needs to support file downloads.
const COMMANDS_WITH_ACCESS_TO_FILES = (0, _std.deepFreeze)({
  'get-file': {
    endpoint: true,
    sentinel_one: true,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  execute: {
    endpoint: true,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  'running-processes': {
    endpoint: false,
    sentinel_one: true,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  upload: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  scan: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  isolate: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  unisolate: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  'kill-process': {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  'suspend-process': {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  runscript: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: true
  }
});

/**
 * Checks to ensure that the user has the correct authz for the response action associated with the action id.
 *
 * FYI: Additional check is needed because the File info and download APIs are used by multiple response actions,
 *      thus we want to ensure that we don't allow access to file associated with response actions the user does
 *      not have authz to.
 *
 * @param context
 * @param request
 */
const ensureUserHasAuthzToFilesForAction = async (context, request) => {
  const securitySolution = await context.securitySolution;
  const spaceId = securitySolution.getSpaceId();
  const endpointService = securitySolution.getEndpointService();
  const userAuthz = await securitySolution.getEndpointAuthz();
  const logger = endpointService.createLogger('ensureUserHasAuthzToFilesForAction');
  const {
    action_id: actionId
  } = request.params;
  logger.debug(`Validating action id [${actionId}] has access to files in space [${spaceId}]`);
  const {
    EndpointActions: {
      data: {
        command
      },
      input_type: agentType
    }
  } = await (0, _fetch_action_request_by_id.fetchActionRequestById)(endpointService, spaceId, actionId);
  logger.debug(`Action [${actionId}] is for command [${command}] with agentType [${agentType}]`);

  // Check if command is supported by the agent type
  if (!(0, _is_response_action_supported.isActionSupportedByAgentType)(agentType, command, 'manual')) {
    throw new _custom_http_request_error.CustomHttpRequestError(`Response action [${command}] not supported for agent type [${agentType}]`, 400);
  }

  // Check if the command is marked as having access to files
  if (!(0, _lodash.get)(COMMANDS_WITH_ACCESS_TO_FILES, `${command}.${agentType}`, false)) {
    throw new _custom_http_request_error.CustomHttpRequestError(`Response action [${command}] for agent type [${agentType}] does not support file downloads`, 400);
  }
  let hasAuthzToCommand = false;
  switch (command) {
    case 'get-file':
      hasAuthzToCommand = userAuthz.canWriteFileOperations;
      break;
    case 'execute':
      hasAuthzToCommand = userAuthz.canWriteExecuteOperations;
      break;
    case 'running-processes':
      hasAuthzToCommand = userAuthz.canGetRunningProcesses;
      break;
    case 'runscript':
      hasAuthzToCommand = userAuthz.canWriteExecuteOperations;
      break;
  }
  if (!hasAuthzToCommand) {
    throw new _errors.EndpointAuthorizationError();
  }
};
exports.ensureUserHasAuthzToFilesForAction = ensureUserHasAuthzToFilesForAction;