"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAstContext = getAstContext;
exports.isMarkerNode = isMarkerNode;
exports.removeMarkerArgFromArgsList = removeMarkerArgFromArgsList;
var _esqlAst = require("@kbn/esql-ast");
var _types = require("../definitions/types");
var _constants = require("./constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function findCommand(ast, offset) {
  const commandIndex = ast.findIndex(({
    location
  }) => location.min <= offset && location.max >= offset);
  const command = ast[commandIndex] || ast[ast.length - 1];
  return command;
}
function findOption(nodes, offset) {
  return findCommandSubType(nodes, offset, _helpers.isOptionItem);
}
function findCommandSubType(nodes, offset, isOfTypeFn) {
  for (const node of nodes) {
    if (isOfTypeFn(node)) {
      if (node.location.min <= offset && node.location.max >= offset || nodes[nodes.length - 1] === node && node.location.max < offset) {
        return node;
      }
    }
  }
}
function isMarkerNode(node) {
  if (Array.isArray(node)) {
    return false;
  }
  return Boolean(node && ((0, _helpers.isColumnItem)(node) || (0, _esqlAst.isIdentifier)(node) || (0, _helpers.isSourceItem)(node)) && node.name.endsWith(_constants.EDITOR_MARKER));
}
function cleanMarkerNode(node) {
  return isMarkerNode(node) ? undefined : node;
}
function isNotMarkerNodeOrArray(arg) {
  return Array.isArray(arg) || !isMarkerNode(arg);
}
function mapToNonMarkerNode(arg) {
  return Array.isArray(arg) ? arg.filter(isNotMarkerNodeOrArray).map(mapToNonMarkerNode) : arg;
}
function removeMarkerArgFromArgsList(node) {
  if (!node) {
    return;
  }
  if (node.type === 'command' || node.type === 'option' || node.type === 'function') {
    return {
      ...node,
      args: node.args.filter(isNotMarkerNodeOrArray).map(mapToNonMarkerNode)
    };
  }
  return node;
}
const removeMarkerNode = node => {
  _esqlAst.Walker.walk(node, {
    visitAny: current => {
      if ('args' in current) {
        current.args = current.args.filter(n => !isMarkerNode(n));
      } else if ('values' in current) {
        current.values = current.values.filter(n => !isMarkerNode(n));
      }
    }
  });
};
function findAstPosition(ast, offset) {
  const command = findCommand(ast, offset);
  if (!command) {
    return {
      command: undefined,
      node: undefined
    };
  }
  let containingFunction;
  let node;
  _esqlAst.Walker.walk(command, {
    visitSource: (_node, parent, walker) => {
      if (_node.location.max >= offset && _node.text !== _constants.EDITOR_MARKER) {
        node = _node;
        walker.abort();
      }
    },
    visitAny: _node => {
      var _node$location;
      if (_node.type === 'function' && _node.subtype === 'variadic-call' && ((_node$location = _node.location) === null || _node$location === void 0 ? void 0 : _node$location.max) >= offset) {
        containingFunction = _node;
      }
      if (_node.location.max >= offset && _node.text !== _constants.EDITOR_MARKER && (!(0, _esqlAst.isList)(_node) || _node.subtype !== 'tuple')) {
        node = _node;
      }
    }
  });
  if (node) removeMarkerNode(node);
  return {
    command: removeMarkerArgFromArgsList(command),
    containingFunction: removeMarkerArgFromArgsList(containingFunction),
    option: removeMarkerArgFromArgsList(findOption(command.args, offset)),
    node: removeMarkerArgFromArgsList(cleanMarkerNode(node))
  };
}
function isNotEnrichClauseAssigment(node, command) {
  return node.name !== '=' && command.name !== 'enrich';
}
function isOperator(node) {
  var _getFunctionDefinitio;
  return ((_getFunctionDefinitio = (0, _helpers.getFunctionDefinition)(node.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === _types.FunctionDefinitionTypes.OPERATOR;
}

/**
 * Given a ES|QL query string, its AST and the cursor position,
 * it returns the type of context for the position ("list", "function", "option", "setting", "expression", "newCommand")
 * plus the whole hierarchy of nodes (command, option, setting and actual position node) context.
 *
 * Type details:
 * * "list": the cursor is inside a "in" list of values (i.e. `a in (1, 2, <here>)`)
 * * "function": the cursor is inside a function call (i.e. `fn(<here>)`)
 * * "expression": the cursor is inside a command expression (i.e. `command ... <here>` or `command a = ... <here>`)
 * * "newCommand": the cursor is at the beginning of a new command (i.e. `command1 | command2 | <here>`)
 */
function getAstContext(queryString, ast, offset) {
  let inComment = false;
  _esqlAst.Walker.visitComments(ast, node => {
    // if the cursor (offset) is within the range of a comment node
    if ((0, _helpers.within)(offset, node.location)) {
      inComment = true;
      // or if the cursor (offset) is right after a single-line comment (which means there was no newline)
    } else if (node.subtype === 'single-line' && node.location && offset === node.location.max + 1) {
      inComment = true;
    }
  });
  if (inComment) {
    return {
      type: 'comment'
    };
  }
  let withinStatsWhereClause = false;
  _esqlAst.Walker.walk(ast, {
    visitFunction: fn => {
      if (fn.name === 'where' && (0, _helpers.within)(offset, fn.location)) {
        withinStatsWhereClause = true;
      }
    }
  });
  const {
    command,
    option,
    node,
    containingFunction
  } = findAstPosition(ast, offset);
  if (node) {
    if (node.type === 'literal' && node.literalType === 'keyword') {
      // command ... "<here>"
      return {
        type: 'value',
        command,
        node,
        option,
        containingFunction
      };
    }
    if (node.type === 'function') {
      if (['in', 'not in'].includes(node.name)) {
        // command ... a in ( <here> )
        return {
          type: 'list',
          command,
          node,
          option,
          containingFunction
        };
      }
      if (isNotEnrichClauseAssigment(node, command) && (!isOperator(node) || command.name === 'stats' && !withinStatsWhereClause)) {
        // command ... fn( <here> )
        return {
          type: 'function',
          command,
          node,
          option,
          containingFunction
        };
      }
    }
  }
  if (!command || queryString.length <= offset && (0, _helpers.pipePrecedesCurrentWord)(queryString) && command.location.max < queryString.length) {
    //   // ... | <here>
    return {
      type: 'newCommand',
      command: undefined,
      node,
      option,
      containingFunction
    };
  }

  // command a ... <here> OR command a = ... <here>
  return {
    type: 'expression',
    command,
    containingFunction,
    option,
    node
  };
}