"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createWorkloadAggregator = createWorkloadAggregator;
exports.estimateRecurringTaskScheduling = estimateRecurringTaskScheduling;
exports.padBuckets = padBuckets;
exports.summarizeWorkloadStat = summarizeWorkloadStat;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _intervals = require("../lib/intervals");
var _monitoring_stats_stream = require("./monitoring_stats_stream");
var _task = require("../task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// The type of a bucket in the scheduleDensity range aggregation

// @ts-expect-error cannot infer histogram

// Set an upper bound just in case a customer sets a really high refresh rate
const MAX_SCHEDULE_DENSITY_BUCKETS = 50;
function createWorkloadAggregator({
  taskStore,
  elasticsearchAndSOAvailability$,
  refreshInterval,
  pollInterval,
  logger,
  taskDefinitions
}) {
  // calculate scheduleDensity going two refreshIntervals or 1 minute into into the future
  // (the longer of the two)
  const scheduleDensityBuckets = Math.min(Math.max(Math.round(60000 / pollInterval), Math.round(refreshInterval * 2 / pollInterval)), MAX_SCHEDULE_DENSITY_BUCKETS);
  const totalNumTaskDefinitions = taskDefinitions.getAllTypes().length;
  const taskTypeTermAggSize = Math.min(totalNumTaskDefinitions, 10000);
  return (0, _rxjs.combineLatest)([(0, _rxjs.timer)(0, refreshInterval), elasticsearchAndSOAvailability$]).pipe((0, _rxjs.filter)(([, areElasticsearchAndSOAvailable]) => areElasticsearchAndSOAvailable), (0, _rxjs.mergeMap)(() => taskStore.aggregate({
    aggs: {
      taskType: {
        terms: {
          size: taskTypeTermAggSize,
          field: 'task.taskType'
        },
        aggs: {
          status: {
            terms: {
              field: 'task.status'
            }
          }
        }
      },
      schedule: {
        terms: {
          field: 'task.schedule.interval',
          size: 100
        }
      },
      nonRecurringTasks: {
        missing: {
          field: 'task.schedule.interval'
        },
        aggs: {
          taskType: {
            terms: {
              size: taskTypeTermAggSize,
              field: 'task.taskType'
            }
          }
        }
      },
      idleTasks: {
        filter: {
          term: {
            'task.status': 'idle'
          }
        },
        aggs: {
          scheduleDensity: {
            // create a window of upcoming tasks
            range: {
              field: 'task.runAt',
              ranges: [{
                // @ts-expect-error type regression introduced by https://github.com/elastic/elasticsearch-specification/pull/2552
                from: `now`,
                // @ts-expect-error type regression introduced by https://github.com/elastic/elasticsearch-specification/pull/2552
                to: `now+${(0, _intervals.asInterval)(scheduleDensityBuckets * pollInterval)}`
              }]
            },
            aggs: {
              // create histogram of scheduling in the window, with each bucket being a polling interval
              histogram: {
                date_histogram: {
                  field: 'task.runAt',
                  fixed_interval: (0, _intervals.asInterval)(pollInterval)
                },
                // break down each bucket in the histogram by schedule
                aggs: {
                  interval: {
                    terms: {
                      field: 'task.schedule.interval'
                    }
                  }
                }
              }
            }
          },
          overdue: {
            filter: {
              range: {
                'task.runAt': {
                  lt: 'now'
                }
              }
            },
            aggs: {
              taskTypes: {
                terms: {
                  size: taskTypeTermAggSize,
                  field: 'task.taskType'
                }
              },
              nonRecurring: {
                missing: {
                  field: 'task.schedule.interval'
                }
              }
            }
          }
        }
      }
    }
  })), (0, _rxjs.map)(result => {
    var _total$value;
    const {
      aggregations,
      hits: {
        total
      }
    } = result;
    const count = typeof total === 'number' ? total : (_total$value = total === null || total === void 0 ? void 0 : total.value) !== null && _total$value !== void 0 ? _total$value : 0;
    if (!hasAggregations(aggregations)) {
      throw new Error(`Invalid workload: ${JSON.stringify(result)}`);
    }
    const taskTypes = aggregations.taskType.buckets;
    const nonRecurring = aggregations.nonRecurringTasks.doc_count;
    const nonRecurringTaskTypes = aggregations.nonRecurringTasks.taskType.buckets;
    const {
      overdue: {
        doc_count: overdue,
        taskTypes: {
          buckets: taskTypesOverdue = []
        } = {},
        nonRecurring: {
          doc_count: overdueNonRecurring
        }
      },
      scheduleDensity: {
        buckets: [scheduleDensity] = []
      } = {}
    } = aggregations.idleTasks;
    const {
      schedules,
      cadence
    } = aggregations.schedule.buckets.reduce((accm, schedule) => {
      const parsedSchedule = {
        interval: schedule.key,
        asSeconds: (0, _intervals.parseIntervalAsSecond)(schedule.key),
        count: schedule.doc_count
      };
      accm.schedules.push(parsedSchedule);
      if (parsedSchedule.asSeconds <= 60) {
        accm.cadence.perMinute += parsedSchedule.count * Math.round(60 / parsedSchedule.asSeconds);
      } else if (parsedSchedule.asSeconds <= 3600) {
        accm.cadence.perHour += parsedSchedule.count * Math.round(3600 / parsedSchedule.asSeconds);
      } else {
        accm.cadence.perDay += parsedSchedule.count * Math.round(3600 * 24 / parsedSchedule.asSeconds);
      }
      return accm;
    }, {
      cadence: {
        perMinute: 0,
        perHour: 0,
        perDay: 0
      },
      schedules: []
    });
    const totalNonRecurringCost = getTotalCost(nonRecurringTaskTypes, taskDefinitions);
    const totalOverdueCost = getTotalCost(taskTypesOverdue, taskDefinitions);
    let totalCost = 0;
    const taskTypeSummary = taskTypes.reduce((acc, bucket) => {
      const value = bucket;
      const taskDef = taskDefinitions.get(value.key);
      if (taskDef) {
        var _ref;
        const cost = (_ref = value.doc_count * (taskDef === null || taskDef === void 0 ? void 0 : taskDef.cost)) !== null && _ref !== void 0 ? _ref : _task.TaskCost.Normal;
        totalCost += cost;
        return Object.assign(acc, {
          [value.key]: {
            count: value.doc_count,
            cost,
            status: (0, _lodash.mapValues)((0, _lodash.keyBy)(value.status.buckets, 'key'), 'doc_count')
          }
        });
      } else {
        // task type is not registered with dictionary, do not add to summary
        return acc;
      }
    }, {});
    const summary = {
      count,
      cost: totalCost,
      task_types: taskTypeSummary,
      non_recurring: nonRecurring,
      non_recurring_cost: totalNonRecurringCost,
      schedule: schedules.sort((scheduleLeft, scheduleRight) => scheduleLeft.asSeconds - scheduleRight.asSeconds).map(schedule => [schedule.interval, schedule.count]),
      overdue,
      overdue_cost: totalOverdueCost,
      overdue_non_recurring: overdueNonRecurring,
      estimated_schedule_density: padBuckets(scheduleDensityBuckets, pollInterval, scheduleDensity),
      capacity_requirements: {
        per_minute: cadence.perMinute,
        per_hour: cadence.perHour,
        per_day: cadence.perDay
      }
    };
    return {
      key: 'workload',
      value: summary
    };
  }), (0, _rxjs.catchError)((ex, caught) => {
    logger.error(`[WorkloadAggregator]: ${ex}`);
    // continue to pull values from the same observable but only on the next refreshInterval
    return (0, _rxjs.timer)(refreshInterval).pipe((0, _rxjs.switchMap)(() => caught));
  }));
}
function padBuckets(scheduleDensityBuckets, pollInterval, scheduleDensity) {
  var _scheduleDensity$hist, _scheduleDensity$hist2;
  // @ts-expect-error cannot infer histogram
  if (scheduleDensity.from && scheduleDensity.to && (_scheduleDensity$hist = scheduleDensity.histogram) !== null && _scheduleDensity$hist !== void 0 && (_scheduleDensity$hist2 = _scheduleDensity$hist.buckets) !== null && _scheduleDensity$hist2 !== void 0 && _scheduleDensity$hist2.length) {
    // @ts-expect-error cannot infer histogram
    const {
      histogram,
      from,
      to
    } = scheduleDensity;
    const firstBucket = histogram.buckets[0].key;
    const lastBucket = histogram.buckets[histogram.buckets.length - 1].key;

    // detect when the first bucket is before the `from` so that we can take that into
    // account by begining the timeline earlier
    // This can happen when you have overdue tasks and Elasticsearch returns their bucket
    // as begining before the `from`
    const firstBucketStartsInThePast = firstBucket - from < 0;
    const bucketsToPadBeforeFirstBucket = firstBucketStartsInThePast ? [] : calculateBucketsBetween(firstBucket, from, pollInterval);
    const bucketsToPadAfterLast = calculateBucketsBetween(lastBucket + pollInterval, firstBucketStartsInThePast ? to - pollInterval : to, pollInterval);
    return estimateRecurringTaskScheduling([...bucketsToPadBeforeFirstBucket, ...histogram.buckets.map(countByIntervalInBucket), ...bucketsToPadAfterLast], pollInterval);
  }
  return new Array(scheduleDensityBuckets).fill(0);
}
function countByIntervalInBucket(bucket) {
  if (bucket.doc_count === 0) {
    return {
      nonRecurring: 0,
      key: bucket.key
    };
  }
  const recurring = [];
  let nonRecurring = bucket.doc_count;
  for (const intervalBucket of bucket.interval.buckets) {
    recurring.push([intervalBucket.doc_count, intervalBucket.key]);
    nonRecurring -= intervalBucket.doc_count;
  }
  return {
    nonRecurring,
    recurring,
    key: bucket.key
  };
}
function calculateBucketsBetween(from, to, interval, bucketInterval = interval) {
  const calcForwardInTime = from < to;

  // as task interval might not divide by the pollInterval (aka the bucket interval)
  // we have to adjust for the "drift" that occurs when estimating when the next
  // bucket the task might actually get scheduled in
  const actualInterval = Math.ceil(interval / bucketInterval) * bucketInterval;
  const buckets = [];
  const toBound = calcForwardInTime ? to : -(to + actualInterval);
  let fromBound = calcForwardInTime ? from : -from;
  while (fromBound < toBound) {
    buckets.push({
      key: fromBound
    });
    fromBound += actualInterval;
  }
  return calcForwardInTime ? buckets : buckets.reverse().map(bucket => {
    bucket.key = Math.abs(bucket.key);
    return bucket;
  });
}
function estimateRecurringTaskScheduling(scheduleDensity, pollInterval) {
  const lastKey = scheduleDensity[scheduleDensity.length - 1].key;
  return scheduleDensity.map((bucket, currentBucketIndex) => {
    var _bucket$nonRecurring;
    for (const [count, interval] of (_bucket$recurring = bucket.recurring) !== null && _bucket$recurring !== void 0 ? _bucket$recurring : []) {
      var _bucket$recurring;
      for (const recurrance of calculateBucketsBetween(bucket.key,
      // `calculateBucketsBetween` uses the `to` as a non-inclusive upper bound
      // but lastKey is a bucket we wish to include
      lastKey + pollInterval, (0, _intervals.parseIntervalAsMillisecond)(interval), pollInterval)) {
        const recurranceBucketIndex = currentBucketIndex + Math.ceil((recurrance.key - bucket.key) / pollInterval);
        if (recurranceBucketIndex < scheduleDensity.length) {
          var _scheduleDensity$recu;
          scheduleDensity[recurranceBucketIndex].nonRecurring = count + ((_scheduleDensity$recu = scheduleDensity[recurranceBucketIndex].nonRecurring) !== null && _scheduleDensity$recu !== void 0 ? _scheduleDensity$recu : 0);
        }
      }
    }
    return (_bucket$nonRecurring = bucket.nonRecurring) !== null && _bucket$nonRecurring !== void 0 ? _bucket$nonRecurring : 0;
  });
}
function summarizeWorkloadStat(workloadStats) {
  return {
    value: workloadStats,
    status: _monitoring_stats_stream.HealthStatus.OK
  };
}
function hasAggregations(aggregations) {
  var _aggregations$idleTas, _aggregations$idleTas2;
  return !!(aggregations !== null && aggregations !== void 0 && aggregations.taskType && aggregations !== null && aggregations !== void 0 && aggregations.schedule && aggregations !== null && aggregations !== void 0 && (_aggregations$idleTas = aggregations.idleTasks) !== null && _aggregations$idleTas !== void 0 && _aggregations$idleTas.overdue && aggregations !== null && aggregations !== void 0 && (_aggregations$idleTas2 = aggregations.idleTasks) !== null && _aggregations$idleTas2 !== void 0 && _aggregations$idleTas2.scheduleDensity);
}

// @ts-expect-error key doesn't accept a string

function getTotalCost(taskTypeBuckets, definitions) {
  let cost = 0;
  for (const bucket of taskTypeBuckets) {
    const taskDef = definitions.get(bucket.key);
    if (taskDef) {
      var _ref2;
      cost += (_ref2 = bucket.doc_count * (taskDef === null || taskDef === void 0 ? void 0 : taskDef.cost)) !== null && _ref2 !== void 0 ? _ref2 : _task.TaskCost.Normal;
    } else {
      // task type is not registered with dictionary, do not add to cost
    }
  }
  return cost;
}