"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PREFERRED_MAX_POLL_INTERVAL = exports.MIN_WORKERS = exports.MIN_COST = exports.INTERVAL_AFTER_BLOCK_EXCEPTION = exports.ADJUST_THROUGHPUT_INTERVAL = void 0;
exports.calculateStartingCapacity = calculateStartingCapacity;
exports.countErrors = countErrors;
exports.createCapacityScan = createCapacityScan;
exports.createPollIntervalScan = createPollIntervalScan;
var _statsLite = _interopRequireDefault(require("stats-lite"));
var _rxjs = require("rxjs");
var _server = require("@kbn/core/server");
var _identify_es_error = require("./identify_es_error");
var _config = require("../config");
var _task = require("../task");
var _msearch_error = require("./msearch_error");
var _bulk_update_error = require("./bulk_update_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FLUSH_MARKER = Symbol('flush');
const ADJUST_THROUGHPUT_INTERVAL = exports.ADJUST_THROUGHPUT_INTERVAL = 10 * 1000;
const PREFERRED_MAX_POLL_INTERVAL = exports.PREFERRED_MAX_POLL_INTERVAL = 60 * 1000;
const INTERVAL_AFTER_BLOCK_EXCEPTION = exports.INTERVAL_AFTER_BLOCK_EXCEPTION = 61 * 1000;

// Capacity is measured in number of normal cost tasks that can be run
// At a minimum, we need to be able to run a single task with the greatest cost
// so we should convert the greatest cost to normal cost
const MIN_COST = exports.MIN_COST = _task.TaskCost.ExtraLarge / _task.TaskCost.Normal;

// For default claim strategy
const MIN_WORKERS = exports.MIN_WORKERS = 1;

// When errors occur, reduce capacity by CAPACITY_DECREASE_PERCENTAGE
// When errors no longer occur, start increasing capacity by CAPACITY_INCREASE_PERCENTAGE
// until starting value is reached
const CAPACITY_DECREASE_PERCENTAGE = 0.8;
const CAPACITY_INCREASE_PERCENTAGE = 1.05;

// When errors occur, increase pollInterval by POLL_INTERVAL_INCREASE_PERCENTAGE
// When errors no longer occur, start decreasing pollInterval by POLL_INTERVAL_DECREASE_PERCENTAGE
// until starting value is reached
const POLL_INTERVAL_DECREASE_PERCENTAGE = 0.95;
const POLL_INTERVAL_INCREASE_PERCENTAGE = 1.2;
function createCapacityScan(config, logger, startingCapacity) {
  return (0, _rxjs.scan)((previousCapacity, {
    count: errorCount,
    isBlockException
  }) => {
    let newCapacity;
    if (isBlockException) {
      newCapacity = previousCapacity;
    } else {
      if (errorCount > 0) {
        const minCapacity = getMinCapacity(config);
        // Decrease capacity by CAPACITY_DECREASE_PERCENTAGE while making sure it doesn't go lower than minCapacity.
        // Using Math.floor to make sure the number is different than previous while not being a decimal value.
        newCapacity = Math.max(Math.floor(previousCapacity * CAPACITY_DECREASE_PERCENTAGE), minCapacity);
      } else {
        // Increase capacity by CAPACITY_INCREASE_PERCENTAGE while making sure it doesn't go
        // higher than the starting value. Using Math.ceil to make sure the number is different than
        // previous while not being a decimal value
        newCapacity = Math.min(startingCapacity, Math.ceil(previousCapacity * CAPACITY_INCREASE_PERCENTAGE));
      }
    }
    if (newCapacity !== previousCapacity) {
      logger.debug(`Capacity configuration changing from ${previousCapacity} to ${newCapacity} after seeing ${errorCount} "too many request" and/or "execute [inline] script" error(s)`);
      if (previousCapacity === startingCapacity) {
        logger.warn(`Capacity configuration is temporarily reduced after Elasticsearch returned ${errorCount} "too many request" and/or "execute [inline] script" error(s).`);
      }
    }
    return newCapacity;
  }, startingCapacity);
}
function createPollIntervalScan(logger, startingPollInterval, claimStrategy, tmUtilizationQueue) {
  return (0, _rxjs.scan)((previousPollInterval, [{
    count: errorCount,
    isBlockException
  }, tmUtilization]) => {
    let newPollInterval;
    let updatedForCapacity = false;
    let avgTmUtilization = 0;
    if (isBlockException) {
      newPollInterval = INTERVAL_AFTER_BLOCK_EXCEPTION;
    } else {
      if (errorCount > 0) {
        // Increase poll interval by POLL_INTERVAL_INCREASE_PERCENTAGE and use Math.ceil to
        // make sure the number is different than previous while not being a decimal value.
        // Also ensure we don't go over PREFERRED_MAX_POLL_INTERVAL or startingPollInterval,
        // whichever is greater.
        newPollInterval = Math.min(Math.ceil(previousPollInterval * POLL_INTERVAL_INCREASE_PERCENTAGE), Math.ceil(Math.max(PREFERRED_MAX_POLL_INTERVAL, startingPollInterval)));
        if (!Number.isSafeInteger(newPollInterval) || newPollInterval < 0) {
          logger.error(`Poll interval configuration had an issue calculating the new poll interval: Math.min(Math.ceil(${previousPollInterval} * ${POLL_INTERVAL_INCREASE_PERCENTAGE}), Math.max(${PREFERRED_MAX_POLL_INTERVAL}, ${startingPollInterval})) = ${newPollInterval}, will keep the poll interval unchanged (${previousPollInterval})`);
          newPollInterval = previousPollInterval;
        }
      } else {
        if (previousPollInterval === INTERVAL_AFTER_BLOCK_EXCEPTION) {
          newPollInterval = startingPollInterval;
        } else {
          // Decrease poll interval by POLL_INTERVAL_DECREASE_PERCENTAGE and use Math.floor to
          // make sure the number is different than previous while not being a decimal value.
          newPollInterval = Math.max(startingPollInterval, Math.floor(previousPollInterval * POLL_INTERVAL_DECREASE_PERCENTAGE));
        }
        if (!Number.isSafeInteger(newPollInterval) || newPollInterval < 0) {
          logger.error(`Poll interval configuration had an issue calculating the new poll interval: Math.max(${startingPollInterval}, Math.floor(${previousPollInterval} * ${POLL_INTERVAL_DECREASE_PERCENTAGE})) = ${newPollInterval}, will keep the poll interval unchanged (${previousPollInterval})`);
          newPollInterval = previousPollInterval;
        }

        // If the task claim strategy is mget, increase the poll interval if the the avg used capacity over 15s is less than 25%.
        const queue = tmUtilizationQueue(tmUtilization);
        avgTmUtilization = _statsLite.default.mean(queue);
        if (claimStrategy === _config.CLAIM_STRATEGY_MGET && newPollInterval < _config.DEFAULT_POLL_INTERVAL) {
          updatedForCapacity = true;
          if (avgTmUtilization < 25) {
            newPollInterval = _config.DEFAULT_POLL_INTERVAL;
          } else {
            // If the the used capacity is greater than or equal to 25% reset the polling interval.
            newPollInterval = startingPollInterval;
          }
        }
      }
    }
    if (newPollInterval !== previousPollInterval) {
      if (previousPollInterval !== INTERVAL_AFTER_BLOCK_EXCEPTION) {
        if (updatedForCapacity) {
          logger.debug(`Poll interval configuration changing from ${previousPollInterval} to ${newPollInterval} after a change in the average task load: ${avgTmUtilization}.`);
        } else {
          logger.warn(`Poll interval configuration changing from ${previousPollInterval} to ${newPollInterval} after seeing ${errorCount} "too many request" and/or "execute [inline] script" error(s) and/or "cluster_block_exception" error(s).`);
        }
      }
    }
    return newPollInterval;
  }, startingPollInterval);
}
function countErrors(errors$, countInterval) {
  return (0, _rxjs.merge)(
  // Flush error count at fixed interval
  (0, _rxjs.interval)(countInterval).pipe((0, _rxjs.map)(() => FLUSH_MARKER)), errors$.pipe((0, _rxjs.filter)(e => _server.SavedObjectsErrorHelpers.isTooManyRequestsError(e) || _server.SavedObjectsErrorHelpers.isEsUnavailableError(e) || _server.SavedObjectsErrorHelpers.isGeneralError(e) || (0, _identify_es_error.isEsCannotExecuteScriptError)(e) || (0, _msearch_error.getMsearchStatusCode)(e) === 429 || (0, _msearch_error.getMsearchStatusCode)(e) !== undefined && (0, _msearch_error.getMsearchStatusCode)(e) >= 500 || (0, _bulk_update_error.getBulkUpdateStatusCode)(e) === 429 || (0, _bulk_update_error.getBulkUpdateStatusCode)(e) !== undefined && (0, _bulk_update_error.getBulkUpdateStatusCode)(e) >= 500 || (0, _bulk_update_error.isClusterBlockException)(e)))).pipe(
  // When tag is "flush", reset the error counter
  // Otherwise increment the error counter
  (0, _rxjs.mergeScan)(({
    count,
    isBlockException
  }, next) => {
    return next === FLUSH_MARKER ? (0, _rxjs.of)(emitErrorCount(count, isBlockException), resetErrorCount()) : (0, _rxjs.of)(incrementOrEmitErrorCount(count, (0, _bulk_update_error.isClusterBlockException)(next)));
  }, emitErrorCount(0, false)), (0, _rxjs.filter)(isEmitEvent), (0, _rxjs.map)(({
    count,
    isBlockException
  }) => {
    return {
      count,
      isBlockException
    };
  }));
}
function emitErrorCount(count, isBlockException) {
  return {
    tag: 'emit',
    isBlockException,
    count
  };
}
function isEmitEvent(event) {
  return event.tag === 'emit';
}
function incrementOrEmitErrorCount(count, isBlockException) {
  if (isBlockException) {
    return {
      tag: 'emit',
      isBlockException,
      count: count + 1
    };
  }
  return {
    tag: 'inc',
    isBlockException,
    count: count + 1
  };
}
function resetErrorCount() {
  return {
    tag: 'initial',
    isBlockException: false,
    count: 0
  };
}
function getMinCapacity(config) {
  switch (config.claim_strategy) {
    case _config.CLAIM_STRATEGY_MGET:
      return MIN_COST;
    default:
      return MIN_WORKERS;
  }
}
function calculateStartingCapacity(config, logger, defaultCapacity) {
  if (config.capacity !== undefined && config.max_workers !== undefined) {
    logger.warn(`Both "xpack.task_manager.capacity" and "xpack.task_manager.max_workers" configs are set, max_workers will be ignored in favor of capacity and the setting should be removed.`);
  }
  if (config.capacity) {
    // Use capacity if explicitly set
    return config.capacity;
  } else if (config.max_workers) {
    // Otherwise use max_worker value as capacity, capped at MAX_CAPACITY
    return Math.min(config.max_workers, _config.MAX_CAPACITY);
  }

  // Neither are set, use the given default capacity
  return defaultCapacity;
}