"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.simulateProcessorRoute = exports.processingSuggestionRoute = exports.processingDateSuggestionsRoute = exports.internalProcessingRoutes = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _zod = require("@kbn/zod");
var _constants = require("../../../../../common/constants");
var _security_error = require("../../../../lib/streams/errors/security_error");
var _stream_crud = require("../../../../lib/streams/stream_crud");
var _create_server_route = require("../../../create_server_route");
var _simulation_handler = require("./simulation_handler");
var _suggestions_handler = require("./suggestions_handler");
var _date_suggestions_handler = require("./suggestions/date_suggestions_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const paramsSchema = _zod.z.object({
  path: _zod.z.object({
    name: _zod.z.string()
  }),
  body: _zod.z.object({
    processing: _zod.z.array(_streamsSchema.processorWithIdDefinitionSchema),
    documents: _zod.z.array(_streamsSchema.flattenRecord),
    detected_fields: _zod.z.array(_streamsSchema.namedFieldDefinitionConfigSchema).optional()
  })
});
const simulateProcessorRoute = exports.simulateProcessorRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/processing/_simulate',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: paramsSchema,
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name: params.path.name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot read stream ${params.path.name}, insufficient privileges`);
    }
    return (0, _simulation_handler.simulateProcessing)({
      params,
      scopedClusterClient,
      streamsClient
    });
  }
});
const processingSuggestionSchema = _zod.z.object({
  field: _zod.z.string(),
  connectorId: _zod.z.string(),
  samples: _zod.z.array(_streamsSchema.flattenRecord)
});
const suggestionsParamsSchema = _zod.z.object({
  path: _zod.z.object({
    name: _zod.z.string()
  }),
  body: processingSuggestionSchema
});
const processingSuggestionRoute = exports.processingSuggestionRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/processing/_suggestions',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: suggestionsParamsSchema,
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const isAvailableForTier = server.core.pricing.isFeatureAvailable(_constants.STREAMS_TIERED_ML_FEATURE.id);
    if (!isAvailableForTier) {
      throw new _security_error.SecurityError(`Cannot access API on the current pricing tier`);
    }
    const {
      inferenceClient,
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    return (0, _suggestions_handler.handleProcessingSuggestion)(params.path.name, params.body, inferenceClient, scopedClusterClient, streamsClient);
  }
});
const processingDateSuggestionsRoute = exports.processingDateSuggestionsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/processing/_suggestions/date',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _date_suggestions_handler.processingDateSuggestionsSchema,
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const isAvailableForTier = server.core.pricing.isFeatureAvailable(_constants.STREAMS_TIERED_ML_FEATURE.id);
    if (!isAvailableForTier) {
      throw new _security_error.SecurityError(`Cannot access API on the current pricing tier`);
    }
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot read stream ${name}, insufficient privileges`);
    }
    const {
      text_structure: hasTextStructurePrivileges
    } = await streamsClient.getPrivileges(name);
    if (!hasTextStructurePrivileges) {
      throw new _security_error.SecurityError(`Cannot access text structure capabilities, insufficient privileges`);
    }
    return (0, _date_suggestions_handler.handleProcessingDateSuggestions)({
      params,
      scopedClusterClient
    });
  }
});
const internalProcessingRoutes = exports.internalProcessingRoutes = {
  ...simulateProcessorRoute,
  ...processingSuggestionRoute,
  ...processingDateSuggestionsRoute
};