"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MANAGED_BY_STREAMS = void 0;
exports.translateUnwiredStreamPipelineActions = translateUnwiredStreamPipelineActions;
var _esErrors = require("@kbn/es-errors");
var _lodash = require("lodash");
var _constants = require("../../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MANAGED_BY_STREAMS = exports.MANAGED_BY_STREAMS = 'streams';
/**
 * UnwiredStreams sometimes share index templates and ingest pipelines (user managed or Streams managed)
 * In order to modify this pipelines in an atomic way and be able to clean up any Streams managed pipeline when no longer needed
 * We need to translate some actions
 */
async function translateUnwiredStreamPipelineActions(actionsByType, scopedClusterClient) {
  const maybeActions = [...actionsByType.append_processor_to_ingest_pipeline, ...actionsByType.delete_processor_from_ingest_pipeline];
  if (maybeActions.length === 0) {
    return;
  }
  const actionsByPipeline = (0, _lodash.groupBy)(maybeActions, 'pipeline');
  for (const [pipelineName, actions] of Object.entries(actionsByPipeline)) {
    const pipeline = await getPipeline(pipelineName, scopedClusterClient);
    if (pipeline) {
      var _pipeline$_meta;
      if (((_pipeline$_meta = pipeline._meta) === null || _pipeline$_meta === void 0 ? void 0 : _pipeline$_meta.managed_by) === MANAGED_BY_STREAMS) {
        await updateExistingStreamsManagedPipeline({
          pipelineName,
          pipeline,
          actions,
          actionsByType,
          scopedClusterClient
        });
      } else {
        await updateExistingUserManagedPipeline({
          pipelineName,
          actions,
          actionsByType,
          scopedClusterClient
        });
      }
    } else {
      await createStreamsManagedPipeline({
        actions,
        actionsByType,
        scopedClusterClient
      });
    }
  }
  actionsByType.append_processor_to_ingest_pipeline = [];
  actionsByType.delete_processor_from_ingest_pipeline = [];
}
async function createStreamsManagedPipeline({
  actions,
  actionsByType,
  scopedClusterClient
}) {
  var _safeTemplate$templat, _safeTemplate$templat2, _safeTemplate$templat3, _safeTemplate$templat4, _safeTemplate$templat5, _safeTemplate$templat6;
  assertOnlyAppendActions(actions);
  const targetTemplateNames = (0, _lodash.uniq)(actions.map(action => action.template));
  if (targetTemplateNames.length !== 1) {
    throw new Error('Append actions targeting the same new pipeline target different templates');
  }
  const indexTemplate = await getIndexTemplate(targetTemplateNames[0], scopedClusterClient);
  const pipelineName = `${indexTemplate.name}-pipeline`;
  actionsByType.upsert_ingest_pipeline.push({
    type: 'upsert_ingest_pipeline',
    // All of these are UnwiredStreams so take any stream name to use for the ordering of operations
    stream: actions[0].dataStream,
    request: {
      id: pipelineName,
      processors: actions.map(action => action.processor),
      _meta: {
        description: `Streams managed pipeline to connect Unwired streams to the Streams layer`,
        managed: true,
        managed_by: MANAGED_BY_STREAMS
      },
      version: _constants.ASSET_VERSION
    }
  });

  // Remove properties from the GET response that cannot be in the PUT request
  // eslint-disable-next-line @typescript-eslint/naming-convention
  const {
    created_date_millis,
    modified_date_millis,
    ...safeTemplate
  } = indexTemplate.index_template;
  actionsByType.upsert_index_template.push({
    type: 'upsert_index_template',
    request: {
      name: indexTemplate.name,
      ...safeTemplate,
      ignore_missing_component_templates: safeTemplate.ignore_missing_component_templates ? (0, _lodash.castArray)(safeTemplate.ignore_missing_component_templates) : [],
      template: {
        ...((_safeTemplate$templat = safeTemplate.template) !== null && _safeTemplate$templat !== void 0 ? _safeTemplate$templat : {}),
        settings: {
          ...((_safeTemplate$templat2 = (_safeTemplate$templat3 = safeTemplate.template) === null || _safeTemplate$templat3 === void 0 ? void 0 : _safeTemplate$templat3.settings) !== null && _safeTemplate$templat2 !== void 0 ? _safeTemplate$templat2 : {}),
          index: {
            ...((_safeTemplate$templat4 = (_safeTemplate$templat5 = safeTemplate.template) === null || _safeTemplate$templat5 === void 0 ? void 0 : (_safeTemplate$templat6 = _safeTemplate$templat5.settings) === null || _safeTemplate$templat6 === void 0 ? void 0 : _safeTemplate$templat6.index) !== null && _safeTemplate$templat4 !== void 0 ? _safeTemplate$templat4 : {}),
            default_pipeline: pipelineName
          }
        }
      }
    }
  });
  actionsByType.upsert_write_index_or_rollover.push(...actions.map(action => ({
    type: 'upsert_write_index_or_rollover',
    request: {
      name: action.dataStream
    }
  })));
}
async function updateExistingStreamsManagedPipeline({
  pipelineName,
  pipeline,
  actions,
  actionsByType,
  scopedClusterClient
}) {
  var _pipeline$processors;
  let processors = (_pipeline$processors = pipeline.processors) !== null && _pipeline$processors !== void 0 ? _pipeline$processors : [];
  const existingPipelineProcessors = processors.filter(processor => processor.pipeline !== undefined).map(processor => {
    var _processor$pipeline;
    return (_processor$pipeline = processor.pipeline) === null || _processor$pipeline === void 0 ? void 0 : _processor$pipeline.name;
  });
  for (const action of actions) {
    if (action.type === 'append_processor_to_ingest_pipeline') {
      var _action$processor$pip;
      if (!existingPipelineProcessors.includes((_action$processor$pip = action.processor.pipeline) === null || _action$processor$pip === void 0 ? void 0 : _action$processor$pip.name)) {
        processors.push(action.processor);
      }
    } else {
      processors = processors.filter(processor => processor.pipeline === undefined || processor.pipeline.name !== action.referencePipeline);
    }
  }
  if (processors.length !== 0) {
    actionsByType.upsert_ingest_pipeline.push({
      type: 'upsert_ingest_pipeline',
      // All of these are UnwiredStreams so take any stream name to use for the ordering of operations
      stream: actions[0].dataStream,
      request: {
        id: pipelineName,
        ...pipeline,
        processors
      }
    });
  } else {
    var _indexTemplate$index_, _indexTemplate$index_2, _indexTemplate$index_3, _indexTemplate$index_4, _indexTemplate$index_5, _indexTemplate$index_6;
    actionsByType.delete_ingest_pipeline.push({
      type: 'delete_ingest_pipeline',
      request: {
        name: pipelineName
      }
    });
    const targetTemplateNames = (0, _lodash.uniq)(actions.map(action => action.template));
    if (targetTemplateNames.length !== 1) {
      throw new Error('Actions targeting the same existing Streams managed pipeline target different templates');
    }
    const indexTemplate = await getIndexTemplate(targetTemplateNames[0], scopedClusterClient);
    actionsByType.upsert_index_template.push({
      type: 'upsert_index_template',
      request: {
        name: indexTemplate.name,
        ...indexTemplate.index_template,
        ignore_missing_component_templates: indexTemplate.index_template.ignore_missing_component_templates ? (0, _lodash.castArray)(indexTemplate.index_template.ignore_missing_component_templates) : [],
        template: {
          ...((_indexTemplate$index_ = indexTemplate.index_template.template) !== null && _indexTemplate$index_ !== void 0 ? _indexTemplate$index_ : {}),
          settings: {
            ...((_indexTemplate$index_2 = (_indexTemplate$index_3 = indexTemplate.index_template.template) === null || _indexTemplate$index_3 === void 0 ? void 0 : _indexTemplate$index_3.settings) !== null && _indexTemplate$index_2 !== void 0 ? _indexTemplate$index_2 : {}),
            index: {
              ...((_indexTemplate$index_4 = (_indexTemplate$index_5 = indexTemplate.index_template.template) === null || _indexTemplate$index_5 === void 0 ? void 0 : (_indexTemplate$index_6 = _indexTemplate$index_5.settings) === null || _indexTemplate$index_6 === void 0 ? void 0 : _indexTemplate$index_6.index) !== null && _indexTemplate$index_4 !== void 0 ? _indexTemplate$index_4 : {}),
              default_pipeline: undefined
            }
          }
        }
      }
    });
    actionsByType.upsert_write_index_or_rollover.push(...actions.map(action => ({
      type: 'upsert_write_index_or_rollover',
      request: {
        name: action.dataStream
      }
    })));
  }
}
async function updateExistingUserManagedPipeline({
  pipelineName,
  actions,
  actionsByType,
  scopedClusterClient
}) {
  var _targetPipeline$proce;
  const referencePipelineNames = actions.map(action => action.referencePipeline);
  const {
    targetPipelineName,
    targetPipeline
  } = await findPipelineToModify(pipelineName, referencePipelineNames, scopedClusterClient);
  let processors = (_targetPipeline$proce = targetPipeline === null || targetPipeline === void 0 ? void 0 : targetPipeline.processors) !== null && _targetPipeline$proce !== void 0 ? _targetPipeline$proce : [];
  const existingPipelineProcessors = processors.filter(processor => processor.pipeline !== undefined).map(processor => {
    var _processor$pipeline2;
    return (_processor$pipeline2 = processor.pipeline) === null || _processor$pipeline2 === void 0 ? void 0 : _processor$pipeline2.name;
  });
  for (const action of actions) {
    if (action.type === 'append_processor_to_ingest_pipeline') {
      var _action$processor$pip2;
      if (!existingPipelineProcessors.includes((_action$processor$pip2 = action.processor.pipeline) === null || _action$processor$pip2 === void 0 ? void 0 : _action$processor$pip2.name)) {
        processors.push(action.processor);
      }
    } else {
      processors = processors.filter(processor => processor.pipeline === undefined || processor.pipeline.name !== action.referencePipeline);
    }
  }
  actionsByType.upsert_ingest_pipeline.push({
    type: 'upsert_ingest_pipeline',
    // All of these are UnwiredStreams so take any stream name to use for the ordering of operations
    stream: actions[0].dataStream,
    request: {
      id: targetPipelineName,
      ...(targetPipeline !== null && targetPipeline !== void 0 ? targetPipeline : {}),
      processors
    }
  });
}
async function findPipelineToModify(pipelineName, referencePipelineNames, scopedClusterClient) {
  var _pipeline$processors2, _pipeline$processors3;
  const pipeline = await getPipeline(pipelineName, scopedClusterClient);
  if (!pipeline) {
    return {
      targetPipelineName: pipelineName,
      targetPipeline: undefined
    };
  }
  const streamProcessor = (_pipeline$processors2 = pipeline.processors) === null || _pipeline$processors2 === void 0 ? void 0 : _pipeline$processors2.find(processor => processor.pipeline && processor.pipeline.name.includes('@stream.'));
  if (streamProcessor) {
    return {
      targetPipelineName: pipelineName,
      targetPipeline: pipeline
    };
  }
  const customProcessor = (_pipeline$processors3 = pipeline.processors) === null || _pipeline$processors3 === void 0 ? void 0 : _pipeline$processors3.findLast(processor => processor.pipeline && processor.pipeline.name.endsWith('@custom'));
  if (customProcessor) {
    // go one level deeper, find the latest @custom leaf pipeline
    return await findPipelineToModify(customProcessor.pipeline.name, referencePipelineNames, scopedClusterClient);
  }
  return {
    targetPipelineName: pipelineName,
    targetPipeline: pipeline
  };
}
async function getPipeline(id, scopedClusterClient) {
  return scopedClusterClient.asCurrentUser.ingest.getPipeline({
    id
  }).then(response => response[id]).catch(error => {
    if ((0, _esErrors.isNotFoundError)(error)) {
      return undefined;
    }
    throw error;
  });
}
async function getIndexTemplate(name, scopedClusterClient) {
  const indexTemplates = await scopedClusterClient.asCurrentUser.indices.getIndexTemplate({
    name
  });
  return indexTemplates.index_templates[0];
}
function assertOnlyAppendActions(actions) {
  const allAppendActions = actions.every(action => action.type === 'append_processor_to_ingest_pipeline');
  if (!allAppendActions) {
    throw new Error('Expected only append_processor_to_ingest_pipeline actions');
  }
}