"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateThreeWayRuleFieldsDiff = void 0;
var _utility_types = require("../../../../../../../common/utility_types");
var _invariant = require("../../../../../../../common/utils/invariant");
var _three_way_diff = require("../../../../../../../common/api/detection_engine/prebuilt_rules/model/diff/three_way_diff/three_way_diff");
var _diff_calculation_helpers = require("./diff_calculation_helpers");
var _three_way_diff_algorithms = require("./three_way_diff_algorithms");
var _scalar_array_diff_algorithm = require("./three_way_diff_algorithms/scalar_array_diff_algorithm");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BASE_TYPE_ERROR = `Base version can't be of different rule type`;
const TARGET_TYPE_ERROR = `Target version can't be of different rule type`;

/**
 * Calculates a three-way diff per each top-level rule field.
 * Returns an object which keys are equal to rule's field names and values are
 * three-way diffs calculated for those fields.
 */
const calculateThreeWayRuleFieldsDiff = (ruleVersions, isRuleCustomized = false) => {
  const commonFieldsDiff = calculateCommonFieldsDiff(ruleVersions, isRuleCustomized);
  // eslint-disable-next-line @typescript-eslint/naming-convention
  const {
    base_version,
    current_version,
    target_version
  } = ruleVersions;
  const hasBaseVersion = base_version !== _three_way_diff.MissingVersion;
  const isRuleTypeDifferentInTargetVersion = current_version.type !== target_version.type;
  const isRuleTypeDifferentInBaseVersion = hasBaseVersion ? current_version.type !== base_version.type : false;
  if (isRuleTypeDifferentInTargetVersion || isRuleTypeDifferentInBaseVersion) {
    // If rule type has been changed by Elastic in the target version (can happen)
    // or by user in the current version (should never happen), we can't calculate the diff
    // only for fields of a single rule type, and need to calculate it for all fields
    // of all the rule types we have.
    // TODO: Try to get rid of "as" casting
    return calculateAllFieldsDiff({
      base_version: base_version,
      current_version: current_version,
      target_version: target_version
    }, isRuleCustomized);
  }
  switch (current_version.type) {
    case 'query':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'query', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'query', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateCustomQueryFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'saved_query':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'saved_query', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'saved_query', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateSavedQueryFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'eql':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'eql', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'eql', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateEqlFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'threat_match':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'threat_match', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'threat_match', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateThreatMatchFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'threshold':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'threshold', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'threshold', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateThresholdFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'machine_learning':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'machine_learning', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'machine_learning', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateMachineLearningFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'new_terms':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'new_terms', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'new_terms', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateNewTermsFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'esql':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'esql', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'esql', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateEsqlFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    default:
      {
        return (0, _utility_types.assertUnreachable)(current_version, 'Unhandled rule type');
      }
  }
};
exports.calculateThreeWayRuleFieldsDiff = calculateThreeWayRuleFieldsDiff;
const calculateCommonFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, commonFieldsDiffAlgorithms, isRuleCustomized);
};
const commonFieldsDiffAlgorithms = {
  rule_id: _three_way_diff_algorithms.simpleDiffAlgorithm,
  /**
   * `version` shouldn't have a conflict. It always get target value automatically.
   * Diff has informational purpose.
   */
  version: _three_way_diff_algorithms.forceTargetVersionDiffAlgorithm,
  name: _three_way_diff_algorithms.singleLineStringDiffAlgorithm,
  tags: (0, _scalar_array_diff_algorithm.createScalarArrayDiffAlgorithm)({
    missingBaseVersionStrategy: _scalar_array_diff_algorithm.ScalarArrayDiffMissingBaseVersionStrategy.UseTarget
  }),
  description: _three_way_diff_algorithms.multiLineStringDiffAlgorithm,
  severity: _three_way_diff_algorithms.singleLineStringDiffAlgorithm,
  severity_mapping: _three_way_diff_algorithms.simpleDiffAlgorithm,
  risk_score: _three_way_diff_algorithms.numberDiffAlgorithm,
  risk_score_mapping: _three_way_diff_algorithms.simpleDiffAlgorithm,
  references: (0, _scalar_array_diff_algorithm.createScalarArrayDiffAlgorithm)({
    missingBaseVersionStrategy: _scalar_array_diff_algorithm.ScalarArrayDiffMissingBaseVersionStrategy.UseTarget
  }),
  false_positives: _three_way_diff_algorithms.simpleDiffAlgorithm,
  threat: _three_way_diff_algorithms.simpleDiffAlgorithm,
  note: _three_way_diff_algorithms.multiLineStringDiffAlgorithm,
  setup: _three_way_diff_algorithms.multiLineStringDiffAlgorithm,
  related_integrations: _three_way_diff_algorithms.simpleDiffAlgorithm,
  required_fields: _three_way_diff_algorithms.simpleDiffAlgorithm,
  rule_schedule: _three_way_diff_algorithms.simpleDiffAlgorithm,
  max_signals: _three_way_diff_algorithms.numberDiffAlgorithm,
  rule_name_override: _three_way_diff_algorithms.simpleDiffAlgorithm,
  timestamp_override: _three_way_diff_algorithms.simpleDiffAlgorithm,
  timeline_template: _three_way_diff_algorithms.simpleDiffAlgorithm,
  building_block: _three_way_diff_algorithms.simpleDiffAlgorithm,
  investigation_fields: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateCustomQueryFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, customQueryFieldsDiffAlgorithms, isRuleCustomized);
};
const customQueryFieldsDiffAlgorithms = {
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm,
  kql_query: _three_way_diff_algorithms.kqlQueryDiffAlgorithm,
  data_source: _three_way_diff_algorithms.dataSourceDiffAlgorithm,
  alert_suppression: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateSavedQueryFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, savedQueryFieldsDiffAlgorithms, isRuleCustomized);
};
const savedQueryFieldsDiffAlgorithms = {
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm,
  kql_query: _three_way_diff_algorithms.kqlQueryDiffAlgorithm,
  data_source: _three_way_diff_algorithms.dataSourceDiffAlgorithm,
  alert_suppression: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateEqlFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, eqlFieldsDiffAlgorithms, isRuleCustomized);
};
const eqlFieldsDiffAlgorithms = {
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm,
  eql_query: _three_way_diff_algorithms.eqlQueryDiffAlgorithm,
  data_source: _three_way_diff_algorithms.dataSourceDiffAlgorithm,
  alert_suppression: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateEsqlFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, esqlFieldsDiffAlgorithms, isRuleCustomized);
};
const esqlFieldsDiffAlgorithms = {
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm,
  esql_query: _three_way_diff_algorithms.esqlQueryDiffAlgorithm,
  alert_suppression: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateThreatMatchFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, threatMatchFieldsDiffAlgorithms, isRuleCustomized);
};
const threatMatchFieldsDiffAlgorithms = {
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm,
  kql_query: _three_way_diff_algorithms.kqlQueryDiffAlgorithm,
  data_source: _three_way_diff_algorithms.dataSourceDiffAlgorithm,
  threat_query: _three_way_diff_algorithms.kqlQueryDiffAlgorithm,
  threat_index: (0, _scalar_array_diff_algorithm.createScalarArrayDiffAlgorithm)({
    missingBaseVersionStrategy: _scalar_array_diff_algorithm.ScalarArrayDiffMissingBaseVersionStrategy.UseTarget
  }),
  threat_mapping: _three_way_diff_algorithms.simpleDiffAlgorithm,
  threat_indicator_path: _three_way_diff_algorithms.singleLineStringDiffAlgorithm,
  alert_suppression: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateThresholdFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, thresholdFieldsDiffAlgorithms, isRuleCustomized);
};
const thresholdFieldsDiffAlgorithms = {
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm,
  kql_query: _three_way_diff_algorithms.kqlQueryDiffAlgorithm,
  data_source: _three_way_diff_algorithms.dataSourceDiffAlgorithm,
  threshold: _three_way_diff_algorithms.simpleDiffAlgorithm,
  alert_suppression: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateMachineLearningFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, machineLearningFieldsDiffAlgorithms, isRuleCustomized);
};
const machineLearningFieldsDiffAlgorithms = {
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm,
  machine_learning_job_id: _three_way_diff_algorithms.simpleDiffAlgorithm,
  anomaly_threshold: _three_way_diff_algorithms.numberDiffAlgorithm,
  alert_suppression: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateNewTermsFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, newTermsFieldsDiffAlgorithms, isRuleCustomized);
};
const newTermsFieldsDiffAlgorithms = {
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm,
  kql_query: _three_way_diff_algorithms.kqlQueryDiffAlgorithm,
  data_source: _three_way_diff_algorithms.dataSourceDiffAlgorithm,
  new_terms_fields: (0, _scalar_array_diff_algorithm.createScalarArrayDiffAlgorithm)({
    missingBaseVersionStrategy: _scalar_array_diff_algorithm.ScalarArrayDiffMissingBaseVersionStrategy.UseTarget
  }),
  history_window_start: _three_way_diff_algorithms.singleLineStringDiffAlgorithm,
  alert_suppression: _three_way_diff_algorithms.simpleDiffAlgorithm
};
const calculateAllFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, allFieldsDiffAlgorithms, isRuleCustomized);
};
const allFieldsDiffAlgorithms = {
  ...commonFieldsDiffAlgorithms,
  ...customQueryFieldsDiffAlgorithms,
  ...savedQueryFieldsDiffAlgorithms,
  ...eqlFieldsDiffAlgorithms,
  ...esqlFieldsDiffAlgorithms,
  ...threatMatchFieldsDiffAlgorithms,
  ...thresholdFieldsDiffAlgorithms,
  ...machineLearningFieldsDiffAlgorithms,
  ...newTermsFieldsDiffAlgorithms,
  type: _three_way_diff_algorithms.ruleTypeDiffAlgorithm
};