"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUnExpiredActionsEsQuery = exports.fetchSpaceIdsWithMaybePendingActions = void 0;
var _common = require("@kbn/spaces-plugin/common");
var _utils = require("../../../utils");
var _constants = require("../../../../../common/endpoint/constants");
var _stringify = require("../../../utils/stringify");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns the ES query that can be used to match action requests that may still be pending
 * @param agentType
 */
const getUnExpiredActionsEsQuery = agentType => {
  return {
    bool: {
      must: {
        // Only actions for this agent type
        term: {
          'EndpointActions.input_type': agentType
        }
      },
      must_not: {
        // No action requests that have an `error` property defined
        exists: {
          field: 'error'
        }
      },
      filter: [
      // We only want actions requests whose expiration date is greater than now
      {
        range: {
          'EndpointActions.expiration': {
            gte: 'now'
          }
        }
      }]
    }
  };
};

/**
 * Fetches all the space IDS associated with response actions of a given agent type that may still be
 * pending. ("may" because we don't actually calculate if they are here)
 */
exports.getUnExpiredActionsEsQuery = getUnExpiredActionsEsQuery;
const fetchSpaceIdsWithMaybePendingActions = async (endpointService, agentType) => {
  var _searchResult$aggrega, _searchResult$aggrega2;
  if (!endpointService.experimentalFeatures.endpointManagementSpaceAwarenessEnabled) {
    return [_common.DEFAULT_SPACE_ID];
  }
  const logger = endpointService.createLogger('fetchSpaceIdsWithMaybePendingActions');
  const esClient = endpointService.getInternalEsClient();
  const esSearchRequest = {
    index: _constants.ENDPOINT_ACTIONS_INDEX,
    query: getUnExpiredActionsEsQuery(agentType),
    _source: false,
    size: 0,
    aggs: {
      spaceIds: {
        terms: {
          field: 'originSpaceId',
          size: 10000
        }
      }
    },
    ignore_unavailable: true
  };
  logger.debug(() => `Searching for spaces ids of possibly pending action for agent type [${agentType}]\n${(0, _stringify.stringify)(esSearchRequest)}`);
  const searchResult = await esClient.search(esSearchRequest).catch(_utils.catchAndWrapError);
  logger.debug(() => `Search Results: ${(0, _stringify.stringify)(searchResult)}`);
  return ((_searchResult$aggrega = (_searchResult$aggrega2 = searchResult.aggregations) === null || _searchResult$aggrega2 === void 0 ? void 0 : _searchResult$aggrega2.spaceIds.buckets) !== null && _searchResult$aggrega !== void 0 ? _searchResult$aggrega : []).map(({
    key
  }) => key);
};
exports.fetchSpaceIdsWithMaybePendingActions = fetchSpaceIdsWithMaybePendingActions;