"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.shallowObjectViewTruncated = exports.shallowObjectView = exports.mapFieldDescriptorToNestedObject = exports.getNestedValue = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Gets the entries at a given key in an index mapping
 */
const getNestedValue = (obj, keyPath) => {
  const normalizedKeyPath = keyPath.trim() === '.' ? '' : keyPath.trim();
  return normalizedKeyPath ? (0, _lodash.get)(obj, normalizedKeyPath) : obj;
};

/**
 * Returns a shallow view of the object
 * @param obj The object
 * @param maxDepth The maximum depth to recurse into the object
 * @returns A shallow view of the mapping
 */
exports.getNestedValue = getNestedValue;
const shallowObjectView = (obj, maxDepth = 1) => {
  if (obj === undefined || typeof obj === 'string' || typeof obj === 'number' || typeof obj === 'boolean') {
    var _obj$toString;
    return (_obj$toString = obj === null || obj === void 0 ? void 0 : obj.toString()) !== null && _obj$toString !== void 0 ? _obj$toString : undefined;
  }
  if (Array.isArray(obj)) {
    return maxDepth <= 0 ? '...' : obj;
  }
  if (typeof obj === 'object' && obj !== null) {
    if (maxDepth <= 0) {
      return '...';
    }
    return Object.fromEntries(Object.entries(obj).map(([key, value]) => {
      var _value$toString;
      return [key, typeof value === 'object' ? shallowObjectView(value, maxDepth - 1) : (_value$toString = value === null || value === void 0 ? void 0 : value.toString()) !== null && _value$toString !== void 0 ? _value$toString : undefined];
    }));
  }
  return 'unknown';
};

/**
 * Same as shallowObjectView but reduces the maxDepth if the stringified view is longer than maxCharacters
 * @param mapping The index mapping
 * @param maxCharacters The maximum number of characters to return
 * @param maxDepth The maximum depth to recurse into the object
 * @returns A shallow view of the mapping
 */
exports.shallowObjectView = shallowObjectView;
const shallowObjectViewTruncated = (obj, maxCharacters, maxDepth = 4) => {
  const view = shallowObjectView(obj, maxDepth);
  if (maxDepth > 1 && view && JSON.stringify(view).length > maxCharacters) {
    return shallowObjectViewTruncated(view, maxCharacters, maxDepth - 1);
  }
  return view;
};
exports.shallowObjectViewTruncated = shallowObjectViewTruncated;
const mapFieldDescriptorToNestedObject = arr => {
  return arr.reduce((acc, obj) => {
    const keys = obj.name.split('.');
    keys.reduce((nested, key, index) => {
      if (!(key in nested)) {
        nested[key] = index === keys.length - 1 ? Object.fromEntries(Object.entries(obj).filter(([k]) => k !== 'name')) : {};
      }
      return nested[key];
    }, acc);
    return acc;
  }, {});
};
exports.mapFieldDescriptorToNestedObject = mapFieldDescriptorToNestedObject;