"use strict";

var _tsd = require("tsd");
var _zod = require("@kbn/zod");
var _tools = require("../../chat_complete/tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MinimalPromptInputSchema = _zod.z.object({
  query: _zod.z.string()
});
const minimalInput = {
  query: 'test'
};
const mustacheTemplate = {
  mustache: {
    template: 'User query: {{query}}'
  }
};
const getWeatherToolDefinition = {
  schema: {
    type: 'object',
    properties: {
      location: {
        type: 'string'
      }
    },
    required: ['location']
  },
  description: 'Get weather'
};
const getStockPriceToolDefinition = {
  schema: {
    type: 'object',
    properties: {
      ticker: {
        type: 'string'
      }
    },
    required: ['ticker']
  },
  description: 'Get stock price'
};
const promptTools = {
  get_weather: getWeatherToolDefinition,
  get_stock_price: getStockPriceToolDefinition
};
/**
 * Prompts for testing
 */
const promptNoTools = {
  name: 'NoToolsPrompt',
  description: 'A prompt with no tools defined in its version.',
  input: MinimalPromptInputSchema,
  versions: [{
    template: mustacheTemplate
    // No tools or toolChoice here
  }]
};
const promptWithToolsAuto = {
  name: 'WithToolsAutoPrompt',
  description: 'A prompt with tools and toolChoice:auto.',
  input: MinimalPromptInputSchema,
  versions: [{
    template: mustacheTemplate,
    tools: promptTools,
    toolChoice: _tools.ToolChoiceType.auto
  }]
};
const promptWithToolsNone = {
  name: 'WithToolsNonePrompt',
  description: 'A prompt with tools and toolChoice:none.',
  input: MinimalPromptInputSchema,
  versions: [{
    template: mustacheTemplate,
    tools: promptTools,
    toolChoice: _tools.ToolChoiceType.none
  }]
};
const promptWithGetWeatherToolChoice = {
  name: 'WithToolsSpecificPrompt',
  description: 'A prompt with tools and specific toolChoice.',
  input: MinimalPromptInputSchema,
  versions: [{
    template: mustacheTemplate,
    tools: promptTools,
    toolChoice: {
      function: 'get_weather'
    }
  }]
};

/**
 * Test PromptResponse type
 */

// No tools in prompt version

(0, _tsd.expectType)(noToolsPromptResponse);

// With tools defined in prompt version

(0, _tsd.expectType)(specificToolPromptResponse);
if (specificToolPromptResponse.toolCalls && specificToolPromptResponse.toolCalls.length > 0) {
  const firstToolCall = specificToolPromptResponse.toolCalls[0];
  (0, _tsd.expectAssignable)(firstToolCall.function.name);
}

/**
 * Test PromptCompositeResponse type
 */

// stream: false

(0, _tsd.expectType)({});

// stream: true

(0, _tsd.expectType)({});

// stream: boolean

(0, _tsd.expectAssignable)({});

/**
 * Test PromptAPI runtime inference
 */

// Default stream (false), with tools
const resWithTools = mockApi({
  connectorId: 'c1',
  prompt: promptWithToolsAuto,
  input: minimalInput
});
(0, _tsd.expectType)(resWithTools);
resWithTools.then(r => {
  if (r.toolCalls && r.toolCalls.length > 0) {
    (0, _tsd.expectType)(r.toolCalls[0].function.name);
    (0, _tsd.expectAssignable)(r.toolCalls[0].function.arguments);
  }
});

// stream: true, with tools
const resStreamWithTools = mockApi({
  connectorId: 'c1',
  prompt: promptWithToolsAuto,
  input: minimalInput,
  stream: true
});
(0, _tsd.expectType)(resStreamWithTools);

// toolChoice: none
const resToolChoiceNone = mockApi({
  connectorId: 'c1',
  prompt: promptWithToolsNone,
  input: minimalInput
});
resToolChoiceNone.then(r => {
  (0, _tsd.expectType)(r.toolCalls);
});

// toolChoice: specific function
const resToolChoiceSpecific = mockApi({
  connectorId: 'c1',
  prompt: promptWithGetWeatherToolChoice,
  input: minimalInput
});
(0, _tsd.expectType)(resToolChoiceSpecific);
resToolChoiceSpecific.then(r => {
  if (r.toolCalls && r.toolCalls.length > 0) {
    // With a specific tool choice, only that tool should be possible.
    // The ToolOptionsOfPrompt<typeof promptWithToolsSpecific> should correctly narrow this.
    (0, _tsd.expectType)(r.toolCalls[0].function.name);
    (0, _tsd.expectType)(r.toolCalls[0].function.arguments);
  }
});

// New tool definition for API call options
const getGeolocationToolDefinition = {
  schema: {
    type: 'object',
    properties: {
      address: {
        type: 'string'
      }
    },
    required: ['address']
  },
  description: 'Get geolocation for an address'
};
const apiCallTools = {
  get_geolocation: getGeolocationToolDefinition
};

// scenario: no tools in prompt, tools in API call
const resNoPromptToolsWithApiTools = mockApi({
  connectorId: 'c1',
  prompt: promptNoTools,
  input: minimalInput,
  tools: apiCallTools,
  // Tools provided directly in API call options
  toolChoice: _tools.ToolChoiceType.auto
});

// should just return the tools from the API call
(0, _tsd.expectType)(resNoPromptToolsWithApiTools);
resNoPromptToolsWithApiTools.then(r => {
  (0, _tsd.expectType)(r.toolCalls[0]);
});

// scenario: prompt tools + api tools
const resPromptToolsWithApiTools = mockApi({
  connectorId: 'c1',
  prompt: promptWithToolsAuto,
  input: minimalInput,
  tools: apiCallTools,
  toolChoice: _tools.ToolChoiceType.auto
});
(0, _tsd.expectType)(resPromptToolsWithApiTools);

// merges tools from prompt with api
resPromptToolsWithApiTools.then(r => {
  if (r.toolCalls && r.toolCalls.length > 0) {
    (0, _tsd.expectType)(r.toolCalls[0].function.name);
  }
});

// scenario: prompt tools + api tools + api tool choice override
const resPromptSpecificToolApiChoiceOverride = mockApi({
  connectorId: 'c1',
  prompt: promptWithGetWeatherToolChoice,
  input: minimalInput,
  tools: promptTools,
  toolChoice: {
    type: 'function',
    function: 'get_stock_price'
  }
});
(0, _tsd.expectType)(resPromptSpecificToolApiChoiceOverride);
resPromptSpecificToolApiChoiceOverride.then(r => {
  if (r.toolCalls && r.toolCalls.length > 0) {
    (0, _tsd.expectType)(r.toolCalls[0].function.name);
  }
});