"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../common/constants");
var _api_privileges = require("../../constants/api_privileges");
var _types = require("../../types");
var _errors = require("../schema/errors");
var _utils = require("../schema/utils");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerRoutes = router => {
  router.versioned.get({
    path: _constants.FLEET_PROXY_API_ROUTES.LIST_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ]
      }
    },
    summary: `Get proxies`,
    options: {
      tags: ['oas-tag:Fleet proxies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          body: () => (0, _utils.ListResponseSchema)(_types.FleetProxySchema)
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getAllFleetProxyHandler);
  router.versioned.post({
    path: _constants.FLEET_PROXY_API_ROUTES.CREATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: `Create a proxy`,
    options: {
      tags: ['oas-tag:Fleet proxies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostFleetProxyRequestSchema,
      response: {
        200: {
          body: () => _types.FleetProxyResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.postFleetProxyHandler);
  router.versioned.put({
    path: _constants.FLEET_PROXY_API_ROUTES.UPDATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: `Update a proxy`,
    description: `Update a proxy by ID.`,
    options: {
      tags: ['oas-tag:Fleet proxies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PutFleetProxyRequestSchema,
      response: {
        200: {
          body: () => _types.FleetProxyResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.putFleetProxyHandler);
  router.versioned.get({
    path: _constants.FLEET_PROXY_API_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ]
      }
    },
    summary: `Get a proxy`,
    description: `Get a proxy by ID.`,
    options: {
      tags: ['oas-tag:Fleet proxies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOneFleetProxyRequestSchema,
      response: {
        200: {
          body: () => _types.FleetProxyResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getFleetProxyHandler);
  router.versioned.delete({
    path: _constants.FLEET_PROXY_API_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.SETTINGS.ALL]
      }
    },
    summary: `Delete a proxy`,
    description: `Delete a proxy by ID`,
    options: {
      tags: ['oas-tag:Fleet proxies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOneFleetProxyRequestSchema,
      response: {
        200: {
          body: () => _configSchema.schema.object({
            id: _configSchema.schema.string()
          })
        },
        400: {
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.deleteFleetProxyHandler);
};
exports.registerRoutes = registerRoutes;