"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.areFieldAndUserDefinedColumnTypesCompatible = areFieldAndUserDefinedColumnTypesCompatible;
exports.checkFunctionArgMatchesDefinition = checkFunctionArgMatchesDefinition;
exports.createMapFromList = createMapFromList;
exports.endsInWhitespace = endsInWhitespace;
exports.findFinalWord = findFinalWord;
exports.findPreviousWord = findPreviousWord;
exports.getAllArrayTypes = getAllArrayTypes;
exports.getAllArrayValues = getAllArrayValues;
exports.getAllCommands = getAllCommands;
exports.getAllFunctions = getAllFunctions;
exports.getColumnByName = getColumnByName;
exports.getColumnExists = getColumnExists;
exports.getColumnForASTNode = getColumnForASTNode;
exports.getCommandDefinition = getCommandDefinition;
exports.getCommandsByName = getCommandsByName;
exports.getCurrentQueryAvailableFields = getCurrentQueryAvailableFields;
exports.getExpressionType = getExpressionType;
exports.getFieldsFromES = getFieldsFromES;
exports.getFunctionDefinition = getFunctionDefinition;
exports.getLastNonWhitespaceChar = getLastNonWhitespaceChar;
exports.getParamAtPosition = getParamAtPosition;
exports.getQuotedColumnName = void 0;
exports.getSignaturesWithMatchingArity = getSignaturesWithMatchingArity;
exports.hasWildcard = hasWildcard;
exports.inKnownTimeInterval = inKnownTimeInterval;
exports.isAggFunction = void 0;
exports.isArrayType = isArrayType;
exports.isAssignment = isAssignment;
exports.isAssignmentComplete = isAssignmentComplete;
exports.isColumnItem = isColumnItem;
exports.isFunctionItem = isFunctionItem;
exports.isFunctionOperatorParam = void 0;
exports.isIncompleteItem = isIncompleteItem;
exports.isInlineCastItem = isInlineCastItem;
exports.isLiteralItem = isLiteralItem;
exports.isMaybeAggFunction = void 0;
exports.isOptionItem = isOptionItem;
exports.isParametrized = exports.isParamExpressionType = exports.isParam = void 0;
exports.isRestartingExpression = isRestartingExpression;
exports.isSingleItem = isSingleItem;
exports.isSourceCommand = isSourceCommand;
exports.isSourceItem = isSourceItem;
exports.isSupportedFunction = isSupportedFunction;
exports.isTimeIntervalItem = isTimeIntervalItem;
exports.isUserDefinedColumn = isUserDefinedColumn;
exports.isValidLiteralOption = isValidLiteralOption;
exports.noCaseCompare = void 0;
exports.nonNullable = nonNullable;
exports.pipePrecedesCurrentWord = pipePrecedesCurrentWord;
exports.printFunctionSignature = printFunctionSignature;
exports.shouldBeQuotedSource = shouldBeQuotedSource;
exports.shouldBeQuotedText = shouldBeQuotedText;
exports.sourceExists = sourceExists;
exports.transformMapToESQLFields = transformMapToESQLFields;
exports.unescapeColumnName = unescapeColumnName;
exports.unwrapArrayOneLevel = unwrapArrayOneLevel;
exports.within = void 0;
exports.withinQuotes = withinQuotes;
var _esqlAst = require("@kbn/esql-ast");
var _lodash = require("lodash");
var _ecs_metadata_helper = require("../autocomplete/utils/ecs_metadata_helper");
var _all_operators = require("../definitions/all_operators");
var _commands = require("../definitions/commands");
var _aggregation_functions = require("../definitions/generated/aggregation_functions");
var _grouping_functions = require("../definitions/generated/grouping_functions");
var _scalar_functions = require("../definitions/generated/scalar_functions");
var _helpers = require("../definitions/helpers");
var _literals = require("../definitions/literals");
var _types = require("../definitions/types");
var _constants = require("./constants");
var _context = require("./context");
var _test_functions = require("./test_functions");
var _user_defined_columns = require("./user_defined_columns");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function nonNullable(v) {
  return v != null;
}
function isSingleItem(arg) {
  return arg && !Array.isArray(arg);
}
function isFunctionItem(arg) {
  return isSingleItem(arg) && arg.type === 'function';
}
function isOptionItem(arg) {
  return isSingleItem(arg) && arg.type === 'option';
}
function isSourceItem(arg) {
  return isSingleItem(arg) && arg.type === 'source';
}
function isColumnItem(arg) {
  return isSingleItem(arg) && arg.type === 'column';
}
function isLiteralItem(arg) {
  return isSingleItem(arg) && arg.type === 'literal';
}
function isInlineCastItem(arg) {
  return isSingleItem(arg) && arg.type === 'inlineCast';
}
function isTimeIntervalItem(arg) {
  return isSingleItem(arg) && arg.type === 'timeInterval';
}
function isAssignment(arg) {
  return isFunctionItem(arg) && arg.name === '=';
}
function isAssignmentComplete(node) {
  var _removeMarkerArgFromA, _removeMarkerArgFromA2;
  const assignExpression = (_removeMarkerArgFromA = (0, _context.removeMarkerArgFromArgsList)(node)) === null || _removeMarkerArgFromA === void 0 ? void 0 : (_removeMarkerArgFromA2 = _removeMarkerArgFromA.args) === null || _removeMarkerArgFromA2 === void 0 ? void 0 : _removeMarkerArgFromA2[1];
  return Boolean(assignExpression && Array.isArray(assignExpression) && assignExpression.length);
}
function isIncompleteItem(arg) {
  return !arg || !Array.isArray(arg) && arg.incomplete;
}
const within = (position, location) => Boolean(location && location.min <= position && location.max >= position);
exports.within = within;
function isSourceCommand({
  label
}) {
  return ['FROM', 'ROW', 'SHOW', 'TS'].includes(label);
}
let fnLookups;
let commandLookups;
function buildFunctionLookup() {
  // we always refresh if we have test functions
  if (!fnLookups || (0, _test_functions.getTestFunctions)().length) {
    fnLookups = _all_operators.operatorsDefinitions.concat(_scalar_functions.scalarFunctionDefinitions, _aggregation_functions.aggFunctionDefinitions, _grouping_functions.groupingFunctionDefinitions, (0, _test_functions.getTestFunctions)()).reduce((memo, def) => {
      memo.set(def.name, def);
      if (def.alias) {
        for (const alias of def.alias) {
          memo.set(alias, def);
        }
      }
      return memo;
    }, new Map());
  }
  return fnLookups;
}
function isSupportedFunction(name, parentCommand, option) {
  if (!parentCommand) {
    return {
      supported: false,
      reason: 'missingCommand'
    };
  }
  const fn = buildFunctionLookup().get(name);
  const isSupported = Boolean(fn === null || fn === void 0 ? void 0 : fn.locationsAvailable.includes((0, _types.getLocationFromCommandOrOptionName)(option !== null && option !== void 0 ? option : parentCommand)));
  return {
    supported: isSupported,
    reason: isSupported ? undefined : fn ? 'unsupportedFunction' : 'unknownFunction'
  };
}
function getAllFunctions(options) {
  const fns = buildFunctionLookup();
  if (!(options !== null && options !== void 0 && options.type)) {
    return Array.from(fns.values());
  }
  const types = new Set(Array.isArray(options.type) ? options.type : [options.type]);
  return Array.from(fns.values()).filter(fn => types.has(fn.type));
}
function getFunctionDefinition(name) {
  return buildFunctionLookup().get(name.toLowerCase());
}
const unwrapStringLiteralQuotes = value => value.slice(1, -1);
function buildCommandLookup() {
  if (!commandLookups) {
    commandLookups = _commands.commandDefinitions.reduce((memo, def) => {
      memo.set(def.name, def);
      return memo;
    }, new Map());
  }
  return commandLookups;
}
function getCommandDefinition(name) {
  return buildCommandLookup().get(name.toLowerCase());
}
function getAllCommands() {
  return Array.from(buildCommandLookup().values());
}
function getCommandsByName(names) {
  const commands = buildCommandLookup();
  return names.map(name => commands.get(name)).filter(command => command);
}
function doesLiteralMatchParameterType(argType, item) {
  if (item.literalType === argType) {
    return true;
  }
  if (bothStringTypes(argType, item.literalType)) {
    // all functions accept keyword literals for text parameters
    return true;
  }
  if (item.literalType === 'null') {
    // all parameters accept null, but this is not yet reflected
    // in our function definitions so we let it through here
    return true;
  }

  // some parameters accept string literals because of ES auto-casting
  if (item.literalType === 'keyword' && (argType === 'date' || argType === 'date_period' || argType === 'version' || argType === 'ip' || argType === 'boolean')) {
    return true;
  }
  return false;
}

/**
 * This function returns the userDefinedColumn or field matching a column
 */
function getColumnForASTNode(node, {
  fields,
  userDefinedColumns
}) {
  const formatted = node.type === 'identifier' ? node.name : node.parts.join('.');
  return getColumnByName(formatted, {
    fields,
    userDefinedColumns
  });
}

/**
 * Take a column name like "`my``column`"" and return "my`column"
 */
function unescapeColumnName(columnName) {
  // TODO this doesn't cover all escaping scenarios... the best thing to do would be
  // to use the AST column node parts array, but in some cases the AST node isn't available.
  if (columnName.startsWith(_constants.SINGLE_BACKTICK) && columnName.endsWith(_constants.SINGLE_BACKTICK)) {
    return columnName.slice(1, -1).replace(_constants.DOUBLE_TICKS_REGEX, _constants.SINGLE_BACKTICK);
  }
  return columnName;
}

/**
 * This function returns the userDefinedColumn or field matching a column
 */
function getColumnByName(columnName, {
  fields,
  userDefinedColumns
}) {
  var _userDefinedColumns$g;
  const unescaped = unescapeColumnName(columnName);
  return fields.get(unescaped) || ((_userDefinedColumns$g = userDefinedColumns.get(unescaped)) === null || _userDefinedColumns$g === void 0 ? void 0 : _userDefinedColumns$g[0]);
}
function isArrayType(type) {
  return type.endsWith('[]');
}

/**
 * Given an array type for example `string[]` it will return `string`
 */
function unwrapArrayOneLevel(type) {
  return isArrayType(type) ? type.slice(0, -2) : type;
}
function createMapFromList(arr) {
  const arrMap = new Map();
  for (const item of arr) {
    arrMap.set(item.name, item);
  }
  return arrMap;
}
function areFieldAndUserDefinedColumnTypesCompatible(fieldType, userColumnType) {
  if (fieldType == null) {
    return false;
  }
  return fieldType === userColumnType;
}
function printFunctionSignature(arg) {
  const fnDef = getFunctionDefinition(arg.name);
  if (fnDef) {
    const signature = (0, _helpers.getFunctionSignatures)({
      ...fnDef,
      signatures: [{
        ...(fnDef === null || fnDef === void 0 ? void 0 : fnDef.signatures[0]),
        params: arg.args.map(innerArg => Array.isArray(innerArg) ? {
          name: `InnerArgument[]`,
          type: 'any'
        } :
        // this cast isn't actually correct, but we're abusing the
        // getFunctionSignatures API anyways
        {
          name: innerArg.text,
          type: innerArg.type
        }),
        // this cast isn't actually correct, but we're abusing the
        // getFunctionSignatures API anyways
        returnType: ''
      }]
    }, {
      withTypes: false,
      capitalize: true
    });
    return signature[0].declaration;
  }
  return '';
}
function getAllArrayValues(arg) {
  const values = [];
  if (Array.isArray(arg)) {
    for (const subArg of arg) {
      if (Array.isArray(subArg)) {
        break;
      }
      if (subArg.type === 'literal') {
        values.push(String(subArg.value));
      }
      if (isColumnItem(subArg) || isTimeIntervalItem(subArg)) {
        values.push(subArg.name);
      }
      if (subArg.type === 'function') {
        const signature = printFunctionSignature(subArg);
        if (signature) {
          values.push(signature);
        }
      }
    }
  }
  return values;
}
function getAllArrayTypes(arg, parentCommand, references) {
  const types = [];
  if (Array.isArray(arg)) {
    for (const subArg of arg) {
      if (Array.isArray(subArg)) {
        break;
      }
      if (subArg.type === 'literal') {
        types.push(subArg.literalType);
      }
      if (subArg.type === 'column') {
        const hit = getColumnForASTNode(subArg, references);
        types.push((hit === null || hit === void 0 ? void 0 : hit.type) || 'unsupported');
      }
      if (subArg.type === 'timeInterval') {
        types.push('time_duration');
      }
      if (subArg.type === 'function') {
        if (isSupportedFunction(subArg.name, parentCommand).supported) {
          const fnDef = buildFunctionLookup().get(subArg.name);
          types.push(fnDef.signatures[0].returnType);
        }
      }
    }
  }
  return types;
}
function inKnownTimeInterval(timeIntervalUnit) {
  return _literals.timeUnits.some(unit => unit === timeIntervalUnit.toLowerCase());
}

/**
 * Checks if this argument is one of the possible options
 * if they are defined on the arg definition.
 *
 * TODO - Consider merging with isEqualType to create a unified arg validation function
 */
function isValidLiteralOption(arg, argDef) {
  return arg.literalType === 'keyword' && argDef.acceptedValues && !argDef.acceptedValues.map(option => option.toLowerCase()).includes(unwrapStringLiteralQuotes(arg.value).toLowerCase());
}

/**
 * Checks if both types are string types.
 *
 * Functions in ES|QL accept `text` and `keyword` types interchangeably.
 * @param type1
 * @param type2
 * @returns
 */
function bothStringTypes(type1, type2) {
  return (type1 === 'text' || type1 === 'keyword') && (type2 === 'text' || type2 === 'keyword');
}

/**
 * Checks if an AST function argument is of the correct type
 * given the definition.
 */
function checkFunctionArgMatchesDefinition(arg, parameterDefinition, references, parentCommand) {
  const parameterType = parameterDefinition.type;
  if (parameterType === 'any') {
    return true;
  }
  if (isParam(arg)) {
    return true;
  }
  if (arg.type === 'literal') {
    const matched = doesLiteralMatchParameterType(parameterType, arg);
    return matched;
  }
  if (arg.type === 'function') {
    if (isSupportedFunction(arg.name, parentCommand).supported) {
      const fnDef = buildFunctionLookup().get(arg.name);
      return fnDef.signatures.some(signature => signature.returnType === 'unknown' || parameterType === signature.returnType || bothStringTypes(parameterType, signature.returnType));
    }
  }
  if (arg.type === 'timeInterval') {
    return parameterType === 'time_duration' && inKnownTimeInterval(arg.unit);
  }
  if (arg.type === 'column') {
    const hit = getColumnForASTNode(arg, references);
    const validHit = hit;
    if (!validHit) {
      return false;
    }
    const wrappedTypes = Array.isArray(validHit.type) ? validHit.type : [validHit.type];
    return wrappedTypes.some(ct => ct === parameterType || bothStringTypes(ct, parameterType) || ct === 'null' || ct === 'unknown');
  }
  if (arg.type === 'inlineCast') {
    const lowerArgType = parameterType === null || parameterType === void 0 ? void 0 : parameterType.toLowerCase();
    const castedType = getExpressionType(arg);
    return castedType === lowerArgType;
  }
  return false;
}
function fuzzySearch(fuzzyName, resources) {
  const wildCardPosition = getWildcardPosition(fuzzyName);
  if (wildCardPosition !== 'none') {
    const matcher = getMatcher(fuzzyName, wildCardPosition);
    for (const resourceName of resources) {
      if (matcher(resourceName)) {
        return true;
      }
    }
  }
}
function getMatcher(name, position) {
  if (position === 'start') {
    const prefix = name.substring(1);
    return resource => resource.endsWith(prefix);
  }
  if (position === 'end') {
    const prefix = name.substring(0, name.length - 1);
    return resource => resource.startsWith(prefix);
  }
  if (position === 'multiple-within') {
    // make sure to remove the * at the beginning of the name if present
    const safeName = name.startsWith('*') ? name.slice(1) : name;
    // replace 2 ore more consecutive wildcards with a single one
    const setOfChars = safeName.replace(/\*{2+}/g, '*').split('*');
    return resource => {
      let index = -1;
      return setOfChars.every(char => {
        index = resource.indexOf(char, index + 1);
        return index !== -1;
      });
    };
  }
  const [prefix, postFix] = name.split('*');
  return resource => resource.startsWith(prefix) && resource.endsWith(postFix);
}
function getWildcardPosition(name) {
  if (!hasWildcard(name)) {
    return 'none';
  }
  const wildCardCount = name.match(/\*/g).length;
  if (wildCardCount > 1) {
    return 'multiple-within';
  }
  if (name.startsWith('*')) {
    return 'start';
  }
  if (name.endsWith('*')) {
    return 'end';
  }
  return 'middle';
}
function hasWildcard(name) {
  return /\*/.test(name);
}
function isUserDefinedColumn(column) {
  return Boolean(column && 'location' in column);
}

/**
 * This returns the name with any quotes that were present.
 *
 * E.g. "`bytes`" will be "`bytes`"
 *
 * @param node
 * @returns
 */
const getQuotedColumnName = node => node.type === 'identifier' ? node.name : node.quoted ? node.text : node.name;

/**
 * TODO - consider calling lookupColumn under the hood of this function. Seems like they should really do the same thing.
 */
exports.getQuotedColumnName = getQuotedColumnName;
function getColumnExists(node, {
  fields,
  userDefinedColumns
}) {
  const columnName = node.type === 'identifier' ? node.name : node.parts.join('.');
  if (fields.has(columnName) || userDefinedColumns.has(columnName)) {
    return true;
  }

  // TODO — I don't see this fuzzy searching in lookupColumn... should it be there?
  if (Boolean(fuzzySearch(columnName, fields.keys()) || fuzzySearch(columnName, userDefinedColumns.keys()))) {
    return true;
  }
  return false;
}
const removeSourceNameQuotes = sourceName => sourceName.startsWith('"') && sourceName.endsWith('"') ? sourceName.slice(1, -1) : sourceName;

// Function to clean a single index string from failure stores
const cleanIndex = inputIndex => {
  let cleaned = inputIndex.trim();

  // Remove '::data' suffix
  if (cleaned.endsWith('::data')) {
    cleaned = cleaned.slice(0, -6);
  }
  // Remove '::failures' suffix
  if (cleaned.endsWith('::failures')) {
    cleaned = cleaned.slice(0, -10);
  }
  return cleaned;
};

/**
 * Checks if the source exists in the provided sources set.
 * It supports both exact matches and fuzzy searches.
 *
 * @param index - The index to check, which can be a single value or a comma-separated list.
 * @param sources - A Set of source names to check against.
 * @returns true if the source exists, false otherwise.
 */

// The comma-separated index and the ::data or ::failures suffixes solution is temporary
// till we fix the AST for the quoted index names https://github.com/elastic/kibana/issues/222505.
function sourceExists(index, sources) {
  if (index.startsWith('-')) {
    return true;
  }
  // Split the index by comma to handle multiple values and clean each part
  const individualIndices = index.split(',').map(item => cleanIndex(item));
  // Check if all individual indices exist in sources
  const allExist = individualIndices.every(singleIndex => {
    // First, check for exact match after removing source name quotes
    if (sources.has(removeSourceNameQuotes(singleIndex))) {
      return true;
    }
    // If not an exact match, perform a fuzzy search
    return Boolean(fuzzySearch(singleIndex, sources.keys()));
  });
  return allExist;
}

/**
 * Works backward from the cursor position to determine if
 * the final character of the previous word matches the given character.
 */
function characterPrecedesCurrentWord(text, char) {
  let inCurrentWord = true;
  for (let i = text.length - 1; i >= 0; i--) {
    if (inCurrentWord && /\s/.test(text[i])) {
      inCurrentWord = false;
    }
    if (!inCurrentWord && !/\s/.test(text[i])) {
      return text[i] === char;
    }
  }
}
function pipePrecedesCurrentWord(text) {
  return characterPrecedesCurrentWord(text, '|');
}
function getLastNonWhitespaceChar(text) {
  return text[text.trimEnd().length - 1];
}

/**
 * Are we after a comma? i.e. STATS fieldA, <here>
 */
function isRestartingExpression(text) {
  return getLastNonWhitespaceChar(text) === ',' || characterPrecedesCurrentWord(text, ',');
}
function findPreviousWord(text) {
  const words = text.split(/\s+/);
  return words[words.length - 2];
}
function withinQuotes(text) {
  const quoteCount = (text.match(/"/g) || []).length;
  return quoteCount % 2 === 1;
}
function endsInWhitespace(text) {
  return /\s$/.test(text);
}

/**
 * Returns the word at the end of the text if there is one.
 * @param text
 * @returns
 */
function findFinalWord(text) {
  const words = text.split(/\s+/);
  return words[words.length - 1];
}
function shouldBeQuotedSource(text) {
  // Based on lexer `fragment UNQUOTED_SOURCE_PART`
  return /[:"=|,[\]\/ \t\r\n]/.test(text);
}
function shouldBeQuotedText(text, {
  dashSupported
} = {}) {
  return dashSupported ? /[^a-zA-Z\d_\.@-]/.test(text) : /[^a-zA-Z\d_\.@]/.test(text);
}
const isAggFunction = arg => {
  var _getFunctionDefinitio;
  return ((_getFunctionDefinitio = getFunctionDefinition(arg.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === _types.FunctionDefinitionTypes.AGG;
};
exports.isAggFunction = isAggFunction;
const isParam = x => !!x && typeof x === 'object' && x.type === 'literal' && x.literalType === 'param';
exports.isParam = isParam;
const isFunctionOperatorParam = fn => !!fn.operator && isParam(fn.operator);

/**
 * Returns `true` if the function is an aggregation function or a function
 * name is a parameter, which potentially could be an aggregation function.
 */
exports.isFunctionOperatorParam = isFunctionOperatorParam;
const isMaybeAggFunction = fn => isAggFunction(fn) || isFunctionOperatorParam(fn);
exports.isMaybeAggFunction = isMaybeAggFunction;
const isParametrized = node => _esqlAst.Walker.params(node).length > 0;

/**
 * Compares two strings in a case-insensitive manner
 */
exports.isParametrized = isParametrized;
const noCaseCompare = (a, b) => a.toLowerCase() === b.toLowerCase();

/**
 * Gets the signatures of a function that match the number of arguments
 * provided in the AST.
 */
exports.noCaseCompare = noCaseCompare;
function getSignaturesWithMatchingArity(fnDef, astFunction) {
  return fnDef.signatures.filter(def => {
    if (def.minParams) {
      return astFunction.args.length >= def.minParams;
    }
    return astFunction.args.length >= def.params.filter(({
      optional
    }) => !optional).length && astFunction.args.length <= def.params.length;
  });
}

/**
 * Given a function signature, returns the parameter at the given position.
 *
 * Takes into account variadic functions (minParams), returning the last
 * parameter if the position is greater than the number of parameters.
 *
 * @param signature
 * @param position
 * @returns
 */
function getParamAtPosition({
  params,
  minParams
}, position) {
  return params.length > position ? params[position] : minParams ? params[params.length - 1] : null;
}

// --- Expression types helpers ---

/**
 * Type guard to check if the type is 'param'
 */
const isParamExpressionType = type => type === 'param';

/**
 * Determines the type of the expression
 */
exports.isParamExpressionType = isParamExpressionType;
function getExpressionType(root, fields, userDefinedColumns) {
  if (!root) {
    return 'unknown';
  }
  if (!isSingleItem(root)) {
    if (root.length === 0) {
      return 'unknown';
    }
    return getExpressionType(root[0], fields, userDefinedColumns);
  }
  if (isLiteralItem(root)) {
    return root.literalType;
  }
  if (isTimeIntervalItem(root)) {
    return 'time_duration';
  }

  // from https://github.com/elastic/elasticsearch/blob/122e7288200ee03e9087c98dff6cebbc94e774aa/docs/reference/esql/functions/kibana/inline_cast.json
  if (isInlineCastItem(root)) {
    switch (root.castType) {
      case 'int':
        return 'integer';
      case 'bool':
        return 'boolean';
      case 'string':
        return 'keyword';
      case 'text':
        return 'keyword';
      case 'datetime':
        return 'date';
      default:
        return root.castType;
    }
  }
  if (isColumnItem(root) && fields && userDefinedColumns) {
    const column = getColumnForASTNode(root, {
      fields,
      userDefinedColumns
    });
    const lastArg = (0, _esqlAst.lastItem)(root.args);
    // If the last argument is a param, we return its type (param literal type)
    // This is useful for cases like `where ??field`
    if (isParam(lastArg)) {
      return lastArg.literalType;
    }
    if (!column) {
      return 'unknown';
    }
    return column.type;
  }
  if (root.type === 'list') {
    return getExpressionType(root.values[0], fields, userDefinedColumns);
  }
  if (isFunctionItem(root)) {
    const fnDefinition = getFunctionDefinition(root.name);
    if (!fnDefinition) {
      return 'unknown';
    }

    /**
     * Special case for COUNT(*) because
     * the "*" column doesn't match any
     * of COUNT's function definitions
     */
    if (fnDefinition.name === 'count' && root.args[0] && isColumnItem(root.args[0]) && root.args[0].name === '*') {
      return 'long';
    }
    if (fnDefinition.name === 'case' && root.args.length) {
      /**
       * The CASE function doesn't fit our system of function definitions
       * and needs special handling. This is imperfect, but it's a start because
       * at least we know that the final argument to case will never be a conditional
       * expression, always a result expression.
       *
       * One problem with this is that if a false case is not provided, the return type
       * will be null, which we aren't detecting. But this is ok because we consider
       * userDefinedColumns and fields to be nullable anyways and account for that during validation.
       */
      return getExpressionType(root.args[root.args.length - 1], fields, userDefinedColumns);
    }
    const signaturesWithCorrectArity = getSignaturesWithMatchingArity(fnDefinition, root);
    if (!signaturesWithCorrectArity.length) {
      return 'unknown';
    }
    const argTypes = root.args.map(arg => getExpressionType(arg, fields, userDefinedColumns));

    // When functions are passed null for any argument, they generally return null
    // This is a special case that is not reflected in our function definitions
    if (argTypes.some(argType => argType === 'null')) return 'null';
    const matchingSignature = signaturesWithCorrectArity.find(signature => {
      return argTypes.every((argType, i) => {
        const param = getParamAtPosition(signature, i);
        return param && (param.type === 'any' || param.type === argType || argType === 'keyword' && ['date', 'date_period'].includes(param.type) || isParamExpressionType(argType));
      });
    });
    if (!matchingSignature) {
      return 'unknown';
    }
    return matchingSignature.returnType === 'any' ? 'unknown' : matchingSignature.returnType;
  }
  return 'unknown';
}

// --- Fields helpers ---

function transformMapToESQLFields(inputMap) {
  const esqlFields = [];
  for (const [, userDefinedColumns] of inputMap) {
    for (const userDefinedColumn of userDefinedColumns) {
      // Only include userDefinedColumns that have a known type
      if (userDefinedColumn.type) {
        esqlFields.push({
          name: userDefinedColumn.name,
          type: userDefinedColumn.type
        });
      }
    }
  }
  return esqlFields;
}
async function getEcsMetadata(resourceRetriever) {
  if (!(resourceRetriever !== null && resourceRetriever !== void 0 && resourceRetriever.getFieldsMetadata)) {
    return undefined;
  }
  const client = await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : resourceRetriever.getFieldsMetadata);
  if (client.find) {
    // Fetch full list of ECS field
    // This list should be cached already by fieldsMetadataClient
    const results = await client.find({
      attributes: ['type']
    });
    return results === null || results === void 0 ? void 0 : results.fields;
  }
}
// Get the fields from the FROM clause, enrich them with ECS metadata
async function getFieldsFromES(query, resourceRetriever) {
  var _resourceRetriever$ge;
  const metadata = await getEcsMetadata();
  const fieldsOfType = await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ge = resourceRetriever.getColumnsFor) === null || _resourceRetriever$ge === void 0 ? void 0 : _resourceRetriever$ge.call(resourceRetriever, {
    query
  }));
  const fieldsWithMetadata = (0, _ecs_metadata_helper.enrichFieldsWithECSInfo)(fieldsOfType || [], metadata);
  return fieldsWithMetadata;
}

/**
 * @param query, the ES|QL query
 * @param commands, the AST commands
 * @param previousPipeFields, the fields from the previous pipe
 * @returns a list of fields that are available for the current pipe
 */
async function getCurrentQueryAvailableFields(query, commands, previousPipeFields) {
  const cacheCopy = new Map();
  previousPipeFields.forEach(field => cacheCopy.set(field.name, field));
  const lastCommand = commands[commands.length - 1];
  const commandDef = getCommandDefinition(lastCommand.name);

  // If the command has a fieldsSuggestionsAfter function, use it to get the fields
  if (commandDef.fieldsSuggestionsAfter) {
    const userDefinedColumns = (0, _user_defined_columns.collectUserDefinedColumns)([lastCommand], cacheCopy, query);
    const arrayOfUserDefinedColumns = transformMapToESQLFields(userDefinedColumns !== null && userDefinedColumns !== void 0 ? userDefinedColumns : new Map());
    return commandDef.fieldsSuggestionsAfter(lastCommand, previousPipeFields, arrayOfUserDefinedColumns);
  } else {
    // If the command doesn't have a fieldsSuggestionsAfter function, use the default behavior
    const userDefinedColumns = (0, _user_defined_columns.collectUserDefinedColumns)(commands, cacheCopy, query);
    const arrayOfUserDefinedColumns = transformMapToESQLFields(userDefinedColumns !== null && userDefinedColumns !== void 0 ? userDefinedColumns : new Map());
    const allFields = (0, _lodash.uniqBy)([...(previousPipeFields !== null && previousPipeFields !== void 0 ? previousPipeFields : []), ...arrayOfUserDefinedColumns], 'name');
    return allFields;
  }
}