"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchHitsInInterval = fetchHitsInInterval;
var _rxjs = require("rxjs");
var _public = require("@kbn/data-plugin/public");
var _discoverUtils = require("@kbn/discover-utils");
var _common = require("@kbn/inspector-plugin/common");
var _date_conversion = require("./date_conversion");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Fetch the hits between a given `interval` up to a maximum of `maxCount` documents.
 * The documents are sorted by `sort`
 *
 * The `searchSource` is assumed to have the appropriate data view
 * and filters set.
 */
async function fetchHitsInInterval(searchSource, timeField, sort, sortDir, interval, searchAfter, maxCount, nanosValue, anchorId, type, services, scopedProfilesManager) {
  var _rawResponse$hits;
  const range = {
    format: 'strict_date_optional_time'
  };
  const [start, stop] = interval;
  if (start) {
    range[sortDir === _public.SortDirection.asc ? 'gte' : 'lte'] = (0, _date_conversion.convertTimeValueToIso)(start, nanosValue);
  }
  if (stop) {
    range[sortDir === _public.SortDirection.asc ? 'lte' : 'gte'] = (0, _date_conversion.convertTimeValueToIso)(stop, nanosValue);
  }
  const adapter = new _common.RequestAdapter();
  const fetch$ = searchSource.setField('size', maxCount).setField('query', {
    query: {
      bool: {
        must: {
          constant_score: {
            filter: {
              range: {
                [timeField]: range
              }
            }
          }
        },
        must_not: {
          ids: {
            values: [anchorId]
          }
        }
      }
    },
    language: 'lucene'
  }).setField('searchAfter', searchAfter).setField('sort', sort).setField('version', true).fetch$({
    disableWarningToasts: true,
    inspector: {
      adapter,
      title: type
    }
  });
  const {
    rawResponse
  } = await (0, _rxjs.lastValueFrom)(fetch$);
  const dataView = searchSource.getField('index');
  const rows = (0, _discoverUtils.buildDataTableRecordList)({
    records: (_rawResponse$hits = rawResponse.hits) === null || _rawResponse$hits === void 0 ? void 0 : _rawResponse$hits.hits,
    dataView,
    processRecord: record => scopedProfilesManager.resolveDocumentProfile({
      record
    })
  });
  const interceptedWarnings = [];
  services.data.search.showWarnings(adapter, warning => {
    interceptedWarnings.push(warning);
    return true; // suppress the default behaviour
  });
  return {
    rows: rows !== null && rows !== void 0 ? rows : [],
    interceptedWarnings
  };
}