"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkGet = void 0;
var _lodash = require("lodash");
var _api = require("../../../common/types/api");
var _runtime_types = require("../../common/runtime_types");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _authorization = require("../../authorization");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves multiple cases by ids.
 */
const bulkGet = async (params, clientArgs) => {
  const {
    services: {
      caseService,
      attachmentService
    },
    logger,
    authorization
  } = clientArgs;
  try {
    const request = (0, _runtime_types.decodeWithExcessOrThrow)(_api.CasesBulkGetRequestRt)(params);
    const cases = await caseService.getCases({
      caseIds: request.ids
    });
    const [validCases, soBulkGetErrors] = (0, _lodash.partition)(cases.saved_objects, caseInfo => caseInfo.error === undefined);
    const {
      authorized: authorizedCases,
      unauthorized: unauthorizedCases
    } = await authorization.getAndEnsureAuthorizedEntities({
      savedObjects: validCases,
      operation: _authorization.Operations.bulkGetCases
    });
    const commentTotals = await attachmentService.getter.getCaseAttatchmentStats({
      caseIds: authorizedCases.map(theCase => theCase.id)
    });
    const flattenedCases = authorizedCases.map(theCase => {
      var _commentTotals$get;
      const {
        userComments,
        alerts
      } = (_commentTotals$get = commentTotals.get(theCase.id)) !== null && _commentTotals$get !== void 0 ? _commentTotals$get : {
        alerts: 0,
        userComments: 0
      };
      return (0, _utils.flattenCaseSavedObject)({
        savedObject: theCase,
        totalComment: userComments,
        totalAlerts: alerts
      });
    });
    const errors = constructErrors(soBulkGetErrors, unauthorizedCases);
    const res = {
      cases: flattenedCases,
      errors
    };
    return (0, _runtime_types.decodeOrThrow)(_api.CasesBulkGetResponseRt)(res);
  } catch (error) {
    var _params$ids;
    const ids = (_params$ids = params.ids) !== null && _params$ids !== void 0 ? _params$ids : [];
    throw (0, _error.createCaseError)({
      message: `Failed to bulk get cases: ${ids.join(', ')}: ${error}`,
      error,
      logger
    });
  }
};
exports.bulkGet = bulkGet;
const constructErrors = (soBulkGetErrors, unauthorizedCases) => {
  const errors = [];
  for (const soError of soBulkGetErrors) {
    errors.push({
      ...(0, _error.generateCaseErrorResponse)(soError.error),
      caseId: soError.id
    });
  }
  for (const theCase of unauthorizedCases) {
    errors.push({
      error: 'Forbidden',
      message: `Unauthorized to access case with owner: "${theCase.attributes.owner}"`,
      status: 403,
      caseId: theCase.id
    });
  }
  return errors;
};