"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.queryMonitorStatusAlert = queryMonitorStatusAlert;
var _pMap = _interopRequireDefault(require("p-map"));
var _lodash = require("lodash");
var _get_search_ping_params = require("./get_search_ping_params");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_MAX_ES_BUCKET_SIZE = 10000;
async function queryMonitorStatusAlert({
  esClient,
  monitorLocationIds,
  range,
  monitorQueryIds,
  numberOfChecks,
  includeRetests = true,
  monitorsData,
  monitors,
  logger
}) {
  const idSize = Math.trunc(DEFAULT_MAX_ES_BUCKET_SIZE / monitorLocationIds.length || 1);
  const pageCount = Math.ceil(monitorQueryIds.length / idSize);
  const upConfigs = {};
  const downConfigs = {};
  for (let i = monitorQueryIds.length - 1; i >= 0; i--) {
    const monitorQueryId = monitorQueryIds[i];
    if (monitorsData[monitorQueryId] === undefined) {
      logger.error(`Monitor ${monitorQueryId} not found in monitorsData, removing from query. This should never happen.`);
      monitorQueryIds.splice(i, 1);
    }
  }
  await (0, _pMap.default)((0, _lodash.times)(pageCount), async i => {
    var _result$aggregations;
    const idsToQuery = monitorQueryIds.slice(i * idSize, i * idSize + idSize);
    const params = (0, _get_search_ping_params.getSearchPingsParams)({
      idSize,
      idsToQuery,
      monitorLocationIds,
      range,
      numberOfChecks,
      includeRetests
    });
    const {
      body: result
    } = await esClient.search(params, 'Monitors status rule query');
    (_result$aggregations = result.aggregations) === null || _result$aggregations === void 0 ? void 0 : _result$aggregations.id.buckets.forEach(({
      location,
      key: queryId
    }) => {
      const locationSummaries = location.buckets.map(({
        key: locationId,
        totalChecks,
        downChecks
      }) => {
        return {
          locationId,
          totalChecks,
          downChecks
        };
      });

      // discard any locations that are not in the monitorLocationsMap for the given monitor as well as those which are
      // in monitorLocationsMap but not in listOfLocations
      const monLocations = monitorsData[queryId].locations;
      const monQueriedLocations = (0, _lodash.intersection)(monLocations, monitorLocationIds);
      monQueriedLocations === null || monQueriedLocations === void 0 ? void 0 : monQueriedLocations.forEach(monLocationId => {
        const locationSummary = locationSummaries.find(summary => summary.locationId === monLocationId);
        if (locationSummary) {
          var _latestPing$summary$u, _latestPing$summary, _latestPing$monitor$d;
          const {
            totalChecks,
            downChecks
          } = locationSummary;
          const latestPing = totalChecks.hits.hits[0]._source;
          const downCount = downChecks.doc_count;
          const isLatestPingUp = ((_latestPing$summary$u = (_latestPing$summary = latestPing.summary) === null || _latestPing$summary === void 0 ? void 0 : _latestPing$summary.up) !== null && _latestPing$summary$u !== void 0 ? _latestPing$summary$u : 0) > 0;
          const configId = latestPing.config_id;
          const monitorQueryId = latestPing.monitor.id;
          const isValidPing = (0, _helpers.calculateIsValidPing)({
            scheduleInMs: monitorsData[monitorQueryId].scheduleInMs,
            previousRunEndTimeISO: latestPing['@timestamp'],
            previousRunDurationUs: (_latestPing$monitor$d = latestPing.monitor.duration) === null || _latestPing$monitor$d === void 0 ? void 0 : _latestPing$monitor$d.us
          });
          const meta = {
            latestPing,
            configId,
            monitorQueryId,
            locationId: monLocationId,
            timestamp: latestPing['@timestamp'],
            checks: {
              downWithinXChecks: totalChecks.hits.hits.reduce((acc, curr) => {
                var _curr$_source$summary;
                return acc + (((_curr$_source$summary = curr._source.summary.down) !== null && _curr$_source$summary !== void 0 ? _curr$_source$summary : 0) > 0 ? 1 : 0);
              }, 0),
              down: downCount
            },
            status: 'up'
          };
          if (isValidPing && downCount > 0) {
            downConfigs[`${configId}-${monLocationId}`] = {
              ...meta,
              status: 'down'
            };
          }
          if (isValidPing && isLatestPingUp) {
            upConfigs[`${configId}-${monLocationId}`] = {
              ...meta,
              status: 'up'
            };
          }
        }
      });
    });
  }, {
    concurrency: 5
  });
  const pendingConfigs = (0, _helpers.getPendingConfigs)({
    monitorQueryIds,
    monitorLocationIds,
    upConfigs,
    downConfigs,
    monitorsData,
    monitors,
    logger
  });
  const configStats = (0, _helpers.getConfigStats)({
    downConfigs,
    upConfigs,
    pendingConfigs,
    monitorQueryIds
  });
  return {
    upConfigs,
    downConfigs,
    pendingConfigs,
    enabledMonitorQueryIds: monitorQueryIds,
    configStats
  };
}