"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkAccess = checkAccess;
exports.checkAccessBulk = checkAccessBulk;
exports.getDataStream = getDataStream;
exports.getDataStreamLifecycle = getDataStreamLifecycle;
exports.getUnmanagedElasticsearchAssetDetails = getUnmanagedElasticsearchAssetDetails;
exports.getUnmanagedElasticsearchAssets = getUnmanagedElasticsearchAssets;
var _definition_not_found_error = require("./errors/definition_not_found_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getDataStreamLifecycle(dataStream) {
  if (!dataStream) {
    return {
      error: {
        message: 'Data stream not found'
      }
    };
  }
  if (dataStream.next_generation_managed_by === 'Index Lifecycle Management') {
    return {
      ilm: {
        policy: dataStream.ilm_policy
      }
    };
  }
  if (dataStream.next_generation_managed_by === 'Data stream lifecycle') {
    var _dataStream$lifecycle;
    const retention = (_dataStream$lifecycle = dataStream.lifecycle) === null || _dataStream$lifecycle === void 0 ? void 0 : _dataStream$lifecycle.data_retention;
    return {
      dsl: {
        data_retention: retention ? String(retention) : undefined
      }
    };
  }
  if (dataStream.next_generation_managed_by === 'Unmanaged') {
    return {
      disabled: {}
    };
  }
  return {
    error: {
      message: `Unknown data stream lifecycle state [${dataStream.next_generation_managed_by}]`
    }
  };
}
async function getUnmanagedElasticsearchAssets({
  dataStream,
  scopedClusterClient
}) {
  var _dataStream$indices$a, _currentIndex$writeIn, _currentIndex$writeIn2;
  // retrieve linked index template, component template and ingest pipeline
  const templateName = dataStream.template;
  const componentTemplates = [];
  const template = await scopedClusterClient.asCurrentUser.indices.getIndexTemplate({
    name: templateName
  });
  if (template.index_templates.length) {
    var _template$index_templ;
    (_template$index_templ = template.index_templates[0].index_template.composed_of) === null || _template$index_templ === void 0 ? void 0 : _template$index_templ.forEach(componentTemplateName => {
      componentTemplates.push(componentTemplateName);
    });
  }
  const writeIndexName = (_dataStream$indices$a = dataStream.indices.at(-1)) === null || _dataStream$indices$a === void 0 ? void 0 : _dataStream$indices$a.index_name;
  const currentIndex = await scopedClusterClient.asCurrentUser.indices.get({
    index: writeIndexName
  });
  const ingestPipelineId = (_currentIndex$writeIn = currentIndex[writeIndexName].settings) === null || _currentIndex$writeIn === void 0 ? void 0 : (_currentIndex$writeIn2 = _currentIndex$writeIn.index) === null || _currentIndex$writeIn2 === void 0 ? void 0 : _currentIndex$writeIn2.default_pipeline;
  return {
    ingestPipeline: ingestPipelineId,
    componentTemplates,
    indexTemplate: templateName,
    dataStream: dataStream.name
  };
}
async function fetchComponentTemplate(scopedClusterClient, name) {
  try {
    var _response$component_t;
    const response = await scopedClusterClient.asCurrentUser.cluster.getComponentTemplate({
      name
    });
    return (_response$component_t = response.component_templates.find(template => template.name === name)) !== null && _response$component_t !== void 0 ? _response$component_t : {
      name,
      component_template: undefined
    };
  } catch (e) {
    var _e$meta;
    if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) === 404) {
      return {
        name,
        component_template: undefined
      };
    }
    throw e;
  }
}
async function fetchComponentTemplates(scopedClusterClient, names, allIndexTemplates) {
  const templates = await Promise.all(names.map(name => fetchComponentTemplate(scopedClusterClient, name)));
  return templates.filter(template => template !== undefined).map(componentTemplate => ({
    ...componentTemplate,
    used_by: allIndexTemplates.filter(template => {
      var _template$index_templ2;
      return (_template$index_templ2 = template.index_template.composed_of) === null || _template$index_templ2 === void 0 ? void 0 : _template$index_templ2.includes(componentTemplate.name);
    }).map(template => template.name)
  }));
}
async function fetchIngestPipeline(scopedClusterClient, pipelineId) {
  if (!pipelineId) return undefined;
  const response = await scopedClusterClient.asCurrentUser.ingest.getPipeline({
    id: pipelineId
  });
  return {
    ...response[pipelineId],
    name: pipelineId
  };
}
async function getUnmanagedElasticsearchAssetDetails({
  scopedClusterClient,
  assets
}) {
  const allIndexTemplates = (await scopedClusterClient.asCurrentUser.indices.getIndexTemplate()).index_templates;
  const [ingestPipeline, componentTemplates, dataStreamResponse] = await Promise.all([fetchIngestPipeline(scopedClusterClient, assets.ingestPipeline), fetchComponentTemplates(scopedClusterClient, assets.componentTemplates, allIndexTemplates), scopedClusterClient.asCurrentUser.indices.getDataStream({
    name: assets.dataStream
  })]);
  const indexTemplate = allIndexTemplates.find(template => template.name === assets.indexTemplate);
  if (!indexTemplate) {
    throw new Error(`Index template ${assets.indexTemplate} not found`);
  }
  return {
    ingestPipeline,
    componentTemplates,
    indexTemplate,
    dataStream: dataStreamResponse.data_streams[0]
  };
}
async function checkAccess({
  name,
  scopedClusterClient
}) {
  return checkAccessBulk({
    names: [name],
    scopedClusterClient
  }).then(privileges => privileges[name]);
}
async function checkAccessBulk({
  names,
  scopedClusterClient
}) {
  if (!names.length) {
    return {};
  }
  const hasPrivilegesResponse = await scopedClusterClient.asCurrentUser.security.hasPrivileges({
    index: [{
      names,
      privileges: ['read', 'write']
    }]
  });
  return Object.fromEntries(names.map(name => {
    const hasReadAccess = hasPrivilegesResponse.index[name].read === true;
    const hasWriteAccess = hasPrivilegesResponse.index[name].write === true;
    return [name, {
      read: hasReadAccess,
      write: hasWriteAccess
    }];
  }));
}
async function getDataStream({
  name,
  scopedClusterClient
}) {
  let dataStream;
  try {
    const response = await scopedClusterClient.asCurrentUser.indices.getDataStream({
      name
    });
    dataStream = response.data_streams[0];
  } catch (e) {
    var _e$meta2;
    if (((_e$meta2 = e.meta) === null || _e$meta2 === void 0 ? void 0 : _e$meta2.statusCode) === 404) {
      // fall through and throw not found
    } else {
      throw e;
    }
  }
  if (!dataStream) {
    throw new _definition_not_found_error.DefinitionNotFoundError(`Stream definition for ${name} not found.`);
  }
  return dataStream;
}