"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteDataStream = deleteDataStream;
exports.getTemplateLifecycle = getTemplateLifecycle;
exports.updateDataStreamsLifecycle = updateDataStreamsLifecycle;
exports.updateOrRolloverDataStream = updateOrRolloverDataStream;
exports.upsertDataStream = upsertDataStream;
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _retry = require("../helpers/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function upsertDataStream({
  esClient,
  name,
  logger
}) {
  const dataStreamExists = await esClient.indices.exists({
    index: name
  });
  if (dataStreamExists) {
    return;
  }
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.indices.createDataStream({
      name
    }), {
      logger
    });
    logger.debug(() => `Installed data stream: ${name}`);
  } catch (error) {
    logger.error(`Error creating data stream: ${error.message}`);
    throw error;
  }
}
async function deleteDataStream({
  esClient,
  name,
  logger
}) {
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.indices.deleteDataStream({
      name
    }, {
      ignore: [404]
    }), {
      logger
    });
  } catch (error) {
    logger.error(`Error deleting data stream: ${error.message}`);
    throw error;
  }
}
async function updateOrRolloverDataStream({
  esClient,
  name,
  logger
}) {
  const dataStreams = await esClient.indices.getDataStream({
    name
  });
  for (const dataStream of dataStreams.data_streams) {
    // simulate index and try to patch the write index
    // if that doesn't work, roll it over
    const simulatedIndex = await esClient.indices.simulateIndexTemplate({
      name
    });
    const writeIndex = dataStream.indices.at(-1);
    if (!writeIndex) {
      continue;
    }
    try {
      // Apply blocklist to avoid changing settings we don't want to
      const simulatedIndexSettings = (0, _lodash.omit)(simulatedIndex.template.settings, ['index.codec', 'index.mapping.ignore_malformed', 'index.mode', 'index.logsdb.sort_on_host_name']);
      await (0, _retry.retryTransientEsErrors)(() => Promise.all([esClient.indices.putMapping({
        index: writeIndex.index_name,
        properties: simulatedIndex.template.mappings.properties
      }), esClient.indices.putSettings({
        index: writeIndex.index_name,
        settings: simulatedIndexSettings
      })]), {
        logger
      });
    } catch (error) {
      if (typeof error.message !== 'string' || !error.message.includes('illegal_argument_exception')) {
        throw error;
      }
      try {
        await (0, _retry.retryTransientEsErrors)(() => esClient.indices.rollover({
          alias: dataStream.name
        }), {
          logger
        });
        logger.debug(() => `Rolled over data stream: ${dataStream.name}`);
      } catch (rolloverError) {
        logger.error(`Error rolling over data stream: ${error.message}`);
        throw error;
      }
    }
  }
}
async function updateDataStreamsLifecycle({
  esClient,
  logger,
  names,
  lifecycle,
  isServerless
}) {
  try {
    if ((0, _streamsSchema.isIlmLifecycle)(lifecycle)) {
      await putDataStreamsSettings({
        esClient,
        names,
        logger,
        settings: {
          'index.lifecycle.name': lifecycle.ilm.policy,
          'index.lifecycle.prefer_ilm': true
        }
      });
    } else if ((0, _streamsSchema.isDslLifecycle)(lifecycle)) {
      await (0, _retry.retryTransientEsErrors)(() => esClient.indices.putDataLifecycle({
        name: names,
        data_retention: lifecycle.dsl.data_retention
      }), {
        logger
      });
      if (!isServerless) {
        // we don't need overrides for serverless since data streams can
        // only be managed by dsl
        await putDataStreamsSettings({
          esClient,
          names,
          logger,
          settings: {
            'index.lifecycle.name': null,
            'index.lifecycle.prefer_ilm': false
          }
        });
      }
    } else if ((0, _streamsSchema.isInheritLifecycle)(lifecycle)) {
      // classic streams only - inheriting a lifecycle means falling back to
      // the template configuration. if we find a DSL we need to set it
      // explicitly since there is no way to fall back to the template value,
      // for ILM or disabled we only have to unset any overrides
      await Promise.all(names.map(async name => {
        const {
          template
        } = await (0, _retry.retryTransientEsErrors)(() => esClient.indices.simulateIndexTemplate({
          name
        }), {
          logger
        });
        const templateLifecycle = getTemplateLifecycle(template);
        if ((0, _streamsSchema.isDslLifecycle)(templateLifecycle)) {
          await (0, _retry.retryTransientEsErrors)(() => esClient.indices.putDataLifecycle({
            name,
            data_retention: templateLifecycle.dsl.data_retention
          }), {
            logger
          });
        } else {
          await (0, _retry.retryTransientEsErrors)(() => esClient.indices.deleteDataLifecycle({
            name
          }), {
            logger
          });
        }
        if (!isServerless) {
          // unset any overriden settings
          await putDataStreamsSettings({
            esClient,
            names: [name],
            logger,
            settings: {
              'index.lifecycle.name': null,
              'index.lifecycle.prefer_ilm': null
            }
          });
        }
      }));
    }
  } catch (err) {
    logger.error(`Error updating data stream lifecycle: ${err.message}`);
    throw err;
  }
}
async function putDataStreamsSettings({
  esClient,
  names,
  logger,
  settings
}) {
  await (0, _retry.retryTransientEsErrors)(() =>
  // TODO: use client method once available
  esClient.transport.request({
    method: 'PUT',
    path: `/_data_stream/${names.join(',')}/_settings`,
    body: settings
  }), {
    logger
  });
}
function getTemplateLifecycle(template) {
  var _template$lifecycle, _template$settings$in, _template$settings$in2, _template$settings$in3, _template$settings$in4, _template$settings$in5, _template$settings$in6;
  const toBoolean = value => {
    if (typeof value === 'boolean') {
      return value;
    }
    return value === 'true';
  };
  const hasEffectiveDsl = toBoolean((_template$lifecycle = template.lifecycle) === null || _template$lifecycle === void 0 ? void 0 : _template$lifecycle.enabled) && !(toBoolean((_template$settings$in = template.settings.index) === null || _template$settings$in === void 0 ? void 0 : (_template$settings$in2 = _template$settings$in.lifecycle) === null || _template$settings$in2 === void 0 ? void 0 : _template$settings$in2.prefer_ilm) && (_template$settings$in3 = template.settings.index) !== null && _template$settings$in3 !== void 0 && (_template$settings$in4 = _template$settings$in3.lifecycle) !== null && _template$settings$in4 !== void 0 && _template$settings$in4.name);
  if (hasEffectiveDsl) {
    return {
      dsl: {
        data_retention: template.lifecycle.data_retention
      }
    };
  }
  if ((_template$settings$in5 = template.settings.index) !== null && _template$settings$in5 !== void 0 && (_template$settings$in6 = _template$settings$in5.lifecycle) !== null && _template$settings$in6 !== void 0 && _template$settings$in6.name) {
    // if dsl is not enabled and a policy is set, the ilm will be effective
    // regardless of the prefer_ilm setting
    return {
      ilm: {
        policy: template.settings.index.lifecycle.name
      }
    };
  }
  return {
    disabled: {}
  };
}