"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.measureInteraction = measureInteraction;
var _timerange = require("@kbn/timerange");
var _performance_markers = require("../../performance_markers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function measureInteraction(pathname) {
  performance.mark(_performance_markers.perfomanceMarkers.startPageChange);
  return {
    /**
     * Marks the end of the page ready state and measures the performance between the start of the page change and the end of the page ready state.
     * @param pathname - The pathname of the page.
     * @param customMetrics - Custom metrics to be included in the performance measure.
     */
    pageReady(eventData) {
      var _eventData$meta, _eventData$meta2, _eventData$meta3;
      const performanceMeta = {};
      performance.mark(_performance_markers.perfomanceMarkers.endPageReady);
      if (eventData !== null && eventData !== void 0 && (_eventData$meta = eventData.meta) !== null && _eventData$meta !== void 0 && _eventData$meta.rangeFrom && eventData !== null && eventData !== void 0 && (_eventData$meta2 = eventData.meta) !== null && _eventData$meta2 !== void 0 && _eventData$meta2.rangeTo) {
        const {
          rangeFrom,
          rangeTo
        } = eventData.meta;

        // Convert the date range  to epoch timestamps (in milliseconds)
        const dateRangesInEpoch = (0, _timerange.getDateRange)({
          from: rangeFrom,
          to: rangeTo
        });
        performanceMeta.queryRangeSecs = (0, _timerange.getTimeDifferenceInSeconds)(dateRangesInEpoch);
        performanceMeta.queryFromOffsetSecs = rangeFrom === 'now' ? 0 : (0, _timerange.getOffsetFromNowInSeconds)(dateRangesInEpoch.startDate);
        performanceMeta.queryToOffsetSecs = rangeTo === 'now' ? 0 : (0, _timerange.getOffsetFromNowInSeconds)(dateRangesInEpoch.endDate);
      }
      if (eventData !== null && eventData !== void 0 && (_eventData$meta3 = eventData.meta) !== null && _eventData$meta3 !== void 0 && _eventData$meta3.description) {
        performanceMeta.description = eventData.meta.description;
      }
      if (performance.getEntriesByName(_performance_markers.perfomanceMarkers.startPageChange).length > 0 && performance.getEntriesByName(_performance_markers.perfomanceMarkers.endPageReady).length > 0) {
        performance.measure(`[ttfmp:initial] - ${pathname}`, {
          detail: {
            eventName: 'kibana:plugin_render_time',
            type: 'kibana:performance',
            customMetrics: eventData === null || eventData === void 0 ? void 0 : eventData.customMetrics,
            meta: {
              ...performanceMeta,
              isInitialLoad: true
            }
          },
          start: _performance_markers.perfomanceMarkers.startPageChange,
          end: _performance_markers.perfomanceMarkers.endPageReady
        });

        // Clean up the marks once the measure is done
        performance.clearMarks(_performance_markers.perfomanceMarkers.startPageChange);
        performance.clearMarks(_performance_markers.perfomanceMarkers.endPageReady);
      }
      if (performance.getEntriesByName(_performance_markers.perfomanceMarkers.startPageRefresh).length > 0 && performance.getEntriesByName(_performance_markers.perfomanceMarkers.endPageReady).length > 0) {
        performance.measure(`[ttfmp:refresh] - ${pathname}`, {
          detail: {
            eventName: 'kibana:plugin_render_time',
            type: 'kibana:performance',
            customMetrics: eventData === null || eventData === void 0 ? void 0 : eventData.customMetrics,
            meta: {
              ...performanceMeta,
              isInitialLoad: false
            }
          },
          start: _performance_markers.perfomanceMarkers.startPageRefresh,
          end: _performance_markers.perfomanceMarkers.endPageReady
        });

        // // Clean up the marks once the measure is done
        performance.clearMarks(_performance_markers.perfomanceMarkers.startPageRefresh);
        performance.clearMarks(_performance_markers.perfomanceMarkers.endPageReady);
      }
    },
    pageRefreshStart() {
      performance.mark(_performance_markers.perfomanceMarkers.startPageRefresh);
    }
  };
}