"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.topItemsHandlerFactory = void 0;
var _async = require("async");
var _lodash = require("lodash");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _i18n = require("@kbn/i18n");
var _stream_reducer = require("@kbn/aiops-log-rate-analysis/api/stream_reducer");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _fetch_top_categories = require("@kbn/aiops-log-rate-analysis/queries/fetch_top_categories");
var _fetch_top_terms = require("@kbn/aiops-log-rate-analysis/queries/fetch_top_terms");
var _queue_field_candidates = require("@kbn/aiops-log-rate-analysis/queue_field_candidates");
var _constants = require("../response_stream_utils/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const topItemsHandlerFactory = ({
  abortSignal,
  esClient,
  logDebugMessage,
  logger,
  requestBody,
  responseStream,
  stateHandler,
  version
}) => async ({
  keywordFieldCandidates,
  textFieldCandidates
}) => {
  var _requestBody$override, _requestBody$override2, _requestBody$override3, _overrides$significan, _overrides2, _overrides2$significa;
  // This will store the combined count of detected log patterns and keywords
  let fieldValuePairsCount = 0;
  if (version === '3') {
    var _overrides;
    const overridesRemainingTextFieldCandidates = (_overrides = requestBody.overrides) === null || _overrides === void 0 ? void 0 : _overrides.remainingTextFieldCandidates;
    if (Array.isArray(overridesRemainingTextFieldCandidates)) {
      textFieldCandidates.push(...overridesRemainingTextFieldCandidates);
    }
  }
  const topCategories = [];
  topCategories.push(...((_requestBody$override = (_requestBody$override2 = requestBody.overrides) === null || _requestBody$override2 === void 0 ? void 0 : (_requestBody$override3 = _requestBody$override2.significantItems) === null || _requestBody$override3 === void 0 ? void 0 : _requestBody$override3.filter(d => d.type === _mlAggUtils.SIGNIFICANT_ITEM_TYPE.LOG_PATTERN)) !== null && _requestBody$override !== void 0 ? _requestBody$override : []));
  const topTerms = [];
  topTerms.push(...((_overrides$significan = (_overrides2 = requestBody.overrides) === null || _overrides2 === void 0 ? void 0 : (_overrides2$significa = _overrides2.significantItems) === null || _overrides2$significa === void 0 ? void 0 : _overrides2$significa.filter(d => d.type === _mlAggUtils.SIGNIFICANT_ITEM_TYPE.KEYWORD)) !== null && _overrides$significan !== void 0 ? _overrides$significan : []));
  let remainingKeywordFieldCandidates;
  let loadingStepSizeTopTerms = _constants.PROGRESS_STEP_P_VALUES;
  if (version === '2') {
    var _overrides3;
    const overridesRemainingFieldCandidates = (_overrides3 = requestBody.overrides) === null || _overrides3 === void 0 ? void 0 : _overrides3.remainingFieldCandidates;
    if (Array.isArray(overridesRemainingFieldCandidates)) {
      var _requestBody$override4, _requestBody$override5;
      keywordFieldCandidates.push(...overridesRemainingFieldCandidates);
      remainingKeywordFieldCandidates = overridesRemainingFieldCandidates;
      loadingStepSizeTopTerms = _constants.LOADED_FIELD_CANDIDATES + _constants.PROGRESS_STEP_P_VALUES - ((_requestBody$override4 = (_requestBody$override5 = requestBody.overrides) === null || _requestBody$override5 === void 0 ? void 0 : _requestBody$override5.loaded) !== null && _requestBody$override4 !== void 0 ? _requestBody$override4 : _constants.PROGRESS_STEP_P_VALUES);
    } else {
      remainingKeywordFieldCandidates = keywordFieldCandidates;
    }
  } else if (version === '3') {
    var _overrides4;
    const overridesRemainingKeywordFieldCandidates = (_overrides4 = requestBody.overrides) === null || _overrides4 === void 0 ? void 0 : _overrides4.remainingKeywordFieldCandidates;
    if (Array.isArray(overridesRemainingKeywordFieldCandidates)) {
      var _requestBody$override6, _requestBody$override7;
      keywordFieldCandidates.push(...overridesRemainingKeywordFieldCandidates);
      remainingKeywordFieldCandidates = overridesRemainingKeywordFieldCandidates;
      loadingStepSizeTopTerms = _constants.LOADED_FIELD_CANDIDATES + _constants.PROGRESS_STEP_P_VALUES - ((_requestBody$override6 = (_requestBody$override7 = requestBody.overrides) === null || _requestBody$override7 === void 0 ? void 0 : _requestBody$override7.loaded) !== null && _requestBody$override6 !== void 0 ? _requestBody$override6 : _constants.PROGRESS_STEP_P_VALUES);
    } else {
      remainingKeywordFieldCandidates = keywordFieldCandidates;
    }
  }
  logDebugMessage('Fetch top items.');
  const topTermsQueueChunks = [...(0, _lodash.chunk)(keywordFieldCandidates, _queue_field_candidates.QUEUE_CHUNKING_SIZE).map(d => ({
    keywordFieldCandidates: d
  })), ...(0, _lodash.chunk)(textFieldCandidates, _queue_field_candidates.QUEUE_CHUNKING_SIZE).map(d => ({
    textFieldCandidates: d
  }))];
  const loadingStepSize = 1 / topTermsQueueChunks.length * loadingStepSizeTopTerms;
  const topTermsQueue = (0, _async.queue)(async function (payload) {
    if ((0, _queue_field_candidates.isKeywordFieldCandidates)(payload)) {
      const {
        keywordFieldCandidates: fieldNames
      } = payload;
      let fetchedTopTerms;
      try {
        fetchedTopTerms = await (0, _fetch_top_terms.fetchTopTerms)({
          esClient,
          logger,
          emitError: responseStream.pushError,
          abortSignal,
          arguments: {
            ...requestBody,
            fieldNames,
            sampleProbability: stateHandler.sampleProbability()
          }
        });
      } catch (e) {
        if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
          logger.error(`Failed to fetch top items for ${fieldNames.join()}, got: \n${e.toString()}`);
          responseStream.pushError(`Failed to fetch top items for ${fieldNames.join()}.`);
        }
        return;
      }
      remainingKeywordFieldCandidates = remainingKeywordFieldCandidates.filter(d => !fieldNames.includes(d));
      if (fetchedTopTerms.length > 0) {
        topTerms.push(...fetchedTopTerms);
        responseStream.push((0, _stream_reducer.addSignificantItems)(fetchedTopTerms));
      }
      stateHandler.loaded(loadingStepSize, false);
      responseStream.push((0, _stream_reducer.updateLoadingState)({
        ccsWarning: false,
        loaded: stateHandler.loaded(),
        loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.identifiedFieldValuePairs', {
          defaultMessage: 'Identified {fieldValuePairsCount, plural, one {# significant field/value pair} other {# significant field/value pairs}}.',
          values: {
            fieldValuePairsCount
          }
        }),
        remainingKeywordFieldCandidates
      }));
    } else if ((0, _queue_field_candidates.isTextFieldCandidates)(payload)) {
      const {
        textFieldCandidates: fieldNames
      } = payload;
      const topCategoriesForField = await (0, _fetch_top_categories.fetchTopCategories)({
        esClient,
        logger,
        emitError: responseStream.pushError,
        abortSignal,
        arguments: {
          ...requestBody,
          fieldNames,
          sampleProbability: stateHandler.sampleProbability()
        }
      });
      if (topCategoriesForField.length > 0) {
        topCategories.push(...topCategoriesForField);
        responseStream.push((0, _stream_reducer.addSignificantItems)(topCategoriesForField));
        fieldValuePairsCount += topCategoriesForField.length;
      }
      stateHandler.loaded(loadingStepSize, false);
    }
  }, _constants.MAX_CONCURRENT_QUERIES);
  topTermsQueue.push(topTermsQueueChunks, err => {
    if (err) {
      logger.error(`Failed to fetch p-values.', got: \n${err.toString()}`);
      responseStream.pushError(`Failed to fetch p-values.`);
      topTermsQueue.kill();
      responseStream.end();
    } else if (stateHandler.shouldStop()) {
      logDebugMessage('shouldStop fetching p-values.');
      topTermsQueue.kill();
      responseStream.end();
    }
  });
  await topTermsQueue.drain();
  fieldValuePairsCount = topCategories.length + topTerms.length;
  if (fieldValuePairsCount === 0) {
    logDebugMessage('Stopping analysis, did not find any categories or terms.');
    responseStream.endWithUpdatedLoadingState();
    return;
  }
  return {
    fieldValuePairsCount,
    significantCategories: topCategories,
    significantTerms: topTerms
  };
};
exports.topItemsHandlerFactory = topItemsHandlerFactory;