"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runWorkflow = exports.createWorkflowRunner = void 0;
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns a step runner already scoped to the given context.
 */
const createWorkflowRunner = params => {
  return args => {
    return runWorkflow({
      ...params,
      ...args
    });
  };
};
exports.createWorkflowRunner = createWorkflowRunner;
const runWorkflow = async ({
  workflowDefinition,
  inputs,
  internalContext
}) => {
  // update the execution state - context is already a copy, it's safe to reference
  internalContext.executionState = (0, _utils.enterWorkflow)({
    parent: internalContext.executionState,
    workflowId: workflowDefinition.id
  });
  const {
    eventHandler
  } = internalContext;
  const internalRunner = internalContext.getRunner();
  const state = (0, _utils.createInitialState)({
    inputs
  });
  const output = {};
  if (workflowDefinition.type === 'function') {
    const workflowResult = await workflowDefinition.handler({
      inputs,
      eventHandler
    });
    Object.entries(workflowResult).forEach(([key, value]) => {
      output[key] = value;
    });
  } else {
    const stepDefinitions = workflowDefinition.steps;
    for (let i = 0; i < stepDefinitions.length; i++) {
      const nodeDefinition = stepDefinitions[i];
      await internalRunner.runNode({
        nodeDefinition,
        state
      });
    }

    // TODO: maybe need to improve the logic?
    workflowDefinition.outputs.forEach(outputDef => {
      output[outputDef.name] = state.get(outputDef.ref);
    });
  }
  return {
    runId: internalContext.executionState.runId,
    output
  };
};
exports.runWorkflow = runWorkflow;