"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSearchAgentTool = void 0;
var _zod = require("@kbn/zod");
var _tools = require("@langchain/core/tools");
var _run_search_agent = require("./run_search_agent");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createSearchAgentTool = ({
  toolsProvider,
  chatModel,
  logger
}) => {
  const runSearchAgent = (0, _run_search_agent.getSearchAgentCaller)({
    chatModel,
    toolsProvider,
    logger
  });
  return new _tools.DynamicStructuredTool({
    name: 'search',
    description: `
    Use this tool to search for knowledge.

    The tool accepts a natural language query as input, and will delegates the search to an specialized AI agent.

    You can use the 'context' parameter to add additional context that could be useful for the search agent. It can be
    useful for example to add a quick summary of what was previously searched, or a summary of the parts of the conversation
    that could be useful to know for the agent to perform the search task.

    The output of the tool will be the documents, or content, that the search agent retrieved, and a summary of
    the documents in context of the 'query' and 'context' provided as input.

    Examples:

    - If the user asks "Can you find the github issues currently assigned to me",
      You can search for "find github issues currently assigned to the user"
    - If the user asks for "Hello. What is the organization's policy about remote work",
      You can search for "remote work policy in the organization"
    `,
    metadata: {},
    schema: _zod.z.object({
      query: _zod.z.string().describe(`The natural language query to search for`),
      context: _zod.z.string().optional().describe(`Optional additional context that could be useful for the search agent`)
    }),
    responseFormat: 'content_and_artifact',
    func: async ({
      query,
      context
    }) => {
      const result = await runSearchAgent({
        query,
        context
      });
      return [result, result];
    }
  });
};
exports.createSearchAgentTool = createSearchAgentTool;