"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.McpGatewaySessionImpl = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _wciCommon = require("@kbn/wci-common");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Represents a "session" against all registered MCP servers.
 * Allows to aggregate and list tools from all servers,
 * and to close the connectors to all of them in a single call.
 */

class McpGatewaySessionImpl {
  constructor({
    providers,
    logger
  }) {
    (0, _defineProperty2.default)(this, "providers", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "sessionClients", {});
    (0, _defineProperty2.default)(this, "connected", false);
    this.providers = providers;
    this.logger = logger;
  }
  async ensureConnected() {
    if (this.connected) {
      return;
    }
    this.sessionClients = await this.providers.reduce(async (accPromise, provider) => {
      const acc = await accPromise;
      try {
        acc[provider.id] = await provider.connect();
      } catch (e) {
        this.logger.warn(`Error connecting integration: ${provider.id}`);
      }
      return acc;
    }, Promise.resolve({}));
    this.connected = true;
  }
  async listTools() {
    await this.ensureConnected();
    // TODO: memoize / cache
    return (0, _utils.listClientsTools)({
      clients: this.sessionClients,
      logger: this.logger
    });
  }
  async executeTool(serverToolName, params) {
    await this.ensureConnected();
    const {
      integrationId,
      toolName
    } = (0, _wciCommon.parseToolName)(serverToolName);
    const client = this.sessionClients[integrationId];
    if (!client) {
      throw new Error(`Client not found: ${integrationId}`);
    }
    const response = await client.callTool({
      name: toolName,
      arguments: params
    });
    return response;
  }
  async close() {
    if (!this.connected) {
      return;
    }
    await Promise.all(Object.values(this.sessionClients).map(session => session.disconnect()));
    this.sessionClients = {};
    this.connected = false;
  }
}
exports.McpGatewaySessionImpl = McpGatewaySessionImpl;