"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAgentRunner = void 0;
var _rxjs = require("rxjs");
var _agent_graph = require("./agent_graph");
var _utils = require("./utils");
var _graph_events = require("./graph_events");
var _constants = require("./constants");
var _tracing = require("./tracing");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createAgentRunner = async ({
  logger,
  agent,
  chatModel,
  createSession,
  tracingConfig
}) => {
  const session = await createSession();
  const tracers = (0, _tracing.getTracers)({
    config: tracingConfig
  });
  const closeSession = () => {
    session.close().catch(err => {
      logger.warn(`Error disconnecting integrations: ${err.message}`);
    });
  };
  const agentGraph = await (0, _agent_graph.createAgentGraph)({
    agent,
    chatModel,
    session,
    logger: logger.get('agent.graph')
  });
  return {
    run: async ({
      previousEvents
    }) => {
      const initialMessages = (0, _utils.conversationEventsToMessages)(previousEvents);
      const runName = 'defaultAgentGraph';
      const eventStream = agentGraph.streamEvents({
        initialMessages
      }, {
        version: 'v2',
        runName,
        metadata: {
          ...(0, _graph_events.getGraphMeta)({
            graphName: _constants.graphNames.mainAgent
          }),
          agentId: agent.id
        },
        recursionLimit: 10,
        callbacks: [...tracers]
      });
      const events$ = (0, _rxjs.from)(eventStream).pipe((0, _rxjs.filter)(isStreamEvent), (0, _graph_events.convertGraphEvents)(), (0, _rxjs.shareReplay)());
      events$.subscribe({
        complete: () => {
          logger.debug('Completed the event stream');
          closeSession();
        },
        error: () => {
          logger.debug('Encountered error in the event stream');
          closeSession();
        }
      });
      return {
        events$
      };
    }
  };
};
exports.createAgentRunner = createAgentRunner;
const isStreamEvent = input => {
  return 'event' in input;
};