"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentClientImpl = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _agents = require("../../saved_objects/agents");
var _errors = require("../../errors");
var _so_filters = require("../../utils/so_filters");
var _convert_model = require("./convert_model");
var _color = require("../../utils/color");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Agent client scoped to a current user.
 * All APIs exposed by the client are performing actions on behalf of the user,
 * only accessing/returning agents that are accessible to the user.
 */

class AgentClientImpl {
  constructor({
    client,
    user,
    logger
  }) {
    (0, _defineProperty2.default)(this, "client", void 0);
    (0, _defineProperty2.default)(this, "user", void 0);
    // @ts-expect-error will be used at some point
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.client = client;
    this.user = user;
    this.logger = logger;
  }
  async list() {
    const builder = (0, _so_filters.createBuilder)(_agents.agentTypeName);
    const filter = builder.or(builder.equals('user_id', this.user.id), builder.equals('access_control.public', true)).toKQL();
    const {
      saved_objects: results
    } = await this.client.find({
      type: _agents.agentTypeName,
      filter,
      perPage: 1000
    });
    return results.map(_convert_model.savedObjectToModel);
  }
  async get({
    agentId
  }) {
    const conversationSo = await this._rawGet({
      agentId
    });
    return (0, _convert_model.savedObjectToModel)(conversationSo);
  }
  async create(createRequest) {
    var _createRequest$id, _createRequest$avatar, _createRequest$avatar2;
    const now = new Date();
    const id = (_createRequest$id = createRequest.id) !== null && _createRequest$id !== void 0 ? _createRequest$id : (0, _uuid.v4)();
    const color = (_createRequest$avatar = (_createRequest$avatar2 = createRequest.avatar) === null || _createRequest$avatar2 === void 0 ? void 0 : _createRequest$avatar2.color) !== null && _createRequest$avatar !== void 0 ? _createRequest$avatar : (0, _color.getRandomColorFromPalette)();
    const attributes = (0, _convert_model.createRequestToRaw)({
      createRequest,
      id,
      user: this.user,
      creationDate: now,
      color
    });
    const created = await this.client.create(_agents.agentTypeName, attributes, {
      id
    });
    return (0, _convert_model.savedObjectToModel)(created);
  }
  async update(agentId, updatedFields) {
    const conversationSo = await this._rawGet({
      agentId
    });
    const updatedAttributes = {
      ...conversationSo.attributes,
      ...(0, _convert_model.updateToAttributes)({
        updatedFields
      })
    };
    await this.client.update(_agents.agentTypeName, conversationSo.id, updatedAttributes);
    return (0, _convert_model.savedObjectToModel)({
      ...conversationSo,
      attributes: updatedAttributes
    });
  }
  async delete(agentId) {
    let conversationSo;
    try {
      conversationSo = await this._rawGet({
        agentId
      });
    } catch (e) {
      if (e instanceof _errors.WorkchatError && e.statusCode === 404) {
        return false;
      } else {
        throw e;
      }
    }
    await this.client.delete(_agents.agentTypeName, conversationSo.id);
    return true;
  }
  async _rawGet({
    agentId
  }) {
    const builder = (0, _so_filters.createBuilder)(_agents.agentTypeName);
    const filter = builder.and(builder.equals('agent_id', agentId), builder.or(builder.equals('user_id', this.user.id), builder.equals('access_control.public', true))).toKQL();
    const {
      saved_objects: results
    } = await this.client.find({
      type: _agents.agentTypeName,
      filter
    });
    if (results.length > 0) {
      return results[0];
    }
    throw new _errors.WorkchatError(`Conversation ${agentId} not found`, 404);
  }
}
exports.AgentClientImpl = AgentClientImpl;