"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAccounts = getAccounts;
exports.getById = getById;
exports.getCases = getCases;
exports.searchDocs = searchDocs;
var _wciCommon = require("@kbn/wci-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Shared base mappings
const baseObjectMappings = {
  id: 'id',
  title: 'title',
  url: 'url',
  ownerEmail: 'owner.email',
  ownerName: 'owner.name',
  createdAt: 'created_at',
  updatedAt: 'updated_at'
};

/**
 * Retrieves Salesforce cases
 *
 * @param esClient - Elasticsearch client
 * @param logger - Logger instance
 * @param indexName - Index name to query
 * @param filters - Search filters
 */
async function searchDocs({
  esClient,
  logger,
  integrationId,
  indexName,
  filters = {}
}) {
  const size = filters.size || 10;
  const {
    objects,
    ...params
  } = filters;
  let query = {};
  if (objects !== null && objects !== void 0 && objects.includes('support_case')) {
    query = buildQuery(params, baseObjectMappings, 'support_case');
  } else if (objects !== null && objects !== void 0 && objects.includes('account')) {
    query = buildQuery(params, baseObjectMappings, 'account');
  } else {
    query = buildQuery(params, baseObjectMappings);
  }
  const searchRequest = {
    index: indexName,
    query,
    size
  };
  const response = await esClient.search(searchRequest);
  const contextFields = [{
    field: 'id',
    type: 'keyword'
  }, {
    field: 'title',
    type: 'keyword'
  }, {
    field: 'content',
    type: 'text'
  }, {
    field: 'url',
    type: 'keyword'
  }];
  const createRef = (0, _wciCommon.contentRefBuilder)({
    sourceType: _wciCommon.ContentRefSourceType.integration,
    sourceId: integrationId
  });
  const contentFragments = response.hits.hits.map(hit => {
    const source = hit._source;
    return {
      type: 'text',
      reference: createRef(`case:${hit._id}`),
      content: contextFields.reduce((content, {
        field
      }) => {
        const fieldPath = field.split('.');
        let value = '';
        if (source) {
          value = fieldPath.length > 1 ? getNestedValue(source, fieldPath) : (source[field] || '').toString();
        }
        content[field] = value;
        return content;
      }, {})
    };
  });
  return contentFragments;
}

/**
 * Retrieves Salesforce docuemnt by id
 *
 * @param esClient - Elasticsearch client
 * @param logger - Logger instance
 * @param indexName - Index name to query
 * @param dataSource - Data source to search through
 * @param id - Id to search by
 */
async function getById({
  esClient,
  logger,
  integrationId,
  indexName,
  dataSource,
  id
}) {
  let query = {};
  if (dataSource === 'support_case') {
    query = buildQuery({
      id
    }, baseObjectMappings, 'support_case');
  } else if (dataSource === 'account') {
    query = buildQuery({
      id
    }, baseObjectMappings, 'account');
  } else {
    query = buildQuery({
      id
    }, baseObjectMappings);
  }
  const searchRequest = {
    index: indexName,
    query
  };
  logger.debug(`Retrieving document from ${indexName} with id: ${JSON.stringify(id)}`);
  const response = await esClient.search(searchRequest);
  const contextFields = [{
    field: 'title',
    type: 'keyword'
  }, {
    field: 'content',
    type: 'text'
  }, {
    field: 'url',
    type: 'keyword'
  }];
  const createRef = (0, _wciCommon.contentRefBuilder)({
    sourceType: _wciCommon.ContentRefSourceType.integration,
    sourceId: integrationId
  });
  const contentFragments = response.hits.hits.map(hit => {
    const source = hit._source;
    return {
      reference: createRef(`case:${hit._id}`),
      content: contextFields.reduce((content, {
        field
      }) => {
        const fieldPath = field.split('.');
        let value = '';
        if (source) {
          value = fieldPath.length > 1 ? getNestedValue(source, fieldPath) : (source[field] || '').toString();
        }
        content[field] = value;
        return content;
      }, {})
    };
  });
  return contentFragments;
}

/**
 * Retrieves Salesforce cases
 *
 * @param esClient - Elasticsearch client
 * @param logger - Logger instance
 * @param indexName - Index name to query
 * @param params - Search parameters including optional sorting configuration
 */
async function getCases({
  esClient,
  logger,
  integrationId,
  indexName,
  params = {}
}) {
  const size = params.size || 10;
  const sort = params.sortField ? [{
    [params.sortField]: {
      order: params.sortOrder
    }
  }] : [];
  const supportCaseMappings = {
    ...baseObjectMappings,
    caseNumber: 'metadata.case_number',
    priority: 'metadata.priority',
    status: 'metadata.status',
    accountId: 'metadata.account_id',
    accountName: 'metadata.account_name'
  };
  const query = buildQuery(params, supportCaseMappings, 'support_case');
  const searchRequest = {
    index: indexName,
    query,
    sort,
    size
  };
  logger.debug(`Retrieving cases from ${indexName} with search request: ${JSON.stringify(searchRequest)}`);
  const response = await esClient.search(searchRequest);

  // Let's keep this here for now
  const contextFields = [{
    field: 'title',
    type: 'keyword'
  }, {
    field: 'description',
    type: 'text'
  }, {
    field: 'url',
    type: 'keyword'
  }, {
    field: 'metadata.case_number',
    type: 'keyword'
  }, {
    field: 'metadata.priority',
    type: 'keyword'
  }, {
    field: 'metadata.status',
    type: 'keyword'
  }, {
    field: 'metadata.account_id',
    type: 'keyword'
  }, {
    field: 'metadata.account_name',
    type: 'keyword'
  }, {
    field: 'owner.email',
    type: 'keyword'
  }, {
    field: 'owner.name',
    type: 'keyword'
  }, {
    field: 'content',
    type: 'text'
  }];
  const createRef = (0, _wciCommon.contentRefBuilder)({
    sourceType: _wciCommon.ContentRefSourceType.integration,
    sourceId: integrationId
  });
  const contentFragments = response.hits.hits.map(hit => {
    const source = hit._source;

    // Format comments if they exist
    let commentsText = '';
    if (source.comments && source.comments.length > 0) {
      const limitedComments = source.comments.slice(0, 10);
      commentsText = '\n\nComments:\n' + limitedComments.map((comment, index) => {
        var _comment$author, _comment$author2;
        return `Comment ${index + 1}:\n` + `Author: ${((_comment$author = comment.author) === null || _comment$author === void 0 ? void 0 : _comment$author.name) || 'Unknown'} (${((_comment$author2 = comment.author) === null || _comment$author2 === void 0 ? void 0 : _comment$author2.email) || 'No email'})\n` + `Created: ${comment.created_at || 'Unknown date'}\n` + `Content: ${comment.content || 'No content'}\n`;
      }).join('\n');
    }
    return {
      reference: createRef(`case:${hit._id}`),
      content: contextFields.reduce((content, {
        field
      }) => {
        const fieldPath = field.split('.');

        // Use the helper function for both nested and non-nested fields
        const value = fieldPath.length > 1 ? getNestedValue(source, fieldPath) : (source[field] || '').toString();
        content[field] = value;
        return content;
      }, {
        commentsText
      })
    };
  });
  return contentFragments;
}

/**
 * Retrieves Salesforce accounts
 *
 * @param esClient - Elasticsearch client
 * @param logger - Logger instance
 * @param indexName - Index name to query
 * @param params - Search parameters including optional sorting configuration
 */
async function getAccounts({
  esClient,
  logger,
  integrationId,
  indexName,
  params = {}
}) {
  const size = params.size || 10;
  const sort = params.sortField ? [{
    [params.sortField]: {
      order: params.sortOrder
    }
  }] : [];
  const accountMappings = {
    ...baseObjectMappings,
    recordTypeId: 'metadata.record_type_id',
    isPartner: 'metadata.is_partner',
    isCustomerPortal: 'metadata.is_customer_portal'
  };
  const query = buildQuery(params, accountMappings, 'account');
  const searchRequest = {
    index: indexName,
    query,
    sort,
    size
  };
  logger.debug(`Retrieving accounts from ${indexName} with search request: ${JSON.stringify(searchRequest)}`);
  const response = await esClient.search(searchRequest);

  // Define fields to include in the response
  const contextFields = [{
    field: 'id',
    type: 'keyword'
  }, {
    field: 'title',
    type: 'keyword'
  }, {
    field: 'url',
    type: 'keyword'
  }, {
    field: 'owner.email',
    type: 'keyword'
  }, {
    field: 'owner.name',
    type: 'keyword'
  }, {
    field: 'created_at',
    type: 'date'
  }, {
    field: 'updated_at',
    type: 'date'
  }];
  const createRef = (0, _wciCommon.contentRefBuilder)({
    sourceType: _wciCommon.ContentRefSourceType.integration,
    sourceId: integrationId
  });
  const contentFragments = response.hits.hits.map(hit => {
    const source = hit._source;

    // Format contacts if they exist
    let contactsText = '';
    if (source.contacts && source.contacts.length > 0) {
      const limitedContacts = source.contacts.slice(0, 10);
      contactsText = '\n\nContacts:\n' + limitedContacts.map((contact, index) => {
        return `Contact ${index + 1}:\n` + `Name: ${contact.name || 'Unknown'}\n` + `Email: ${contact.email || 'No email'}\n` + `Phone: ${contact.phone || 'No phone'}\n` + `Title: ${contact.title || 'No title'}\n` + `Department: ${contact.department || 'No department'}\n`;
      }).join('\n');
    }
    return {
      reference: createRef(`account:${hit._id}`),
      content: contextFields.reduce((content, {
        field
      }) => {
        const fieldPath = field.split('.');
        const value = fieldPath.length > 1 ? getNestedValue(source, fieldPath) : (source[field] || '').toString();
        content[field] = value;
        return content;
      }, {
        contactsText
      })
    };
  });
  return contentFragments;
}

// Helper function to safely get nested values
function getNestedValue(obj, path) {
  var _path$reduce;
  return ((_path$reduce = path.reduce((prev, curr) => {
    return prev && typeof prev === 'object' && curr in prev ? prev[curr] : '';
  }, obj)) === null || _path$reduce === void 0 ? void 0 : _path$reduce.toString()) || '';
}
function addTermsClause(mustClauses, field, value, mappings) {
  if (mappings && mappings[field]) {
    if (Array.isArray(value)) {
      mustClauses.push({
        terms: {
          [mappings[field]]: value
        }
      });
    } else {
      mustClauses.push({
        term: {
          [mappings[field]]: value
        }
      });
    }
  }
}
function addDateRangeClause(mustClauses, field, createdAfter, createdBefore) {
  if (createdAfter || createdBefore) {
    const range = {
      range: {
        [field]: {}
      }
    };
    if (createdAfter) range.range[field].gte = createdAfter;
    if (createdBefore) range.range[field].lte = createdBefore;
    mustClauses.push(range);
  }
}
function addCommentFilters(mustClauses, params) {
  // Add comment-related queries
  if (params.commentAuthorEmail || params.commentCreatedAfter || params.commentCreatedBefore) {
    const nestedQuery = {
      nested: {
        path: 'comments',
        query: {
          bool: {
            must: []
          }
        }
      }
    };

    // Add comment author filter
    if (params.commentAuthorEmail && params.commentAuthorEmail.length > 0) {
      nestedQuery.nested.query.bool.must.push({
        terms: {
          'comments.author.email': params.commentAuthorEmail
        }
      });
    }

    // Add comment date range filters
    if (params.commentCreatedAfter || params.commentCreatedBefore) {
      const commentDateRange = {
        range: {
          'comments.created_at': {}
        }
      };
      if (params.commentCreatedAfter) commentDateRange.range['comments.created_at'].gte = params.commentCreatedAfter;
      if (params.commentCreatedBefore) commentDateRange.range['comments.created_at'].lte = params.commentCreatedBefore;
      nestedQuery.nested.query.bool.must.push(commentDateRange);
    }
    mustClauses.push(nestedQuery);
  }
}
function addSemanticQuery(mustClauses, semanticQuery) {
  if (semanticQuery) {
    mustClauses.push({
      semantic: {
        field: 'content_semantic',
        query: semanticQuery,
        boost: 2.0
      }
    });
  }
}
function buildQuery(params, mappings, objectType) {
  const mustClauses = [];
  if (objectType) {
    mustClauses.push({
      term: {
        object_type: objectType
      }
    });
  }
  Object.entries(params).forEach(([field, value]) => {
    if (value) {
      if (field === 'query') {
        addSemanticQuery(mustClauses, value);
      } else if (field === 'createdAfter' || field === 'createdBefore') {
        if (!mustClauses.some(clause => clause.range && clause.range.created_at !== undefined)) {
          addDateRangeClause(mustClauses, 'created_at', params.createdAfter, params.createdBefore);
        }
      } else if (field === 'updatedAfter' || field === 'updatedBefore') {
        if (!mustClauses.some(clause => clause.range && clause.range.updated_at !== undefined)) {
          addDateRangeClause(mustClauses, 'updated_at', params.updatedAfter, params.updatedBefore);
        }
      } else if (field === 'commentAuthorEmail' || field === 'commentCreatedAfter' || field === 'commentCreatedBefore') {
        addCommentFilters(mustClauses, {
          commentAuthorEmail: params.commentAuthorEmail,
          commentCreatedAfter: params.commentCreatedAfter,
          commentCreatedBefore: params.commentCreatedBefore
        });
      } else {
        addTermsClause(mustClauses, field, value, mappings);
      }
    }
  });
  return {
    bool: {
      must: mustClauses
    }
  };
}