"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSchemaGraph = void 0;
var _zod = require("@kbn/zod");
var _langgraph = require("@langchain/langgraph");
var _wcIntegrationUtils = require("@kbn/wc-integration-utils");
var _utils = require("../utils");
var _prompts = require("./prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createSchemaGraph = async ({
  chatModel,
  esClient
}) => {
  const StateAnnotation = _langgraph.Annotation.Root({
    indexName: (0, _langgraph.Annotation)(),
    indexInfo: _langgraph.Annotation,
    sampleDocuments: _langgraph.Annotation,
    fieldTopValues: _langgraph.Annotation,
    // temporary
    filterFields: _langgraph.Annotation,
    queryFields: _langgraph.Annotation,
    contentFields: _langgraph.Annotation,
    description: (0, _langgraph.Annotation)(),
    // output
    generatedDefinition: (0, _langgraph.Annotation)({
      reducer: (a, b) => {
        var _a$filterFields, _b$filterFields;
        return {
          ...a,
          ...b,
          filterFields: [...((_a$filterFields = a.filterFields) !== null && _a$filterFields !== void 0 ? _a$filterFields : []), ...((_b$filterFields = b.filterFields) !== null && _b$filterFields !== void 0 ? _b$filterFields : [])]
        };
      },
      default: () => ({})
    })
  });
  const gatherIndexInfo = async state => {
    const indexInfo = await (0, _utils.getIndexInformation)({
      indexName: state.indexName,
      esClient
    });
    const sampleDocuments = await (0, _utils.getSampleDocuments)({
      indexName: state.indexName,
      esClient,
      maxSamples: 3
    });
    const leafFields = (0, _utils.getLeafFields)({
      mappings: indexInfo.mappings
    });
    const fieldTopValues = await (0, _wcIntegrationUtils.getFieldsTopValues)({
      indexName: state.indexName,
      esClient,
      maxSize: 20,
      fieldNames: leafFields.filter(field => field.type === 'keyword').map(field => field.path)
    });
    return {
      indexInfo,
      sampleDocuments,
      fieldTopValues,
      generatedDefinition: {
        index: state.indexName
      }
    };
  };
  const pickFilterFields = async state => {
    const structuredModel = chatModel.withStructuredOutput(_zod.z.object({
      fields: _zod.z.array(_zod.z.string()).describe('The list of fields to use as filter fields')
    }));
    const response = await structuredModel.invoke((0, _prompts.pickFilterFieldsPrompt)({
      indexName: state.indexName,
      indexInfo: state.indexInfo,
      fieldTopValues: state.fieldTopValues,
      sampleDocuments: state.sampleDocuments
    }));
    return {
      filterFields: response.fields
    };
  };
  const dispatchFilterFields = async state => {
    return state.filterFields.map(filterField => {
      return new _langgraph.Send('build_filter_field', {
        ...state,
        fieldName: filterField
      });
    });
  };
  const buildFilterField = async state => {
    var _response$description, _response$asEnum;
    const structuredModel = chatModel.withStructuredOutput(_zod.z.object({
      description: _zod.z.string().describe('the description for the filter. Please refer to the instruction'),
      asEnum: _zod.z.boolean().describe('The asEnum value for the filter. Please refer to the instructions')
    }));
    const fieldName = state.fieldName;
    const fieldType = (0, _wcIntegrationUtils.getFieldTypeByPath)({
      fieldPath: state.fieldName,
      mappings: state.indexInfo.mappings
    });
    const fieldTopValues = state.fieldTopValues[fieldName];
    const response = await structuredModel.invoke((0, _prompts.generateFilterPrompt)({
      indexName: state.indexName,
      fieldName,
      fieldType,
      fieldTopValues,
      sampleDocuments: state.sampleDocuments
    }));
    const filterField = {
      field: fieldName,
      type: fieldType,
      description: (_response$description = response.description) !== null && _response$description !== void 0 ? _response$description : '',
      asEnum: (_response$asEnum = response.asEnum) !== null && _response$asEnum !== void 0 ? _response$asEnum : false
    };
    return {
      generatedDefinition: {
        filterFields: [filterField]
      }
    };
  };
  const pickQueryFields = async state => {
    const structuredModel = chatModel.withStructuredOutput(_zod.z.object({
      fields: _zod.z.array(_zod.z.string()).describe('The list of fields to use as fulltext fields')
    }));
    const response = await structuredModel.invoke((0, _prompts.pickQueryFieldsPrompt)({
      indexName: state.indexName,
      indexInfo: state.indexInfo,
      sampleDocuments: state.sampleDocuments
    }));
    return {
      queryFields: response.fields
    };
  };
  const buildQueryFields = async state => {
    const {
      indexInfo: {
        mappings
      }
    } = state;
    const queryFields = state.queryFields.map(field => {
      return {
        field,
        type: (0, _wcIntegrationUtils.getFieldTypeByPath)({
          fieldPath: field,
          mappings
        })
      };
    });
    return {
      generatedDefinition: {
        queryFields
      }
    };
  };

  ///////
  const pickContentFields = async state => {
    const structuredModel = chatModel.withStructuredOutput(_zod.z.object({
      fields: _zod.z.array(_zod.z.string()).describe('The list of fields to use as content fields')
    }));
    const response = await structuredModel.invoke((0, _prompts.pickContentFieldsPrompt)({
      indexName: state.indexName,
      indexInfo: state.indexInfo,
      sampleDocuments: state.sampleDocuments
    }));
    return {
      contentFields: response.fields
    };
  };
  const buildContentFields = async state => {
    const {
      indexInfo: {
        mappings
      }
    } = state;
    const contentFields = state.contentFields.map(field => {
      return {
        field,
        type: (0, _wcIntegrationUtils.getFieldTypeByPath)({
          fieldPath: field,
          mappings
        })
      };
    });
    return {
      generatedDefinition: {
        contentFields
      }
    };
  };
  ///////

  const generateDescription = async state => {
    const structuredModel = chatModel.withStructuredOutput(_zod.z.object({
      description: _zod.z.string().describe('The description for the tool')
    }));
    const response = await structuredModel.invoke((0, _prompts.generateDescriptionPrompt)({
      sourceDefinition: state.generatedDefinition,
      indexName: state.indexName,
      indexInfo: state.indexInfo,
      sampleDocuments: state.sampleDocuments
    }));
    return {
      generatedDefinition: {
        description: response.description
      }
    };
  };
  const graph = new _langgraph.StateGraph(StateAnnotation)
  // nodes
  .addNode('gather_index_info', gatherIndexInfo).addNode('pick_filter_fields', pickFilterFields).addNode('pick_query_fields', pickQueryFields).addNode('build_query_fields', buildQueryFields).addNode('build_filter_field', buildFilterField).addNode('generate_description', generateDescription).addNode('pick_content_fields', pickContentFields).addNode('build_content_fields', buildContentFields)

  // transitions
  .addEdge('__start__', 'gather_index_info').addEdge('gather_index_info', 'pick_filter_fields').addEdge('gather_index_info', 'pick_query_fields').addEdge('gather_index_info', 'pick_content_fields').addEdge('pick_query_fields', 'build_query_fields').addEdge('build_query_fields', 'generate_description').addEdge('pick_content_fields', 'build_content_fields').addEdge('build_content_fields', 'generate_description').addEdge('generate_description', '__end__').addConditionalEdges('pick_filter_fields', dispatchFilterFields, {
    build_filter_field: 'build_filter_field'
  }).addEdge('build_filter_field', 'generate_description')
  // done
  .compile();
  return graph;
};
exports.createSchemaGraph = createSchemaGraph;