"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ServiceSettings = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _markdownIt = _interopRequireDefault(require("markdown-it"));
var _common = require("@kbn/maps-ems-plugin/common");
var _service_settings_types = require("./service_settings_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * This class provides access to the EMS-layers and the kibana.yml configured layers through a single interface.
 */
class ServiceSettings {
  constructor(mapConfig, tilemapsConfig, emsClient) {
    (0, _defineProperty2.default)(this, "_mapConfig", void 0);
    (0, _defineProperty2.default)(this, "_tilemapsConfig", void 0);
    (0, _defineProperty2.default)(this, "_hasTmsConfigured", void 0);
    (0, _defineProperty2.default)(this, "_emsClient", void 0);
    (0, _defineProperty2.default)(this, "tmsOptionsFromConfig", void 0);
    (0, _defineProperty2.default)(this, "_backfillSettings", fileLayer => {
      // Older version of Kibana stored EMS state in the URL-params
      // Creates object literal with required parameters as key-value pairs
      const format = fileLayer.getDefaultFormatType();
      const meta = fileLayer.getDefaultFormatMeta();
      return {
        name: fileLayer.getDisplayName(),
        origin: _service_settings_types.ORIGIN_LEGACY.EMS,
        id: fileLayer.getId(),
        created_at: fileLayer.getCreatedAt(),
        attribution: getAttributionString(fileLayer),
        fields: fileLayer.getFieldsInLanguage(),
        format,
        // legacy: format and meta are split up
        meta // legacy, format and meta are split up
      };
    });
    this._mapConfig = mapConfig;
    this._tilemapsConfig = tilemapsConfig;
    this._hasTmsConfigured = typeof tilemapsConfig.url === 'string' && tilemapsConfig.url !== '';
    this._emsClient = emsClient;
    const markdownIt = new _markdownIt.default({
      html: false,
      linkify: true
    });

    // TMS Options
    this.tmsOptionsFromConfig = _lodash.default.assign({}, this._tilemapsConfig.options, {
      attribution: _lodash.default.escape(markdownIt.render(this._tilemapsConfig.options.attribution || '')),
      url: this._tilemapsConfig.url
    });
  }
  getMapConfig() {
    return this._mapConfig;
  }
  getTileMapConfig() {
    return this._tilemapsConfig;
  }
  async getFileLayers() {
    if (!this._mapConfig.includeElasticMapsService) {
      return [];
    }
    const fileLayers = await this._emsClient.getFileLayers();
    return fileLayers.map(this._backfillSettings);
  }

  /**
   * Returns all the services published by EMS (if configures)
   * It also includes the service configured in tilemap (override)
   */
  async getTMSServices() {
    let allServices = [];
    if (this._hasTmsConfigured) {
      // use tilemap.* settings from yml
      const tmsService = {
        ..._lodash.default.cloneDeep(this.tmsOptionsFromConfig),
        id: _service_settings_types.TMS_IN_YML_ID,
        origin: _service_settings_types.ORIGIN_LEGACY.KIBANA_YML
      };
      allServices.push(tmsService);
    }
    if (this._mapConfig.includeElasticMapsService && !this._mapConfig.emsUrl) {
      const servicesFromManifest = await this._emsClient.getTMSServices();
      const strippedServiceFromManifest = await Promise.all(servicesFromManifest.filter(tmsService => tmsService.getId() === this._mapConfig.emsTileLayerId.bright).map(async tmsService => {
        // shim for compatibility
        return {
          origin: _service_settings_types.ORIGIN_LEGACY.EMS,
          id: tmsService.getId(),
          minZoom: await tmsService.getMinZoom(),
          maxZoom: await tmsService.getMaxZoom(),
          attribution: getAttributionString(tmsService)
        };
      }));
      allServices = allServices.concat(strippedServiceFromManifest);
    }
    return allServices;
  }
  async getTmsService(id) {
    return this._emsClient.findTMSServiceById(id);
  }
  async getDefaultTmsLayer(isDarkMode, themeName) {
    const {
      dark,
      desaturated
    } = this._mapConfig.emsTileLayerId;
    if (hasUserConfiguredTmsLayer(this._mapConfig)) {
      return _service_settings_types.TMS_IN_YML_ID;
    }

    // To be removed once Borealis is the only theme available
    if (themeName === 'borealis') {
      return isDarkMode ? _common.EMS_DARKMAP_BOREALIS_ID : _common.EMS_ROADMAP_BOREALIS_DESATURATED_ID;
    }
    return isDarkMode ? dark : desaturated;
  }
  async _getAttributesForEMSTMSLayer(isDesaturated, isDarkMode) {
    const tmsServices = await this._emsClient.getTMSServices();
    const emsTileLayerId = this._mapConfig.emsTileLayerId;
    let serviceId;
    if (isDarkMode) {
      serviceId = emsTileLayerId.dark;
    } else {
      if (isDesaturated) {
        serviceId = emsTileLayerId.desaturated;
      } else {
        serviceId = emsTileLayerId.bright;
      }
    }
    const tmsService = tmsServices.find(service => {
      return service.getId() === serviceId;
    });
    return {
      url: await tmsService.getUrlTemplate(),
      minZoom: await tmsService.getMinZoom(),
      maxZoom: await tmsService.getMaxZoom(),
      attribution: getAttributionString(tmsService),
      origin: _service_settings_types.ORIGIN_LEGACY.EMS
    };
  }
  async getAttributesForTMSLayer(tmsServiceConfig, isDesaturated, isDarkMode) {
    if (tmsServiceConfig.origin === _service_settings_types.ORIGIN_LEGACY.EMS) {
      return this._getAttributesForEMSTMSLayer(isDesaturated, isDarkMode);
    } else if (tmsServiceConfig.origin === _service_settings_types.ORIGIN_LEGACY.KIBANA_YML) {
      const attrs = _lodash.default.pick(this._tilemapsConfig, ['url', 'minzoom', 'maxzoom', 'attribution']);
      return {
        ...attrs,
        ...{
          origin: _service_settings_types.ORIGIN_LEGACY.KIBANA_YML
        }
      };
    } else {
      // this is an older config. need to resolve this dynamically.
      if (tmsServiceConfig.id === _service_settings_types.TMS_IN_YML_ID) {
        const attrs = _lodash.default.pick(this._tilemapsConfig, ['url', 'minzoom', 'maxzoom', 'attribution']);
        return {
          ...attrs,
          ...{
            origin: _service_settings_types.ORIGIN_LEGACY.KIBANA_YML
          }
        };
      } else {
        // assume ems
        return this._getAttributesForEMSTMSLayer(isDesaturated, isDarkMode);
      }
    }
  }
  async _getFileUrlFromEMS(fileLayerConfig) {
    const fileLayers = await this._emsClient.getFileLayers();
    const layer = fileLayers.find(fileLayer => {
      const hasIdByName = fileLayer.hasId(fileLayerConfig.name); // legacy
      const hasIdById = fileLayer.hasId(fileLayerConfig.id);
      return hasIdByName || hasIdById;
    });
    if (layer) {
      return layer.getDefaultFormatUrl();
    } else {
      throw new Error(`File  ${fileLayerConfig.name} not recognized`);
    }
  }
  async getUrlForRegionLayer(fileLayerConfig) {
    let url;
    if (fileLayerConfig.origin === _service_settings_types.ORIGIN_LEGACY.EMS) {
      url = this._getFileUrlFromEMS(fileLayerConfig);
    } else if (fileLayerConfig.layerId && fileLayerConfig.layerId.startsWith(`${_service_settings_types.ORIGIN_LEGACY.EMS}.`)) {
      // fallback for older saved objects
      url = this._getFileUrlFromEMS(fileLayerConfig);
    } else if (fileLayerConfig.layerId && fileLayerConfig.layerId.startsWith(`${_service_settings_types.ORIGIN_LEGACY.KIBANA_YML}.`)) {
      // fallback for older saved objects
      url = fileLayerConfig.url;
    } else {
      // generic fallback
      url = fileLayerConfig.url;
    }
    return url;
  }
  getAttributionsFromTMSServce(tmsService) {
    return getAttributionString(tmsService);
  }
}
exports.ServiceSettings = ServiceSettings;
function getAttributionString(emsService) {
  const attributions = emsService.getAttributions();
  const attributionSnippets = attributions.map(attribution => {
    const anchorTag = document.createElement('a');
    anchorTag.setAttribute('rel', 'noreferrer noopener');
    if (attribution.url.startsWith('http://') || attribution.url.startsWith('https://')) {
      anchorTag.setAttribute('href', attribution.url);
    }
    anchorTag.textContent = attribution.label;
    return anchorTag.outerHTML;
  });
  return attributionSnippets.join(' | '); // !!!this is the current convention used in Kibana
}
function hasUserConfiguredTmsLayer(config) {
  var _config$tilemap;
  return Boolean((_config$tilemap = config.tilemap) === null || _config$tilemap === void 0 ? void 0 : _config$tilemap.url);
}