"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToLens = void 0;
var _uuid = require("uuid");
var _common = require("@kbn/data-plugin/common");
var _convert_to_lens = require("@kbn/visualizations-plugin/common/convert_to_lens");
var _enums = require("../../../common/enums");
var _services = require("../../services");
var _datasource = require("../lib/datasource");
var _series = require("../lib/series");
var _xy = require("../lib/configurations/xy");
var _metrics = require("../lib/metrics");
var _convert = require("../lib/convert");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const excludeMetaFromLayers = layers => {
  const newLayers = {};
  Object.entries(layers).forEach(([layerId, layer]) => {
    const columns = layer.columns.map(_convert.excludeMetaFromColumn);
    newLayers[layerId] = {
      ...layer,
      columns
    };
  });
  return newLayers;
};
const invalidModelError = () => new Error('Invalid model');
const convertToLens = async ({
  params: model
}, timeRange) => {
  const dataViews = (0, _services.getDataViewsStart)();
  try {
    const extendedLayers = {};
    const visibleSeries = model.series.filter(series => !series.hidden);
    const seriesNum = visibleSeries.length;

    // handle multiple layers/series
    for (const [layerIdx, series] of model.series.entries()) {
      if (series.hidden) {
        continue;
      }

      // not valid time shift
      if (series.offset_time && (0, _common.parseTimeShift)(series.offset_time) === 'invalid') {
        throw invalidModelError();
      }
      if (!(0, _metrics.isValidMetrics)(series.metrics, _enums.PANEL_TYPES.TOP_N, series.time_range_mode)) {
        throw invalidModelError();
      }
      const datasourceInfo = await (0, _datasource.extractOrGenerateDatasourceInfo)(model.index_pattern, model.time_field, Boolean(series.override_index_pattern), series.series_index_pattern, series.series_time_field, dataViews);
      if (!datasourceInfo) {
        throw invalidModelError();
      }
      const {
        indexPatternId,
        indexPattern
      } = datasourceInfo;
      const reducedTimeRange = (0, _metrics.getReducedTimeRange)(model, series, timeRange);

      // handle multiple metrics
      const metricsColumns = (0, _series.getMetricsColumns)(series, indexPattern, seriesNum, {
        reducedTimeRange
      });
      if (!metricsColumns) {
        throw invalidModelError();
      }
      const bucketsColumns = (0, _series.getBucketsColumns)(model, series, metricsColumns, indexPattern, false);
      if (bucketsColumns === null) {
        throw invalidModelError();
      }
      const layerId = (0, _uuid.v4)();
      extendedLayers[layerIdx] = {
        ignoreGlobalFilters: Boolean(model.ignore_global_filter ||
        // eslint-disable-next-line @typescript-eslint/naming-convention
        visibleSeries.some(({
          ignore_global_filter
        }) => ignore_global_filter)),
        indexPatternId,
        layerId,
        columns: [...metricsColumns, ...bucketsColumns],
        columnOrder: []
      };
    }
    const configLayers = await (0, _xy.getLayers)(extendedLayers, model, dataViews, true);
    if (configLayers === null) {
      throw invalidModelError();
    }
    const layers = Object.values(excludeMetaFromLayers(extendedLayers));
    return {
      type: 'lnsXY',
      layers,
      configuration: (0, _xy.getConfigurationForTopN)(model, configLayers),
      indexPatternIds: (0, _convert_to_lens.getIndexPatternIds)(layers)
    };
  } catch (e) {
    return null;
  }
};
exports.convertToLens = convertToLens;