"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerUpgradeStatusRoute = registerUpgradeStatusRoute;
var _i18n = require("@kbn/i18n");
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../common/constants");
var _es_deprecations_status = require("../lib/es_deprecations_status");
var _es_version_precheck = require("../lib/es_version_precheck");
var _kibana_status = require("../lib/kibana_status");
var _es_system_indices_migration = require("../lib/es_system_indices_migration");
var _upgrade_type = require("../lib/upgrade_type");
var _es_deprecation_logging_apis = require("../lib/es_deprecation_logging_apis");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Note that this route is primarily intended for consumption by Cloud.
 */
function registerUpgradeStatusRoute({
  config: {
    featureSet,
    dataSourceExclusions
  },
  router,
  lib: {
    handleEsError
  },
  current,
  defaultTarget,
  log
}) {
  router.get({
    path: `${_constants.API_BASE_PATH}/status`,
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    options: {
      access: 'public',
      summary: `Get upgrade readiness status`
    },
    validate: {
      query: _configSchema.schema.object({
        targetVersion: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, (0, _es_version_precheck.versionCheckHandlerWrapper)(async ({
    core
  }, request, response) => {
    var _request$query;
    const targetVersion = ((_request$query = request.query) === null || _request$query === void 0 ? void 0 : _request$query.targetVersion) || `${defaultTarget}`;
    const upgradeType = (0, _upgrade_type.getUpgradeType)({
      current,
      target: targetVersion
    });
    if (!upgradeType) return response.forbidden();
    try {
      const {
        elasticsearch: {
          client: esClient
        },
        deprecations: {
          client: deprecationsClient
        }
      } = await core;
      // Fetch ES upgrade status
      const {
        totalCriticalDeprecations,
        // critical deprecations
        totalCriticalHealthIssues // critical health issues
      } = await (0, _es_deprecations_status.getESUpgradeStatus)(esClient.asCurrentUser, {
        featureSet,
        dataSourceExclusions
      });

      // Fetch recent ES deprecation logs (last 24 hours)
      const recentEsDeprecationLogs = await (0, _es_deprecation_logging_apis.getRecentEsDeprecationLogs)(esClient, _constants.RECENT_DURATION_MS);
      const getSystemIndicesMigrationStatus = async () => {
        /**
         * Skip system indices migration status check if `featureSet.migrateSystemIndices`
         * is set to `false`. This flag is enabled from configs for major version stack ugprades.
         * returns `migration_status: 'NO_MIGRATION_NEEDED'` to indicate no migration needed.
         */
        if (!featureSet.migrateSystemIndices) {
          return {
            migration_status: 'NO_MIGRATION_NEEDED',
            features: []
          };
        }

        // Fetch system indices migration status from ES
        return await (0, _es_system_indices_migration.getESSystemIndicesMigrationStatus)(esClient.asCurrentUser);
      };
      const {
        migration_status: systemIndicesMigrationStatus,
        features
      } = await getSystemIndicesMigrationStatus();
      const notMigratedSystemIndices = features.filter(feature => feature.migration_status !== 'NO_MIGRATION_NEEDED').length;

      // Fetch Kibana upgrade status
      const {
        totalCriticalDeprecations: kibanaTotalCriticalDeps,
        apiDeprecations: kibanaApiDeprecations
      } = await (0, _kibana_status.getKibanaUpgradeStatus)(deprecationsClient);

      // non-major upgrades blocked only for health issues (status !== green)
      let upgradeTypeBasedReadyForUpgrade;
      if (upgradeType === 'major') {
        upgradeTypeBasedReadyForUpgrade = totalCriticalHealthIssues === 0 && totalCriticalDeprecations === 0 && kibanaTotalCriticalDeps === 0 && systemIndicesMigrationStatus === 'NO_MIGRATION_NEEDED';
      } else {
        upgradeTypeBasedReadyForUpgrade = totalCriticalHealthIssues === 0;
      }
      const readyForUpgrade = upgradeType && upgradeTypeBasedReadyForUpgrade;
      const getStatusMessage = () => {
        if (readyForUpgrade) {
          return _i18n.i18n.translate('xpack.upgradeAssistant.status.allDeprecationsResolvedMessage', {
            defaultMessage: 'All deprecation warnings have been resolved.'
          });
        }
        const upgradeIssues = [];
        let esTotalCriticalDeps = totalCriticalHealthIssues;
        if (upgradeType === 'major') {
          esTotalCriticalDeps += totalCriticalDeprecations;
        }
        if (upgradeType === 'major' && notMigratedSystemIndices) {
          upgradeIssues.push(_i18n.i18n.translate('xpack.upgradeAssistant.status.systemIndicesMessage', {
            defaultMessage: '{notMigratedSystemIndices} unmigrated system {notMigratedSystemIndices, plural, one {index} other {indices}}',
            values: {
              notMigratedSystemIndices
            }
          }));
        }
        // can be improved by showing health indicator issues separately
        if (esTotalCriticalDeps) {
          upgradeIssues.push(_i18n.i18n.translate('xpack.upgradeAssistant.status.esTotalCriticalDepsMessage', {
            defaultMessage: '{esTotalCriticalDeps} Elasticsearch deprecation {esTotalCriticalDeps, plural, one {issue} other {issues}}',
            values: {
              esTotalCriticalDeps
            }
          }));
        }
        if (upgradeType === 'major' && kibanaTotalCriticalDeps) {
          upgradeIssues.push(_i18n.i18n.translate('xpack.upgradeAssistant.status.kibanaTotalCriticalDepsMessage', {
            defaultMessage: '{kibanaTotalCriticalDeps} Kibana deprecation {kibanaTotalCriticalDeps, plural, one {issue} other {issues}}',
            values: {
              kibanaTotalCriticalDeps
            }
          }));
        }
        return _i18n.i18n.translate('xpack.upgradeAssistant.status.deprecationsUnresolvedMessage', {
          defaultMessage: 'The following issues must be resolved before upgrading: {upgradeIssues}.',
          values: {
            upgradeIssues: upgradeIssues.join(', ')
          }
        });
      };
      return response.ok({
        body: {
          readyForUpgrade,
          details: getStatusMessage(),
          recentEsDeprecationLogs: {
            count: recentEsDeprecationLogs.count,
            logs: recentEsDeprecationLogs.logs
          },
          kibanaApiDeprecations
        }
      });
    } catch (error) {
      log.error(error);
      return handleEsError({
        error,
        response
      });
    }
  }));
}