"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UnwiredStream = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = _interopRequireWildcard(require("lodash"));
var _esErrors = require("@kbn/es-errors");
var _status_error = require("../../errors/status_error");
var _generate_ingest_pipeline = require("../../ingest_pipelines/generate_ingest_pipeline");
var _name = require("../../ingest_pipelines/name");
var _stream_crud = require("../../stream_crud");
var _stream_active_record = require("../stream_active_record/stream_active_record");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class UnwiredStream extends _stream_active_record.StreamActiveRecord {
  constructor(definition, dependencies) {
    super(definition, dependencies);
    (0, _defineProperty2.default)(this, "_processingChanged", false);
    (0, _defineProperty2.default)(this, "_lifecycleChanged", false);
  }
  clone() {
    return new UnwiredStream((0, _lodash.cloneDeep)(this._definition), this.dependencies);
  }
  toPrintable() {
    return {
      ...super.toPrintable(),
      processingChanged: this._processingChanged,
      lifecycleChanged: this._lifecycleChanged
    };
  }
  async doHandleUpsertChange(definition, desiredState, startingState) {
    var _startingState$get;
    if (definition.name !== this._definition.name) {
      return {
        cascadingChanges: [],
        changeStatus: this.changeStatus
      };
    }
    if (!_streamsSchema.Streams.UnwiredStream.Definition.is(definition)) {
      throw new _status_error.StatusError('Cannot change stream types', 400);
    }
    this._definition = definition;
    const startingStateStreamDefinition = (_startingState$get = startingState.get(this._definition.name)) === null || _startingState$get === void 0 ? void 0 : _startingState$get.definition;
    if (startingStateStreamDefinition && !_streamsSchema.Streams.UnwiredStream.Definition.is(startingStateStreamDefinition)) {
      throw new _status_error.StatusError('Unexpected starting state stream type', 400);
    }
    this._processingChanged = !startingStateStreamDefinition || !_lodash.default.isEqual(this._definition.ingest.processing, startingStateStreamDefinition.ingest.processing);
    this._lifecycleChanged = !startingStateStreamDefinition || !_lodash.default.isEqual(this._definition.ingest.lifecycle, startingStateStreamDefinition.ingest.lifecycle);
    return {
      cascadingChanges: [],
      changeStatus: 'upserted'
    };
  }
  async doHandleDeleteChange(target, desiredState, startingState) {
    if (target !== this._definition.name) {
      return {
        cascadingChanges: [],
        changeStatus: this.changeStatus
      };
    }
    return {
      cascadingChanges: [],
      changeStatus: 'deleted'
    };
  }
  async doValidateUpsertion(desiredState, startingState) {
    if (this.dependencies.isServerless && (0, _streamsSchema.isIlmLifecycle)(this.getLifecycle())) {
      return {
        isValid: false,
        errors: [new Error('Using ILM is not supported in Serverless')]
      };
    }

    // Check for conflicts
    if (this._lifecycleChanged || this._processingChanged) {
      try {
        const dataStreamResult = await this.dependencies.scopedClusterClient.asCurrentUser.indices.getDataStream({
          name: this._definition.name
        });
        if (dataStreamResult.data_streams.length === 0) {
          // There is an index but no data stream
          return {
            isValid: false,
            errors: [new Error(`Cannot create Unwired stream ${this.definition.name} due to existing index`)]
          };
        }
      } catch (error) {
        if ((0, _esErrors.isNotFoundError)(error)) {
          return {
            isValid: false,
            errors: [new Error(`Cannot create Unwired stream ${this.definition.name} due to missing backing Data Stream`)]
          };
        }
        throw error;
      }
    }
    return {
      isValid: true,
      errors: []
    };
  }
  async doValidateDeletion(desiredState, startingState) {
    return {
      isValid: true,
      errors: []
    };
  }

  // The actions append_processor_to_ingest_pipeline and delete_processor_from_ingest_pipeline are unique to UnwiredStreams
  // Because there is no guarantee that there is a dedicated index template and ingest pipeline for UnwiredStreams
  // These actions are merged across UnwiredStream instances as part of ExecutionPlan.plan()
  // This is to enable us to clean up any pipeline Streams creates when it is no longer needed
  async doDetermineCreateActions() {
    const actions = [];
    if (this._definition.ingest.processing.length > 0) {
      actions.push(...(await this.createUpsertPipelineActions()));
    }
    if (!(0, _streamsSchema.isInheritLifecycle)(this.getLifecycle())) {
      actions.push({
        type: 'update_lifecycle',
        request: {
          name: this._definition.name,
          lifecycle: this.getLifecycle()
        }
      });
    }
    actions.push({
      type: 'upsert_dot_streams_document',
      request: this._definition
    });
    return actions;
  }
  hasChangedLifecycle() {
    return this._lifecycleChanged;
  }
  getLifecycle() {
    return this._definition.ingest.lifecycle;
  }
  async doDetermineUpdateActions(desiredState, startingState, startingStateStream) {
    const actions = [];
    if (this._processingChanged && this._definition.ingest.processing.length > 0) {
      actions.push(...(await this.createUpsertPipelineActions()));
    }
    if (this._processingChanged && this._definition.ingest.processing.length === 0) {
      const streamManagedPipelineName = (0, _name.getProcessingPipelineName)(this._definition.name);
      actions.push({
        type: 'delete_ingest_pipeline',
        request: {
          name: streamManagedPipelineName
        }
      });
      const pipelineTargets = await this.getPipelineTargets();
      if (!pipelineTargets) {
        throw new _status_error.StatusError('Could not find pipeline targets', 500);
      }
      const {
        pipeline,
        template
      } = pipelineTargets;
      actions.push({
        type: 'delete_processor_from_ingest_pipeline',
        pipeline,
        template,
        dataStream: this._definition.name,
        referencePipeline: streamManagedPipelineName
      });
    }
    if (this._lifecycleChanged) {
      actions.push({
        type: 'update_lifecycle',
        request: {
          name: this._definition.name,
          lifecycle: this.getLifecycle()
        }
      });
    }
    actions.push({
      type: 'upsert_dot_streams_document',
      request: this._definition
    });
    return actions;
  }
  async createUpsertPipelineActions() {
    const actions = [];
    actions.push({
      type: 'upsert_ingest_pipeline',
      stream: this._definition.name,
      request: {
        id: (0, _name.getProcessingPipelineName)(this._definition.name),
        ...(0, _generate_ingest_pipeline.generateClassicIngestPipelineBody)(this._definition)
      }
    });
    const streamManagedPipelineName = (0, _name.getProcessingPipelineName)(this._definition.name);
    const callStreamManagedPipelineProcessor = {
      pipeline: {
        name: streamManagedPipelineName,
        if: `ctx._index == '${this._definition.name}'`,
        ignore_missing_pipeline: true,
        description: "Call the stream's managed pipeline - do not change this manually but instead use the streams UI or API"
      }
    };
    const pipelineTargets = await this.getPipelineTargets();
    if (!pipelineTargets) {
      throw new _status_error.StatusError('Could not find pipeline targets', 500);
    }
    const {
      pipeline,
      template
    } = pipelineTargets;
    actions.push({
      type: 'append_processor_to_ingest_pipeline',
      pipeline,
      template,
      dataStream: this._definition.name,
      processor: callStreamManagedPipelineProcessor,
      referencePipeline: streamManagedPipelineName
    });
    return actions;
  }
  async doDetermineDeleteActions() {
    const actions = [{
      type: 'delete_datastream',
      request: {
        name: this._definition.name
      }
    }, {
      type: 'delete_dot_streams_document',
      request: {
        name: this._definition.name
      }
    }];
    if (this._definition.ingest.processing.length > 0) {
      const streamManagedPipelineName = (0, _name.getProcessingPipelineName)(this._definition.name);
      actions.push({
        type: 'delete_ingest_pipeline',
        request: {
          name: streamManagedPipelineName
        }
      });
      const pipelineTargets = await this.getPipelineTargets();
      if (pipelineTargets) {
        const {
          pipeline,
          template
        } = pipelineTargets;
        actions.push({
          type: 'delete_processor_from_ingest_pipeline',
          pipeline,
          template,
          dataStream: this._definition.name,
          referencePipeline: streamManagedPipelineName
        });
      }
    }
    return actions;
  }
  async getPipelineTargets() {
    var _unmanagedAssets$inge;
    let dataStream;
    try {
      dataStream = await this.dependencies.streamsClient.getDataStream(this._definition.name);
    } catch (error) {
      if ((0, _esErrors.isNotFoundError)(error)) {
        return undefined;
      }
      throw error;
    }
    const unmanagedAssets = await (0, _stream_crud.getUnmanagedElasticsearchAssets)({
      dataStream,
      scopedClusterClient: this.dependencies.scopedClusterClient
    });
    return {
      pipeline: (_unmanagedAssets$inge = unmanagedAssets.ingestPipeline) !== null && _unmanagedAssets$inge !== void 0 ? _unmanagedAssets$inge : `${dataStream.template}-pipeline`,
      template: dataStream.template
    };
  }
}
exports.UnwiredStream = UnwiredStream;