"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemRuleMigrationsDeleteRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../../common/siem_migrations/constants");
var _rule_migration = require("../../../../../common/siem_migrations/model/api/rules/rule_migration.gen");
var _audit = require("./util/audit");
var _authz = require("./util/authz");
var _with_license = require("./util/with_license");
var _with_existing_migration_id = require("./util/with_existing_migration_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemRuleMigrationsDeleteRoute = (router, logger) => {
  router.versioned.delete({
    path: _constants.SIEM_RULE_MIGRATION_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_rule_migration.GetRuleMigrationRequestParams)
      }
    }
  }, (0, _with_license.withLicense)((0, _with_existing_migration_id.withExistingMigration)(async (context, req, res) => {
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution);
    const {
      migration_id: migrationId
    } = req.params;
    try {
      const ctx = await context.resolve(['securitySolution']);
      const ruleMigrationsClient = ctx.securitySolution.getSiemRuleMigrationsClient();
      await siemMigrationAuditLogger.logDeleteMigration({
        migrationId
      });
      if (ruleMigrationsClient.task.isMigrationRunning(migrationId)) {
        return res.conflict({
          body: 'A running migration cannot be deleted. Please stop the migration first and try again'
        });
      }
      await ruleMigrationsClient.data.deleteMigration(migrationId);
      return res.ok();
    } catch (error) {
      logger.error(error);
      await siemMigrationAuditLogger.logDeleteMigration({
        migrationId,
        error
      });
      return res.badRequest({
        body: error.message
      });
    }
  })));
};
exports.registerSiemRuleMigrationsDeleteRoute = registerSiemRuleMigrationsDeleteRoute;