"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isMlJobMissingError = exports.checkErrorDetails = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const USER_ERRORS_EXCEPTIONS = ['index_not_found_exception', 'status_exception', 'verification_exception', 'parsing_exception'];

/**
 *
 * @param error
 * @returns
 */
const isMlJobMissingError = error => {
  /*
  This is the logic I pulled directly from the ML rule type.
  I am also checking length here because the message returned
  from ES for a missing ML job is seemingly always <job name> missing.
  So to ensure we are not checking "missing" as a user error for other
  possible error messages, I added this check.
  */
  return typeof error === 'string' && error.endsWith('missing') && error.split(' ').length === 2;
};

/**
 * if error can be qualified as user error(configurational), returns isUserError: true
 * user errors are excluded from SLO dashboards
 */
exports.isMlJobMissingError = isMlJobMissingError;
const checkErrorDetails = error => {
  var _errBody, _errBody$error;
  const errorType = error === null || error === void 0 ? void 0 : (_errBody = error.errBody) === null || _errBody === void 0 ? void 0 : (_errBody$error = _errBody.error) === null || _errBody$error === void 0 ? void 0 : _errBody$error.type;
  if (USER_ERRORS_EXCEPTIONS.includes(errorType)) {
    return {
      isUserError: true
    };
  }
  if (error instanceof Error && typeof error.message === 'string' && error.message.endsWith('missing') || isMlJobMissingError(error)) {
    return {
      isUserError: true
    };
  }
  const isUserError = error instanceof Error && USER_ERRORS_EXCEPTIONS.some(exception => error.message.includes(exception)) || typeof error === 'string' && USER_ERRORS_EXCEPTIONS.some(exception => error.includes(exception));
  return {
    isUserError
  };
};
exports.checkErrorDetails = checkErrorDetails;