"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGetUpdatedTags = void 0;
var _react = require("react");
var _utils = require("../../../../common/endpoint/service/artifacts/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_FILTERS = Object.freeze({
  policySelection: _utils.isPolicySelectionTag,
  processDescendantsFiltering: _utils.isFilterProcessDescendantsTag,
  ownerSpaceId: _utils.isOwnerSpaceIdTag
});

/**
 * A hook that returns a callback for using in updating the complete list of `tags` on an artifact.
 * The callback will replace a given type of tag with new set of values - example: update the list
 * of tags on an artifact with a new list of policy assignment tags.
 *
 * The hook uses a `filter` object (can be overwritten on input) that contain a simple filter
 * function that is used to identify tags for that category.
 *
 * ```
 * const FILTERS_IN_ORDER = { // preferably defined out of the component
 *  first: (tag) => tag.startsWith('1:'),
 *  second: (tag) => tag.startsWith('2:'),
 * }
 * ...
 * const { getTagsUpdatedBy } = useGetUpdatedTags(exception, FILTERS_IN_ORDER)
 * ```
 *
 * The returned `getTagsUpdatedBy()` function can be used in event handlers of the given tag category
 * without affecting other tags.
 * ```
 * const newTags = getTagsUpdatedBy('second', ['2:new-tag-1', ...])
 * ```
 *
 * @param exception
 * @param filters
 * @returns `getTagsUpdatedBy(tagCategory, ['new', 'tags'])`
 */
const useGetUpdatedTags = (exception, filters = DEFAULT_FILTERS) => {
  const getTagsUpdatedBy = (0, _react.useCallback)((tagType, newTags) => {
    var _exception$tags;
    if (!filters[tagType]) {
      throw new Error(`getTagsUpdateBy() was called with an unknown tag type: ${String(tagType)}`);
    }
    return ((_exception$tags = exception.tags) !== null && _exception$tags !== void 0 ? _exception$tags : []).filter(tag => !filters[tagType](tag)).concat(...newTags);
  }, [exception, filters]);
  return {
    /**
     * @param tagsToUpdate The category of the tags to update, keys of the filter object.
     * @param newTags
     * @return a new tags array
     */
    getTagsUpdatedBy
  };
};
exports.useGetUpdatedTags = useGetUpdatedTags;