"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.usePersistentQuery = usePersistentQuery;
var _react = require("react");
var _lodash = require("lodash");
var _usePrevious = _interopRequireDefault(require("react-use/lib/usePrevious"));
var _shared_imports = require("../../../../shared_imports");
var _utils = require("../../../../../common/detection_engine/utils");
var _query_bar_field = require("../query_bar_field");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EQL_QUERY_LANGUAGE = 'eql';
const ESQL_QUERY_LANGUAGE = 'esql';
/**
 * Persists query when switching between different rule types using different queries (kuery, EQL and ES|QL).
 *
 * When the user changes rule type to or from "threat_match" this will modify the
 * default "Custom query" string to either:
 *   * from '' to '*:*' if the type is switched to "threat_match"
 *   * from '*:*' back to '' if the type is switched back from "threat_match" to another one
 */
function usePersistentQuery({
  form
}) {
  /**
   * The main idea of this hook is persisting of the state when user switches between
   * different rule types. It's necessary since we conditionally render different query type
   * fields. Kibana's Form lib doesn't persist field values when corresponding UseField hooks
   * aren't rendered.
   */
  const [{
    ruleType,
    queryBar: currentQuery,
    eqlOptions: currentEqlOptions
  }] = (0, _shared_imports.useFormData)({
    form,
    watch: ['ruleType', 'queryBar', 'eqlOptions']
  });
  const previousRuleType = (0, _usePrevious.default)(ruleType);
  const queryRef = (0, _react.useRef)(_query_bar_field.DEFAULT_KQL_QUERY_FIELD_VALUE);
  const eqlQueryRef = (0, _react.useRef)(_query_bar_field.DEFAULT_EQL_QUERY_FIELD_VALUE);
  const eqlOptionsRef = (0, _react.useRef)({});
  const esqlQueryRef = (0, _react.useRef)(_query_bar_field.DEFAULT_ESQL_QUERY_FIELD_VALUE);

  /**
   * This useEffect is concerned about persisting the current query in a corresponding React
   * reference. There are 3 query types (kuery, EQL, ES|QL). An additional check is required to
   * make sure query type matches rule type. When switching between rule types it takes a few
   * re-renders to update the form data.
   */
  (0, _react.useEffect)(() => {
    var _currentQuery$query4;
    if (!ruleType) {
      return;
    }
    if ((0, _utils.isEqlRule)(ruleType)) {
      var _currentQuery$query, _currentQuery$query2;
      eqlQueryRef.current = (currentQuery === null || currentQuery === void 0 ? void 0 : (_currentQuery$query = currentQuery.query) === null || _currentQuery$query === void 0 ? void 0 : _currentQuery$query.language) === EQL_QUERY_LANGUAGE ? currentQuery : eqlQueryRef.current;
      eqlOptionsRef.current = (currentQuery === null || currentQuery === void 0 ? void 0 : (_currentQuery$query2 = currentQuery.query) === null || _currentQuery$query2 === void 0 ? void 0 : _currentQuery$query2.language) === EQL_QUERY_LANGUAGE ? currentEqlOptions : eqlOptionsRef.current;
      return;
    }
    if ((0, _utils.isEsqlRule)(ruleType)) {
      var _currentQuery$query3;
      esqlQueryRef.current = (currentQuery === null || currentQuery === void 0 ? void 0 : (_currentQuery$query3 = currentQuery.query) === null || _currentQuery$query3 === void 0 ? void 0 : _currentQuery$query3.language) === ESQL_QUERY_LANGUAGE ? currentQuery : esqlQueryRef.current;
      return;
    }
    if ([EQL_QUERY_LANGUAGE, ESQL_QUERY_LANGUAGE].includes(currentQuery === null || currentQuery === void 0 ? void 0 : (_currentQuery$query4 = currentQuery.query) === null || _currentQuery$query4 === void 0 ? void 0 : _currentQuery$query4.language)) {
      return;
    }
    queryRef.current = currentQuery;
  }, [ruleType, currentQuery, currentEqlOptions]);

  /**
   * This useEffect restores a corresponding query (kuery, EQL or ES|QL) in form data
   * after switching rule types. There is an exceptional case with threat_match rules
   * having a special default kuery `*:*` while the other rules using kuery have en empty
   * string kuery by default.
   */
  (0, _react.useEffect)(() => {
    if (ruleType === previousRuleType || !ruleType) {
      return;
    }
    const queryField = form.getFields().queryBar;
    const eqlOptionsField = form.getFields().eqlOptions;
    if ((0, _utils.isEqlRule)(ruleType)) {
      queryField.reset({
        defaultValue: eqlQueryRef.current
      });
      eqlOptionsField.reset({
        defaultValue: eqlOptionsRef.current
      });
      return;
    }
    if ((0, _utils.isEsqlRule)(ruleType)) {
      queryField.reset({
        defaultValue: esqlQueryRef.current
      });
      return;
    }
    if ((0, _utils.isThreatMatchRule)(ruleType) && (0, _lodash.isEqual)(queryRef.current, _query_bar_field.DEFAULT_KQL_QUERY_FIELD_VALUE)) {
      queryField.reset({
        defaultValue: _query_bar_field.DEFAULT_THREAT_MATCH_KQL_QUERY_FIELD_VALUE
      });
      return;
    }
    if ((0, _utils.isThreatMatchRule)(previousRuleType) && (0, _lodash.isEqual)(queryRef.current, _query_bar_field.DEFAULT_THREAT_MATCH_KQL_QUERY_FIELD_VALUE)) {
      queryField.reset({
        defaultValue: _query_bar_field.DEFAULT_KQL_QUERY_FIELD_VALUE
      });
    }
    if ((0, _utils.isEqlRule)(previousRuleType) || (0, _utils.isEsqlRule)(previousRuleType)) {
      queryField.reset({
        defaultValue: queryRef.current
      });
    }
  }, [ruleType, previousRuleType, form]);
  return (0, _react.useMemo)(() => ({
    setPersistentQuery: value => queryRef.current = value,
    setPersistentEqlQuery: value => eqlQueryRef.current = value,
    setPersistentEqlOptions: value => eqlOptionsRef.current = value,
    setPersistentEsqlQuery: value => esqlQueryRef.current = value
  }), []);
}