"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateESIndexPatterns = generateESIndexPatterns;
exports.generateMappings = generateMappings;
exports.generateTemplateIndexPattern = generateTemplateIndexPattern;
exports.generateTemplateName = generateTemplateName;
exports.getTemplate = getTemplate;
exports.getTemplatePriority = getTemplatePriority;
exports.updateCurrentWriteIndices = void 0;
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _pMap = _interopRequireDefault(require("p-map"));
var _esErrors = require("@kbn/es-errors");
var _fleet_es_assets = require("../../../../constants/fleet_es_assets");
var _constants = require("../../../../constants");
var _ = require("../../..");
var _services = require("../../../../../common/services");
var _meta2 = require("../meta");
var _retry = require("../retry");
var _errors = require("../../../../errors");
var _mappings = require("./mappings");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_IGNORE_ABOVE = 1024;

// see discussion in https://github.com/elastic/kibana/issues/88307
const DEFAULT_TEMPLATE_PRIORITY = 200;
const DATASET_IS_PREFIX_TEMPLATE_PRIORITY = 150;
const META_PROP_KEYS = ['metric_type', 'unit'];

/**
 * getTemplate retrieves the default template but overwrites the index pattern with the given value.
 *
 * @param indexPattern String with the index pattern
 */
function getTemplate({
  templateIndexPattern,
  packageName,
  composedOfTemplates,
  templatePriority,
  hidden,
  registryElasticsearch,
  mappings,
  isIndexModeTimeSeries,
  type
}) {
  var _appContextService$ge;
  const template = getBaseTemplate({
    templateIndexPattern,
    packageName,
    composedOfTemplates,
    templatePriority,
    registryElasticsearch,
    hidden,
    mappings,
    isIndexModeTimeSeries
  });
  if (template.template.settings.index.final_pipeline) {
    throw new _errors.PackageInvalidArchiveError(`Error template for ${templateIndexPattern} contains a final_pipeline`);
  }
  const esBaseComponents = getBaseEsComponents(type, !!isIndexModeTimeSeries);
  const isEventIngestedEnabled = config => Boolean(!(config !== null && config !== void 0 && config.agentIdVerificationEnabled) && (config === null || config === void 0 ? void 0 : config.eventIngestedEnabled));
  template.composed_of = [...esBaseComponents, ...(template.composed_of || []), _constants.STACK_COMPONENT_TEMPLATE_ECS_MAPPINGS, _constants.FLEET_GLOBALS_COMPONENT_TEMPLATE_NAME, ...((_appContextService$ge = _.appContextService.getConfig()) !== null && _appContextService$ge !== void 0 && _appContextService$ge.agentIdVerificationEnabled ? [_constants.FLEET_AGENT_ID_VERIFY_COMPONENT_TEMPLATE_NAME] : []), ...(isEventIngestedEnabled(_.appContextService.getConfig()) ? [_fleet_es_assets.FLEET_EVENT_INGESTED_COMPONENT_TEMPLATE_NAME] : [])];
  template.ignore_missing_component_templates = template.composed_of.filter(_utils.isUserSettingsTemplate);
  return template;
}
const getBaseEsComponents = (type, isIndexModeTimeSeries) => {
  if (type === 'metrics') {
    if (isIndexModeTimeSeries) {
      return [_constants.STACK_COMPONENT_TEMPLATE_METRICS_TSDB_SETTINGS];
    }
    return [_constants.STACK_COMPONENT_TEMPLATE_METRICS_SETTINGS];
  } else if (type === 'logs') {
    return [_fleet_es_assets.STACK_COMPONENT_TEMPLATE_LOGS_MAPPINGS, _constants.STACK_COMPONENT_TEMPLATE_LOGS_SETTINGS];
  }
  return [];
};

/**
 * Generate mapping takes the given nested fields array and creates the Elasticsearch
 * mapping properties out of it.
 *
 * This assumes that all fields with dotted.names have been expanded in a previous step.
 *
 * @param fields
 */
function generateMappings(fields, isIndexModeTimeSeries = false) {
  const dynamicTemplates = [];
  const dynamicTemplateNames = {};
  const runtimeFields = {};
  const {
    properties
  } = _generateMappings(fields, {
    addDynamicMapping: dynamicMapping => {
      const name = dynamicMapping.path;
      if (name in dynamicTemplateNames) {
        var _dynamicMapping$prope;
        if (name.includes('*') && ((_dynamicMapping$prope = dynamicMapping.properties) === null || _dynamicMapping$prope === void 0 ? void 0 : _dynamicMapping$prope.type) === 'object') {
          // This is a conflicting intermediate object, use the last one so
          // more specific templates are chosen before.
          const index = dynamicTemplateNames[name];
          delete dynamicTemplateNames[name];
          dynamicTemplates.splice(index, 1);
        } else {
          return;
        }
      }
      const dynamicTemplate = {};
      if (dynamicMapping.runtimeProperties !== undefined) {
        dynamicTemplate.runtime = dynamicMapping.runtimeProperties;
      } else {
        dynamicTemplate.mapping = dynamicMapping.properties;
      }
      if (dynamicMapping.matchingType) {
        dynamicTemplate.match_mapping_type = dynamicMapping.matchingType;
      }
      if (dynamicMapping.pathMatch) {
        dynamicTemplate.path_match = dynamicMapping.pathMatch;
      }
      const size = dynamicTemplates.push({
        [name]: dynamicTemplate
      });
      dynamicTemplateNames[name] = size - 1;
    },
    addRuntimeField: runtimeField => {
      runtimeFields[`${runtimeField.path}`] = runtimeField.properties;
    }
  }, isIndexModeTimeSeries);
  const indexTemplateMappings = {
    properties
  };
  if (dynamicTemplates.length > 0) {
    indexTemplateMappings.dynamic_templates = dynamicTemplates;
  }
  if (Object.keys(runtimeFields).length > 0) {
    indexTemplateMappings.runtime = runtimeFields;
  }
  return indexTemplateMappings;
}

/**
 * Generate mapping takes the given nested fields array and creates the Elasticsearch
 * mapping properties out of it.
 *
 * This assumes that all fields with dotted.names have been expanded in a previous step.
 *
 * @param fields
 */
function _generateMappings(fields, ctx, isIndexModeTimeSeries) {
  let hasNonDynamicTemplateMappings = false;
  let hasDynamicTemplateMappings = false;
  let subobjects;
  const props = {};
  function addParentObjectAsStaticProperty(field) {
    // Don't add intermediate objects for wildcard names, as it will
    // be added for its parent object.
    if (field.name.includes('*')) {
      return;
    }
    const fieldProps = {
      type: 'object',
      dynamic: true,
      ...(field.subobjects !== undefined && {
        subobjects: field.subobjects
      })
    };
    props[field.name] = fieldProps;
    hasNonDynamicTemplateMappings = true;
  }
  function addDynamicMappingWithIntermediateObjects(path, pathMatch, matchingType, dynProperties, fieldProps) {
    ctx.addDynamicMapping({
      path,
      pathMatch,
      matchingType,
      properties: dynProperties,
      runtimeProperties: fieldProps
    });
    hasDynamicTemplateMappings = true;

    // Add dynamic intermediate objects.
    const parts = pathMatch.split('.');
    for (let i = parts.length - 1; i > 0; i--) {
      const name = parts.slice(0, i).join('.');
      if (!name.includes('*')) {
        continue;
      }
      const dynProps = {
        type: 'object',
        dynamic: true
      };
      ctx.addDynamicMapping({
        path: name,
        pathMatch: name,
        matchingType: 'object',
        properties: dynProps
      });
    }
  }
  function addObjectAsDynamicMapping(field) {
    var _field$object_type_ma, _field$object_type_ma2, _field$object_type_ma3, _field$object_type_ma4, _field$object_type_ma5, _field$object_type_ma6, _field$object_type_ma7, _field$object_type_ma8, _field$object_type_ma9, _field$object_type_ma10, _field$object_type_ma11;
    const path = ctx.groupFieldName ? `${ctx.groupFieldName}.${field.name}` : field.name;
    const pathMatch = path.includes('*') ? path : `${path}.*`;
    let dynProperties = (0, _mappings.getDefaultProperties)(field);
    let matchingType;
    switch (field.object_type) {
      case 'histogram':
        dynProperties = (0, _mappings.histogram)(field);
        matchingType = (_field$object_type_ma = field.object_type_mapping_type) !== null && _field$object_type_ma !== void 0 ? _field$object_type_ma : '*';
        break;
      case 'ip':
        dynProperties.type = field.object_type;
        matchingType = (_field$object_type_ma2 = field.object_type_mapping_type) !== null && _field$object_type_ma2 !== void 0 ? _field$object_type_ma2 : 'string';
        break;
      case 'keyword':
        dynProperties = (0, _mappings.keyword)(field, true);
        matchingType = (_field$object_type_ma3 = field.object_type_mapping_type) !== null && _field$object_type_ma3 !== void 0 ? _field$object_type_ma3 : 'string';
        if (field.multi_fields) {
          dynProperties.fields = generateMultiFields(field.multi_fields);
        }
        break;
      case 'match_only_text':
      case 'text':
      case 'wildcard':
        matchingType = (_field$object_type_ma4 = field.object_type_mapping_type) !== null && _field$object_type_ma4 !== void 0 ? _field$object_type_ma4 : 'string';
        const textMapping = generateTextMappingForDynamic(field);
        dynProperties = {
          ...dynProperties,
          ...textMapping,
          type: field.object_type
        };
        if (field.multi_fields) {
          dynProperties.fields = generateMultiFields(field.multi_fields);
        }
        break;
      case 'scaled_float':
        dynProperties = (0, _mappings.scaledFloat)(field);
        matchingType = (_field$object_type_ma5 = field.object_type_mapping_type) !== null && _field$object_type_ma5 !== void 0 ? _field$object_type_ma5 : '*';
        break;
      case 'aggregate_metric_double':
        dynProperties.type = field.object_type;
        dynProperties.metrics = field.metrics;
        dynProperties.default_metric = field.default_metric;
        matchingType = (_field$object_type_ma6 = field.object_type_mapping_type) !== null && _field$object_type_ma6 !== void 0 ? _field$object_type_ma6 : '*';
        break;
      case 'double':
      case 'float':
      case 'half_float':
        dynProperties.type = field.object_type;
        if (isIndexModeTimeSeries) {
          dynProperties.time_series_metric = field.metric_type;
        }
        matchingType = (_field$object_type_ma7 = field.object_type_mapping_type) !== null && _field$object_type_ma7 !== void 0 ? _field$object_type_ma7 : 'double';
        break;
      case 'byte':
      case 'long':
      case 'short':
      case 'unsigned_long':
        dynProperties.type = field.object_type;
        if (isIndexModeTimeSeries) {
          dynProperties.time_series_metric = field.metric_type;
        }
        matchingType = (_field$object_type_ma8 = field.object_type_mapping_type) !== null && _field$object_type_ma8 !== void 0 ? _field$object_type_ma8 : 'long';
        break;
      case 'integer':
        // Map integers as long, as in other cases.
        dynProperties.type = 'long';
        if (isIndexModeTimeSeries) {
          dynProperties.time_series_metric = field.metric_type;
        }
        matchingType = (_field$object_type_ma9 = field.object_type_mapping_type) !== null && _field$object_type_ma9 !== void 0 ? _field$object_type_ma9 : 'long';
        break;
      case 'boolean':
        dynProperties.type = field.object_type;
        if (isIndexModeTimeSeries) {
          dynProperties.time_series_metric = field.metric_type;
        }
        matchingType = (_field$object_type_ma10 = field.object_type_mapping_type) !== null && _field$object_type_ma10 !== void 0 ? _field$object_type_ma10 : field.object_type;
        break;
      case 'group':
        if (!(field !== null && field !== void 0 && field.fields)) {
          break;
        }
        const subFields = field.fields.map(subField => {
          var _subField$object_type;
          return {
            ...subField,
            type: 'object',
            object_type: (_subField$object_type = subField.object_type) !== null && _subField$object_type !== void 0 ? _subField$object_type : subField.type
          };
        });
        const mappings = _generateMappings(subFields, {
          ...ctx,
          groupFieldName: ctx.groupFieldName ? `${ctx.groupFieldName}.${field.name}` : field.name
        }, isIndexModeTimeSeries);
        if (mappings.hasDynamicTemplateMappings) {
          hasDynamicTemplateMappings = true;
        }
        break;
      case 'flattened':
        dynProperties.type = field.object_type;
        matchingType = (_field$object_type_ma11 = field.object_type_mapping_type) !== null && _field$object_type_ma11 !== void 0 ? _field$object_type_ma11 : 'object';
        break;
      default:
        throw new _errors.PackageInvalidArchiveError(`No dynamic mapping generated for field ${path} of type ${field.object_type}`);
    }
    if (field.dimension && isIndexModeTimeSeries) {
      dynProperties.time_series_dimension = field.dimension;
    }

    // When a wildcard field specifies the subobjects setting,
    // the parent intermediate object should set the subobjects
    // setting.
    //
    // For example, if a wildcard field `foo.*` has subobjects,
    // we should set subobjects on the intermediate object `foo`.
    //
    if (field.subobjects !== undefined && path.includes('*')) {
      subobjects = field.subobjects;
    }
    if (dynProperties && matchingType) {
      addDynamicMappingWithIntermediateObjects(path, pathMatch, matchingType, dynProperties);

      // Add the parent object as static property, this is needed for
      // index templates not using `"dynamic": true`.
      addParentObjectAsStaticProperty(field);
    }
  }

  // TODO: this can happen when the fields property in fields.yml is present but empty
  // Maybe validation should be moved to fields/field.ts
  if (fields) {
    fields.forEach(field => {
      var _field$object_type_ma12, _field$object_type_ma13;
      // If type is not defined, assume keyword
      const type = field.type || 'keyword';
      if (field.runtime !== undefined) {
        const path = ctx.groupFieldName ? `${ctx.groupFieldName}.${field.name}` : field.name;
        let runtimeFieldProps = (0, _mappings.getDefaultProperties)(field);

        // Is it a dynamic template?
        if (type === 'object' && field.object_type) {
          const pathMatch = path.includes('*') ? path : `${path}.*`;
          const dynProperties = (0, _mappings.getDefaultProperties)(field);
          let matchingType;
          switch (field.object_type) {
            case 'keyword':
              dynProperties.type = field.object_type;
              matchingType = (_field$object_type_ma12 = field.object_type_mapping_type) !== null && _field$object_type_ma12 !== void 0 ? _field$object_type_ma12 : 'string';
              break;
            case 'double':
            case 'long':
            case 'boolean':
              dynProperties.type = field.object_type;
              if (isIndexModeTimeSeries) {
                dynProperties.time_series_metric = field.metric_type;
              }
              matchingType = (_field$object_type_ma13 = field.object_type_mapping_type) !== null && _field$object_type_ma13 !== void 0 ? _field$object_type_ma13 : field.object_type;
            default:
              break;
          }

          // get the runtime properies of this field assuming type equals to object_type
          const _field = {
            ...field,
            type: field.object_type
          };
          const fieldProps = generateRuntimeFieldProps(_field);
          if (dynProperties && matchingType) {
            addDynamicMappingWithIntermediateObjects(path, pathMatch, matchingType, dynProperties, fieldProps);

            // Add the parent object as static property, this is needed for
            // index templates not using `"dynamic": true`.
            addParentObjectAsStaticProperty(field);
          }
          return;
        }
        const fieldProps = generateRuntimeFieldProps(field);
        runtimeFieldProps = {
          ...runtimeFieldProps,
          ...fieldProps
        };
        ctx.addRuntimeField({
          path,
          properties: runtimeFieldProps
        });
        return; // runtime fields should not be added as a property
      }
      if (type === 'object' && field.object_type) {
        addObjectAsDynamicMapping(field);
      } else {
        let fieldProps = (0, _mappings.getDefaultProperties)(field);
        switch (type) {
          case 'group':
            const mappings = _generateMappings(field.fields, {
              ...ctx,
              groupFieldName: ctx.groupFieldName ? `${ctx.groupFieldName}.${field.name}` : field.name
            }, isIndexModeTimeSeries);
            if (field.object_type) {
              // A group can have an object_type if it has been merged with an object during deduplication,
              // generate also the dynamic mapping for the object.
              addObjectAsDynamicMapping(field);
              mappings.hasDynamicTemplateMappings = true;
            }
            if (mappings.hasNonDynamicTemplateMappings) {
              fieldProps = {
                properties: Object.keys(mappings.properties).length > 0 ? mappings.properties : undefined,
                ...generateDynamicAndEnabled(field)
              };
              if (mappings.hasDynamicTemplateMappings) {
                fieldProps.type = 'object';
                fieldProps.dynamic = true;
              }
            } else if (mappings.hasDynamicTemplateMappings) {
              fieldProps = {
                type: 'object',
                dynamic: true
              };
              hasDynamicTemplateMappings = true;
            } else {
              return;
            }
            if (mappings.subobjects !== undefined) {
              fieldProps.subobjects = mappings.subobjects;
            }
            break;
          case 'nested':
          case 'group-nested':
            fieldProps = {
              ...generateNestedProps(field),
              type: 'nested'
            };
            if (field.fields) {
              fieldProps.properties = _generateMappings(field.fields, {
                ...ctx,
                groupFieldName: ctx.groupFieldName ? `${ctx.groupFieldName}.${field.name}` : field.name
              }, isIndexModeTimeSeries).properties;
            }
            break;
          case 'integer':
            fieldProps.type = 'long';
            break;
          case 'scaled_float':
            fieldProps = (0, _mappings.scaledFloat)(field);
            break;
          case 'text':
            const textMapping = generateTextMapping(field);
            fieldProps = {
              ...fieldProps,
              ...textMapping,
              type: 'text'
            };
            if (field.multi_fields) {
              fieldProps.fields = generateMultiFields(field.multi_fields);
            }
            break;
          case 'object':
            fieldProps = {
              ...fieldProps,
              ...generateDynamicAndEnabled(field),
              type: 'object'
            };
            break;
          case 'keyword':
            fieldProps = (0, _mappings.keyword)(field);
            if (field.multi_fields) {
              fieldProps.fields = generateMultiFields(field.multi_fields);
            }
            break;
          case 'wildcard':
            const wildcardMapping = generateWildcardMapping(field);
            fieldProps = {
              ...fieldProps,
              ...wildcardMapping,
              type: 'wildcard'
            };
            if (field.multi_fields) {
              fieldProps.fields = generateMultiFields(field.multi_fields);
            }
            break;
          case 'constant_keyword':
            fieldProps.type = field.type;
            if (field.value) {
              fieldProps.value = field.value;
            }
            break;
          case 'array':
            // this assumes array fields were validated in an earlier step
            // adding an array field with no object_type would result in an error
            // when the template is added to ES
            if (field.object_type) {
              fieldProps.type = field.object_type;
            }
            break;
          case 'alias':
            // this assumes alias fields were validated in an earlier step
            // adding a path to a field that doesn't exist would result in an error
            // when the template is added to ES.
            fieldProps.type = 'alias';
            fieldProps.path = field.path;
            break;
          case 'date':
            const dateMappings = generateDateMapping(field);
            fieldProps = {
              ...fieldProps,
              ...dateMappings,
              type: 'date'
            };
            break;
          case 'aggregate_metric_double':
            fieldProps = {
              ...fieldProps,
              metrics: field.metrics,
              default_metric: field.default_metric,
              type: 'aggregate_metric_double'
            };
            break;
          case 'flattened':
            fieldProps.type = type;
            if (field.ignore_above) {
              fieldProps.ignore_above = field.ignore_above;
            }
            break;
          default:
            fieldProps.type = type;
        }
        const fieldHasMetaProps = META_PROP_KEYS.some(key => key in field);
        if (fieldHasMetaProps) {
          switch (type) {
            case 'group':
            case 'group-nested':
              break;
            default:
              {
                const meta = {};
                if ('unit' in field) Reflect.set(meta, 'unit', field.unit);
                fieldProps.meta = meta;
              }
          }
        }
        if ('metric_type' in field && isIndexModeTimeSeries) {
          fieldProps.time_series_metric = field.metric_type;
        }
        if (field.dimension && isIndexModeTimeSeries) {
          fieldProps.time_series_dimension = field.dimension;
        }
        if (field.subobjects !== undefined) {
          fieldProps.subobjects = field.subobjects;
        }

        // Even if we don't add the property because it has a wildcard, notify
        // the parent that there is some kind of property, so the intermediate object
        // is still created.
        // This is done for legacy packages that include ambiguous mappings with objects
        // without object type. This is not allowed starting on Package Spec v3.
        hasNonDynamicTemplateMappings = true;

        // Avoid including maps with wildcards, they have generated dynamic mappings.
        if (field.name.includes('*')) {
          hasDynamicTemplateMappings = true;
          return;
        }
        props[field.name] = fieldProps;
      }
    });
  }
  return {
    properties: props,
    hasNonDynamicTemplateMappings,
    hasDynamicTemplateMappings,
    subobjects
  };
}
function generateDynamicAndEnabled(field) {
  const props = {};
  if (Object.hasOwn(field, 'enabled')) {
    props.enabled = field.enabled;
  }
  if (Object.hasOwn(field, 'dynamic')) {
    props.dynamic = field.dynamic;
  }
  return props;
}
function generateNestedProps(field) {
  const props = generateDynamicAndEnabled(field);
  if (Object.hasOwn(field, 'include_in_parent')) {
    props.include_in_parent = field.include_in_parent;
  }
  if (Object.hasOwn(field, 'include_in_root')) {
    props.include_in_root = field.include_in_root;
  }
  return props;
}
function generateMultiFields(fields) {
  const multiFields = {};
  if (fields) {
    fields.forEach(f => {
      const type = f.type;
      switch (type) {
        case 'text':
          multiFields[f.name] = {
            ...generateTextMapping(f),
            type: f.type
          };
          break;
        case 'keyword':
          multiFields[f.name] = (0, _mappings.keyword)(f);
          break;
        case 'long':
        case 'double':
        case 'match_only_text':
          multiFields[f.name] = {
            type: f.type
          };
          break;
      }
    });
  }
  return multiFields;
}
function generateTextMapping(field) {
  const mapping = {};
  if (field.analyzer) {
    mapping.analyzer = field.analyzer;
  }
  if (field.search_analyzer) {
    mapping.search_analyzer = field.search_analyzer;
  }
  return mapping;
}
function generateWildcardMapping(field) {
  const mapping = {
    ignore_above: DEFAULT_IGNORE_ABOVE
  };
  if (field.null_value) {
    mapping.null_value = field.null_value;
  }
  if (field.ignore_above) {
    mapping.ignore_above = field.ignore_above;
  }
  return mapping;
}
//  This is a duplicate of the above function, but without the default 'ignore_above' value for dynamic mappings. We dont want to enforce due to backwards compatibility
function generateTextMappingForDynamic(field) {
  const mapping = {};
  if (field.null_value) {
    mapping.null_value = field.null_value;
  }
  if (field.ignore_above) {
    mapping.ignore_above = field.ignore_above;
  }
  return mapping;
}
function generateDateMapping(field) {
  const mapping = {};
  if (field.date_format) {
    mapping.format = field.date_format;
  }
  if (field.name === '@timestamp') {
    mapping.ignore_malformed = false;
  }
  return mapping;
}
function generateRuntimeFieldProps(field) {
  let mapping = {};
  const type = field.type || _mappings.keyword;
  switch (type) {
    case 'integer':
      mapping.type = 'long';
      break;
    case 'date':
      const dateMappings = generateDateMapping(field);
      mapping = {
        ...mapping,
        ...dateMappings,
        type: 'date'
      };
      break;
    default:
      mapping.type = type;
  }
  if (typeof field.runtime === 'string') {
    const scriptObject = {
      source: field.runtime.trim()
    };
    mapping.script = scriptObject;
  }
  return mapping;
}

/**
 * Generates the template name out of the given information
 */
function generateTemplateName(dataStream) {
  return (0, _services.getRegistryDataStreamAssetBaseName)(dataStream);
}

/**
 * Given a data stream name, return the indexTemplate name
 */
async function getIndexTemplate(esClient, dataStreamName) {
  const dataStream = await esClient.indices.getDataStream({
    name: dataStreamName,
    expand_wildcards: ['open', 'hidden']
  });
  return dataStream.data_streams[0].template;
}
function generateTemplateIndexPattern(dataStream) {
  // undefined or explicitly set to false
  // See also https://github.com/elastic/package-spec/pull/102
  if (!dataStream.dataset_is_prefix) {
    return (0, _services.getRegistryDataStreamAssetBaseName)(dataStream) + '-*';
  } else {
    return (0, _services.getRegistryDataStreamAssetBaseName)(dataStream) + '.*-*';
  }
}

// Template priorities are discussed in https://github.com/elastic/kibana/issues/88307
// See also https://www.elastic.co/guide/en/elasticsearch/reference/current/index-templates.html
//
// Built-in templates like logs-*-* and metrics-*-* have priority 100
//
// EPM generated templates for data streams have priority 200 (DEFAULT_TEMPLATE_PRIORITY)
//
// EPM generated templates for data streams with dataset_is_prefix: true have priority 150 (DATASET_IS_PREFIX_TEMPLATE_PRIORITY)

function getTemplatePriority(dataStream) {
  // undefined or explicitly set to false
  // See also https://github.com/elastic/package-spec/pull/102
  if (!dataStream.dataset_is_prefix) {
    return DEFAULT_TEMPLATE_PRIORITY;
  } else {
    return DATASET_IS_PREFIX_TEMPLATE_PRIORITY;
  }
}

/**
 * Returns a map of the data stream path fields to elasticsearch index pattern.
 * @param dataStreams an array of RegistryDataStream objects
 */
function generateESIndexPatterns(dataStreams) {
  if (!dataStreams) {
    return {};
  }
  const patterns = {};
  for (const dataStream of dataStreams) {
    patterns[dataStream.path] = generateTemplateIndexPattern(dataStream);
  }
  return patterns;
}
const flattenFieldsToNameAndType = (fields, path = '') => {
  let newFields = [];
  fields.forEach(field => {
    const fieldName = path ? `${path}.${field.name}` : field.name;
    newFields.push({
      name: fieldName,
      type: field.type
    });
    if (field.fields && field.fields.length) {
      newFields = newFields.concat(flattenFieldsToNameAndType(field.fields, fieldName));
    }
  });
  return newFields;
};
function getBaseTemplate({
  templateIndexPattern,
  packageName,
  composedOfTemplates,
  templatePriority,
  hidden,
  registryElasticsearch,
  mappings,
  isIndexModeTimeSeries
}) {
  var _registryElasticsearc;
  const _meta = (0, _meta2.getESAssetMetadata)({
    packageName
  });
  let settingsIndex = {};
  if (isIndexModeTimeSeries) {
    settingsIndex = {
      mode: 'time_series'
    };
  }
  return {
    priority: templatePriority,
    index_patterns: [templateIndexPattern],
    template: {
      settings: {
        index: settingsIndex
      },
      mappings: {
        _meta
      }
    },
    data_stream: {
      hidden: (registryElasticsearch === null || registryElasticsearch === void 0 ? void 0 : (_registryElasticsearc = registryElasticsearch['index_template.data_stream']) === null || _registryElasticsearc === void 0 ? void 0 : _registryElasticsearc.hidden) || hidden
    },
    composed_of: composedOfTemplates,
    _meta
  };
}
const updateCurrentWriteIndices = async (esClient, logger, templates, options) => {
  if (!templates.length) return;
  const allIndices = await queryDataStreamsFromTemplates(esClient, templates);
  const allUpdatablesIndices = allIndices.filter(indice => {
    if (indice.replicated) {
      logger.warn(`Datastream ${indice.dataStreamName} cannot be updated because this is a replicated datastream.`);
      return false;
    }
    return true;
  });
  if (!allUpdatablesIndices.length) return;
  return updateAllDataStreams(allUpdatablesIndices, esClient, logger, options);
};
exports.updateCurrentWriteIndices = updateCurrentWriteIndices;
function isCurrentDataStream(item) {
  return item !== undefined;
}
const queryDataStreamsFromTemplates = async (esClient, templates) => {
  const dataStreamObjects = await (0, _pMap.default)(templates, template => {
    return getDataStreams(esClient, template);
  }, {
    concurrency: _constants.MAX_CONCURRENT_DATASTREAM_OPERATIONS
  });
  return dataStreamObjects.filter(isCurrentDataStream).flat();
};
const getDataStreams = async (esClient, template) => {
  const {
    indexTemplate
  } = template;
  const body = await esClient.indices.getDataStream({
    name: indexTemplate.index_patterns.join(','),
    expand_wildcards: ['open', 'hidden']
  });
  const dataStreams = body.data_streams;
  if (!dataStreams.length) return;
  return dataStreams.map(dataStream => {
    var _dataStream$indices, _dataStream$indices$a;
    return {
      dataStreamName: dataStream.name,
      replicated: dataStream.replicated,
      indexTemplate,
      currentWriteIndex: (_dataStream$indices = dataStream.indices) === null || _dataStream$indices === void 0 ? void 0 : (_dataStream$indices$a = _dataStream$indices.at(-1)) === null || _dataStream$indices$a === void 0 ? void 0 : _dataStream$indices$a.index_name
    };
  });
};
const MAPPER_EXCEPTION_REASONS_REQUIRING_ROLLOVER = ['subobjects', "[enabled] parameter can't be updated for the object mapping"];
function errorNeedRollover(err) {
  var _err$body, _err$body$error, _err$body2, _err$body2$error, _err$body3, _err$body3$error;
  if ((0, _esErrors.isResponseError)(err) && err.statusCode === 400 && ((_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) === 'illegal_argument_exception') {
    return true;
  }
  if (((_err$body2 = err.body) === null || _err$body2 === void 0 ? void 0 : (_err$body2$error = _err$body2.error) === null || _err$body2$error === void 0 ? void 0 : _err$body2$error.type) === 'mapper_exception' && (_err$body3 = err.body) !== null && _err$body3 !== void 0 && (_err$body3$error = _err$body3.error) !== null && _err$body3$error !== void 0 && _err$body3$error.reason && MAPPER_EXCEPTION_REASONS_REQUIRING_ROLLOVER.some(reason => {
    var _err$body4, _err$body4$error, _err$body4$error$reas;
    return (_err$body4 = err.body) === null || _err$body4 === void 0 ? void 0 : (_err$body4$error = _err$body4.error) === null || _err$body4$error === void 0 ? void 0 : (_err$body4$error$reas = _err$body4$error.reason) === null || _err$body4$error$reas === void 0 ? void 0 : _err$body4$error$reas.includes(reason);
  })) {
    return true;
  }
}
const rolloverDataStream = (dataStreamName, esClient) => {
  try {
    // Do no wrap rollovers in retryTransientEsErrors since it is not idempotent
    return esClient.transport.request({
      method: 'POST',
      path: `/${dataStreamName}/_rollover`,
      querystring: {
        lazy: true
      }
    });
  } catch (error) {
    throw new _errors.PackageESError(`Cannot rollover data stream [${dataStreamName}] due to error: ${error}`);
  }
};
const updateAllDataStreams = async (indexNameWithTemplates, esClient, logger, options) => {
  await (0, _pMap.default)(indexNameWithTemplates, templateEntry => {
    return updateExistingDataStream({
      esClient,
      logger,
      currentWriteIndex: templateEntry.currentWriteIndex,
      dataStreamName: templateEntry.dataStreamName,
      options
    });
  }, {
    concurrency: _constants.MAX_CONCURRENT_DATASTREAM_OPERATIONS
  });
};
const updateExistingDataStream = async ({
  dataStreamName,
  currentWriteIndex,
  esClient,
  logger,
  options
}) => {
  var _currentBackingIndexC, _currentBackingIndexC2, _currentBackingIndexC3, _currentBackingIndexC4, _currentBackingIndexC5, _currentBackingIndexC6, _currentBackingIndexC8, _settings, _settings$index, _settings2, _settings2$index, _settings2$index$mapp, _settings2$index$mapp2, _settings3, _settings3$index, _settings4, _settings4$index, _settings4$index$mapp, _settings4$index$mapp2, _lifecycle, _settings5, _settings5$index;
  const existingDs = await esClient.indices.get({
    index: currentWriteIndex
  });
  const existingDsConfig = Object.values(existingDs);
  const currentBackingIndexConfig = existingDsConfig.at(-1);
  const currentIndexMode = currentBackingIndexConfig === null || currentBackingIndexConfig === void 0 ? void 0 : (_currentBackingIndexC = currentBackingIndexConfig.settings) === null || _currentBackingIndexC === void 0 ? void 0 : (_currentBackingIndexC2 = _currentBackingIndexC.index) === null || _currentBackingIndexC2 === void 0 ? void 0 : _currentBackingIndexC2.mode;
  const currentSourceType = currentBackingIndexConfig === null || currentBackingIndexConfig === void 0 ? void 0 : (_currentBackingIndexC3 = currentBackingIndexConfig.settings) === null || _currentBackingIndexC3 === void 0 ? void 0 : (_currentBackingIndexC4 = _currentBackingIndexC3.index) === null || _currentBackingIndexC4 === void 0 ? void 0 : (_currentBackingIndexC5 = _currentBackingIndexC4.mapping) === null || _currentBackingIndexC5 === void 0 ? void 0 : (_currentBackingIndexC6 = _currentBackingIndexC5.source) === null || _currentBackingIndexC6 === void 0 ? void 0 : _currentBackingIndexC6.mode;
  let settings;
  let mappings = {};
  let lifecycle;
  let subobjectsFieldChanged = false;
  let simulateResult = {};
  try {
    var _currentBackingIndexC7;
    simulateResult = await (0, _retry.retryTransientEsErrors)(async () => esClient.indices.simulateTemplate({
      name: await getIndexTemplate(esClient, dataStreamName)
    }));
    settings = simulateResult.template.settings;
    try {
      mappings = (0, _utils.fillConstantKeywordValues)((currentBackingIndexConfig === null || currentBackingIndexConfig === void 0 ? void 0 : currentBackingIndexConfig.mappings) || {}, simulateResult.template.mappings || {});
    } catch (err) {
      logger.error(`Error filling constant keyword values: ${err}`);
      mappings = simulateResult.template.mappings;
    }
    lifecycle = simulateResult.template.lifecycle;

    // for now, remove from object so as not to update stream or data stream properties of the index until type and name
    // are added in https://github.com/elastic/kibana/issues/66551.  namespace value we will continue
    // to skip updating and assume the value in the index mapping is correct
    if (mappings && mappings.properties) {
      delete mappings.properties.stream;
      delete mappings.properties.data_stream;
    }
    if ((currentBackingIndexConfig === null || currentBackingIndexConfig === void 0 ? void 0 : (_currentBackingIndexC7 = currentBackingIndexConfig.mappings) === null || _currentBackingIndexC7 === void 0 ? void 0 : _currentBackingIndexC7.subobjects) !== mappings.subobjects) {
      subobjectsFieldChanged = true;
    }
    logger.info(`Attempt to update the mappings for the ${dataStreamName} (write_index_only)`);
    await (0, _retry.retryTransientEsErrors)(() => esClient.indices.putMapping({
      index: dataStreamName,
      ...mappings,
      write_index_only: true
    }), {
      logger
    });

    // if update fails, rollover data stream and bail out
  } catch (err) {
    if (errorNeedRollover(err) || subobjectsFieldChanged) {
      logger.info(`Mappings update for ${dataStreamName} failed due to ${err}`);
      logger.trace(`Attempted mappings: ${mappings}`);
      if ((options === null || options === void 0 ? void 0 : options.skipDataStreamRollover) === true) {
        logger.info(`Skipping rollover for ${dataStreamName} as "skipDataStreamRollover" is enabled`);
        return;
      } else {
        logger.info(`Triggering a rollover for ${dataStreamName}`);
        await rolloverDataStream(dataStreamName, esClient);
        return;
      }
    }
    logger.error(`Mappings update for ${dataStreamName} failed due to unexpected error: ${err}`);
    logger.trace(`Attempted mappings: ${mappings}`);
    if ((options === null || options === void 0 ? void 0 : options.ignoreMappingUpdateErrors) === true) {
      logger.info(`Ignore mapping update errors as "ignoreMappingUpdateErrors" is enabled`);
      return;
    } else {
      throw err;
    }
  }
  const filterDimensionMappings = templates => {
    var _templates$filter;
    return (_templates$filter = templates === null || templates === void 0 ? void 0 : templates.filter(template => {
      var _Object$values$, _Object$values$$mappi;
      return (_Object$values$ = Object.values(template)[0]) === null || _Object$values$ === void 0 ? void 0 : (_Object$values$$mappi = _Object$values$.mapping) === null || _Object$values$$mappi === void 0 ? void 0 : _Object$values$$mappi.time_series_dimension;
    })) !== null && _templates$filter !== void 0 ? _templates$filter : [];
  };
  const currentDynamicDimensionMappings = filterDimensionMappings(currentBackingIndexConfig === null || currentBackingIndexConfig === void 0 ? void 0 : (_currentBackingIndexC8 = currentBackingIndexConfig.mappings) === null || _currentBackingIndexC8 === void 0 ? void 0 : _currentBackingIndexC8.dynamic_templates);
  const updatedDynamicDimensionMappings = filterDimensionMappings(mappings.dynamic_templates);
  const sortMappings = (a, b) => Object.keys(a)[0].localeCompare(Object.keys(b)[0]);
  const dynamicDimensionMappingsChanged = !(0, _fastDeepEqual.default)(currentDynamicDimensionMappings.sort(sortMappings), updatedDynamicDimensionMappings.sort(sortMappings));
  const packageDefinedIndexMode = (_settings = settings) === null || _settings === void 0 ? void 0 : (_settings$index = _settings.index) === null || _settings$index === void 0 ? void 0 : _settings$index.mode;
  const packageDefinedSourceMode = (_settings2 = settings) === null || _settings2 === void 0 ? void 0 : (_settings2$index = _settings2.index) === null || _settings2$index === void 0 ? void 0 : (_settings2$index$mapp = _settings2$index.mapping) === null || _settings2$index$mapp === void 0 ? void 0 : (_settings2$index$mapp2 = _settings2$index$mapp.source) === null || _settings2$index$mapp2 === void 0 ? void 0 : _settings2$index$mapp2.mode;

  // Trigger a rollover if the index mode or source type has changed
  if (packageDefinedIndexMode !== undefined && currentIndexMode !== ((_settings3 = settings) === null || _settings3 === void 0 ? void 0 : (_settings3$index = _settings3.index) === null || _settings3$index === void 0 ? void 0 : _settings3$index.mode) || packageDefinedSourceMode !== undefined && currentSourceType !== ((_settings4 = settings) === null || _settings4 === void 0 ? void 0 : (_settings4$index = _settings4.index) === null || _settings4$index === void 0 ? void 0 : (_settings4$index$mapp = _settings4$index.mapping) === null || _settings4$index$mapp === void 0 ? void 0 : (_settings4$index$mapp2 = _settings4$index$mapp.source) === null || _settings4$index$mapp2 === void 0 ? void 0 : _settings4$index$mapp2.mode) || dynamicDimensionMappingsChanged) {
    if ((options === null || options === void 0 ? void 0 : options.skipDataStreamRollover) === true) {
      logger.info(`Index mode or source type or dynamic dimension mappings have changed for ${dataStreamName}, skipping rollover as "skipDataStreamRollover" is enabled`);
      return;
    } else {
      logger.info(dynamicDimensionMappingsChanged ? `Dynamic dimension mappings changed for ${dataStreamName}, triggering a rollover` : `Index mode or source type has changed for ${dataStreamName}, triggering a rollover`);
      await rolloverDataStream(dataStreamName, esClient);
    }
  }
  if ((_lifecycle = lifecycle) !== null && _lifecycle !== void 0 && _lifecycle.data_retention) {
    try {
      logger.debug(`Updating lifecycle for ${dataStreamName}`);
      await (0, _retry.retryTransientEsErrors)(() => esClient.transport.request({
        method: 'PUT',
        path: `_data_stream/${dataStreamName}/_lifecycle`,
        body: {
          data_retention: lifecycle.data_retention
        }
      }), {
        logger
      });
    } catch (err) {
      // Check if this error can happen because of invalid settings;
      // We are returning a 500 but in that case it should be a 400 instead
      throw new _errors.PackageESError(`Could not update lifecycle settings for ${dataStreamName}: ${err.message}`);
    }
  }

  // update settings after mappings was successful to ensure
  // pointing to the new pipeline is safe
  // for now, only update the pipeline
  if (!((_settings5 = settings) !== null && _settings5 !== void 0 && (_settings5$index = _settings5.index) !== null && _settings5$index !== void 0 && _settings5$index.default_pipeline)) {
    return;
  }
  try {
    logger.debug(`Updating index settings of data stream  ${dataStreamName}`);
    await (0, _retry.retryTransientEsErrors)(() => esClient.indices.putSettings({
      index: dataStreamName,
      settings: {
        default_pipeline: settings.index.default_pipeline
      }
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Error updating index settings of data stream ${dataStreamName}: ${err}`);
    // Same as above - Check if this error can happen because of invalid settings;
    // We are returning a 500 but in that case it should be a 400 instead
    throw new _errors.PackageESError(`Could not update index settings of data stream ${dataStreamName}: ${err.message}`);
  }
};