"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processAllRuleGaps = exports.PROCESS_GAPS_DEFAULT_PAGE_SIZE = void 0;
var _find_gaps = require("./find_gaps");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PROCESS_GAPS_DEFAULT_PAGE_SIZE = exports.PROCESS_GAPS_DEFAULT_PAGE_SIZE = 500;
// Circuit breaker to prevent infinite loops
// It should be enough to update 5,000,000 gaps
// 10000 * 500 = 5,000,000 million gaps
const DEFAULT_MAX_ITERATIONS = 10000;

/**
 * Fetches all gaps using search_after pagination to process more than 10,000 gaps with stable sorting
 */
const processAllRuleGaps = async ({
  ruleId,
  start,
  end,
  statuses = [_constants.gapStatus.PARTIALLY_FILLED, _constants.gapStatus.UNFILLED],
  options,
  logger,
  eventLogClient,
  processGapsBatch
}) => {
  let searchAfter;
  let pitId;
  let iterationCount = 0;
  let gapsCount = 0;
  const processingResults = [];
  const {
    maxFetchedGaps
  } = options !== null && options !== void 0 ? options : {};
  try {
    while (true) {
      if (iterationCount >= DEFAULT_MAX_ITERATIONS) {
        logger.warn(`Circuit breaker triggered: Reached maximum number of iterations (${DEFAULT_MAX_ITERATIONS}) while processing gaps for rule ${ruleId}`);
        break;
      }
      iterationCount++;
      const gapsResponse = await (0, _find_gaps.findGapsSearchAfter)({
        eventLogClient,
        logger,
        params: {
          ruleId,
          start,
          end,
          perPage: PROCESS_GAPS_DEFAULT_PAGE_SIZE,
          statuses,
          sortField: '@timestamp',
          sortOrder: 'asc',
          searchAfter,
          pitId
        }
      });
      const {
        data: gaps,
        searchAfter: nextSearchAfter,
        pitId: nextPitId
      } = gapsResponse;
      pitId = nextPitId;
      gapsCount += gaps.length;
      let gapsToProcess = gaps;
      if (maxFetchedGaps && gapsCount > maxFetchedGaps) {
        const offset = gapsCount - maxFetchedGaps;
        gapsToProcess = gapsToProcess.slice(0, gaps.length - offset);
      }
      if (gapsToProcess.length > 0) {
        processingResults.push(await processGapsBatch(gapsToProcess));
      }

      // Exit conditions: no more results or no next search_after or maxFetchedGaps reached
      const maxGapsReached = maxFetchedGaps !== undefined && gapsCount >= maxFetchedGaps;
      if (gapsToProcess.length === 0 || !nextSearchAfter || maxGapsReached) {
        break;
      }
      searchAfter = nextSearchAfter;
    }
  } finally {
    if (pitId) {
      await eventLogClient.closePointInTime(pitId);
    }
  }
  return processingResults;
};
exports.processAllRuleGaps = processAllRuleGaps;