"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.responseStreamFactory = void 0;
var _server = require("@kbn/ml-response-stream/server");
var _index_info_handler = require("./analysis_handlers/index_info_handler");
var _grouping_handler = require("./analysis_handlers/grouping_handler");
var _histogram_handler = require("./analysis_handlers/histogram_handler");
var _overrides_handler = require("./analysis_handlers/overrides_handler");
var _significant_items_handler = require("./analysis_handlers/significant_items_handler");
var _top_items_handler = require("./analysis_handlers/top_items_handler");
var _overall_histogram_handler = require("./analysis_handlers/overall_histogram_handler");
var _log_debug_message = require("./response_stream_utils/log_debug_message");
var _state_handler = require("./response_stream_utils/state_handler");
var _stream_end = require("./response_stream_utils/stream_end");
var _stream_end_with_updated_loading_state = require("./response_stream_utils/stream_end_with_updated_loading_state");
var _stream_push_error = require("./response_stream_utils/stream_push_error");
var _stream_push_ping_with_timeout = require("./response_stream_utils/stream_push_ping_with_timeout");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The options to be passed in to `responseStreamFactory`.
 */

/**
 * The full options object that will be passed on to analysis handlers
 * so they're able to access all necessary runtime dependencies.
 */

/**
 * `responseStreamFactory` sets up the response stream, the stream's state (for example
 * if it's running, how far the stream progressed etc.), some custom actions for the stream
 * as well as analysis handlers that fetch data from ES and pass it on to the stream.
 * This factory should avoid to include any logic, its purpose is to take care of setting up
 * and passing around dependencies for the various other parts involved
 * running the analysis.
 */
const responseStreamFactory = options => {
  var _requestBody$samplePr;
  const {
    events,
    headers,
    logger,
    requestBody
  } = options;
  const logDebugMessage = (0, _log_debug_message.logDebugMessageFactory)(logger);
  const state = (0, _state_handler.stateHandlerFactory)({
    groupingEnabled: !!requestBody.grouping,
    sampleProbability: (_requestBody$samplePr = requestBody.sampleProbability) !== null && _requestBody$samplePr !== void 0 ? _requestBody$samplePr : 1
  });
  const controller = new AbortController();
  const abortSignal = controller.signal;
  events.aborted$.subscribe(() => {
    logDebugMessage('aborted$ subscription trigger.');
    state.shouldStop(true);
    controller.abort();
  });
  events.completed$.subscribe(() => {
    logDebugMessage('completed$ subscription trigger.');
    state.shouldStop(true);
    controller.abort();
  });
  const {
    end: streamEnd,
    push,
    responseWithHeaders
  } = (0, _server.streamFactory)(headers, logger, requestBody.compressResponse, requestBody.flushFix);
  const pushPingWithTimeout = (0, _stream_push_ping_with_timeout.streamPushPingWithTimeoutFactory)(state, push, logDebugMessage);
  const end = (0, _stream_end.streamEndFactory)(state, streamEnd, logDebugMessage);
  const endWithUpdatedLoadingState = (0, _stream_end_with_updated_loading_state.streamEndWithUpdatedLoadingStateFactory)(end, push);
  const pushError = (0, _stream_push_error.streamPushErrorFactory)(push, logDebugMessage);
  const streamFetchOptions = {
    ...options,
    abortSignal,
    logDebugMessage,
    responseStream: {
      end,
      endWithUpdatedLoadingState,
      push,
      pushError,
      pushPingWithTimeout
    },
    stateHandler: state
  };
  return {
    ...streamFetchOptions,
    analysis: {
      indexInfoHandler: (0, _index_info_handler.indexInfoHandlerFactory)(streamFetchOptions),
      groupingHandler: (0, _grouping_handler.groupingHandlerFactory)(streamFetchOptions),
      histogramHandler: (0, _histogram_handler.histogramHandlerFactory)(streamFetchOptions),
      overallHistogramHandler: (0, _overall_histogram_handler.overallHistogramHandlerFactory)(streamFetchOptions),
      overridesHandler: (0, _overrides_handler.overridesHandlerFactory)(streamFetchOptions),
      significantItemsHandler: (0, _significant_items_handler.significantItemsHandlerFactory)(streamFetchOptions),
      topItemsHandler: (0, _top_items_handler.topItemsHandlerFactory)(streamFetchOptions)
    },
    responseWithHeaders
  };
};
exports.responseStreamFactory = responseStreamFactory;