"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchOrCreateRuleExceptionList = void 0;
var _react = require("react");
var _securitysolutionListApi = require("@kbn/securitysolution-list-api");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _api = require("../../rule_management/api/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Hook for fetching or creating an exception list
 *
 * @param http Kibana http service
 * @param ruleId id of the rule
 * @param exceptionListType type of the exception list to be fetched or created
 * @param onError error callback
 *
 */
const useFetchOrCreateRuleExceptionList = ({
  http,
  ruleId,
  exceptionListType,
  onError,
  onSuccess
}) => {
  const [isLoading, setIsLoading] = (0, _react.useState)(false);
  const [exceptionList, setExceptionList] = (0, _react.useState)(null);
  (0, _react.useEffect)(() => {
    let isSubscribed = true;
    const abortCtrl = new AbortController();
    async function createExceptionList(ruleResponse) {
      let newExceptionList;
      if (exceptionListType === 'endpoint') {
        const possibleEndpointExceptionList = await (0, _securitysolutionListApi.addEndpointExceptionList)({
          http,
          signal: abortCtrl.signal
        });
        if (Object.keys(possibleEndpointExceptionList).length === 0) {
          // Endpoint exception list already exists, fetch it
          newExceptionList = await (0, _securitysolutionListApi.fetchExceptionListById)({
            http,
            id: _securitysolutionListConstants.ENDPOINT_LIST_ID,
            namespaceType: 'agnostic',
            signal: abortCtrl.signal
          });
        } else {
          newExceptionList = possibleEndpointExceptionList;
        }
      } else {
        const exceptionListToCreate = {
          name: ruleResponse.name,
          description: ruleResponse.description,
          type: exceptionListType,
          namespace_type: 'single',
          list_id: undefined,
          tags: undefined,
          meta: undefined
        };
        newExceptionList = await (0, _securitysolutionListApi.addExceptionList)({
          http,
          list: exceptionListToCreate,
          signal: abortCtrl.signal
        });
      }
      return Promise.resolve(newExceptionList);
    }
    async function createAndAssociateExceptionList(ruleResponse) {
      var _ruleResponse$excepti;
      const newExceptionList = await createExceptionList(ruleResponse);
      const newExceptionListReference = {
        id: newExceptionList.id,
        list_id: newExceptionList.list_id,
        type: newExceptionList.type,
        namespace_type: newExceptionList.namespace_type
      };
      const newExceptionListReferences = [...((_ruleResponse$excepti = ruleResponse.exceptions_list) !== null && _ruleResponse$excepti !== void 0 ? _ruleResponse$excepti : []), newExceptionListReference];
      await (0, _api.patchRule)({
        ruleProperties: {
          rule_id: ruleResponse.rule_id,
          exceptions_list: newExceptionListReferences
        },
        signal: abortCtrl.signal
      });
      return Promise.resolve(newExceptionList);
    }
    async function fetchRule() {
      return (0, _api.fetchRuleById)({
        id: ruleId,
        signal: abortCtrl.signal
      });
    }
    async function fetchRuleExceptionLists(ruleResponse) {
      const exceptionListReferences = ruleResponse.exceptions_list;
      if (exceptionListReferences && exceptionListReferences.length > 0) {
        const exceptionListPromises = exceptionListReferences.map(exceptionListReference => {
          return (0, _securitysolutionListApi.fetchExceptionListById)({
            http,
            id: exceptionListReference.id,
            namespaceType: exceptionListReference.namespace_type,
            signal: abortCtrl.signal
          });
        });
        return Promise.all(exceptionListPromises);
      } else {
        return Promise.resolve([]);
      }
    }
    async function fetchOrCreateRuleExceptionList() {
      try {
        setIsLoading(true);
        const ruleResponse = await fetchRule();
        const exceptionLists = await fetchRuleExceptionLists(ruleResponse);
        let exceptionListToUse;
        const matchingList = exceptionLists.find(list => {
          if (exceptionListType === 'endpoint') {
            return list.type === exceptionListType && list.list_id === _securitysolutionListConstants.ENDPOINT_LIST_ID;
          } else {
            return list.type === exceptionListType;
          }
        });
        if (matchingList !== undefined) {
          exceptionListToUse = matchingList;
        } else {
          exceptionListToUse = await createAndAssociateExceptionList(ruleResponse);
        }
        if (isSubscribed) {
          setExceptionList(exceptionListToUse);
          setIsLoading(false);
          if (onSuccess) {
            onSuccess(false);
          }
        }
      } catch (error) {
        if (isSubscribed) {
          setIsLoading(false);
          setExceptionList(null);
          if (error.body != null) {
            onError(error, error.body.status_code, error.body.message);
          } else {
            onError(error, null, null);
          }
        }
      }
    }
    fetchOrCreateRuleExceptionList();
    return () => {
      isSubscribed = false;
      abortCtrl.abort();
    };
  }, [http, ruleId, exceptionListType, onError, onSuccess]);
  return [isLoading, exceptionList];
};
exports.useFetchOrCreateRuleExceptionList = useFetchOrCreateRuleExceptionList;