"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PrivilegeSummaryCalculator = void 0;
var _securityRoleManagementModel = require("@kbn/security-role-management-model");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class PrivilegeSummaryCalculator {
  constructor(kibanaPrivileges, role) {
    this.kibanaPrivileges = kibanaPrivileges;
    this.role = role;
  }
  getEffectiveFeaturePrivileges(entry) {
    const assignedPrivileges = this.collectAssignedPrivileges(entry);
    const features = this.kibanaPrivileges.getSecuredFeatures();
    return features.reduce((acc, feature) => {
      const displayedPrimaryFeaturePrivilege = this.getDisplayedPrimaryFeaturePrivilege(assignedPrivileges, feature);
      const effectiveSubPrivileges = feature.getSubFeaturePrivileges().filter(ap => assignedPrivileges.grantsPrivilege(ap));
      const hasCustomizedSubFeaturePrivileges = this.hasCustomizedSubFeaturePrivileges(feature, displayedPrimaryFeaturePrivilege, entry);
      return {
        ...acc,
        [feature.id]: {
          primary: displayedPrimaryFeaturePrivilege,
          hasCustomizedSubFeaturePrivileges,
          subFeature: effectiveSubPrivileges.map(p => p.id)
        }
      };
    }, {});
  }
  hasCustomizedSubFeaturePrivileges(feature, displayedPrimaryFeaturePrivilege, entry) {
    const formPrivileges = this.collectAssignedPrivileges(entry);
    return feature.getSubFeaturePrivileges().some(sfp => {
      var _displayedPrimaryFeat;
      const isGranted = formPrivileges.grantsPrivilege(sfp);
      const isGrantedByDisplayedPrimary = (_displayedPrimaryFeat = displayedPrimaryFeaturePrivilege === null || displayedPrimaryFeaturePrivilege === void 0 ? void 0 : displayedPrimaryFeaturePrivilege.grantsPrivilege(sfp)) !== null && _displayedPrimaryFeat !== void 0 ? _displayedPrimaryFeat : isGranted;

      // if displayed primary is derived from base, then excluded sub-feature-privs should not count.
      return isGranted !== isGrantedByDisplayedPrimary;
    });
  }
  getDisplayedPrimaryFeaturePrivilege(assignedPrivileges, feature) {
    const primaryFeaturePrivileges = feature.getPrimaryFeaturePrivileges();
    const minimalPrimaryFeaturePrivileges = feature.getMinimalFeaturePrivileges();
    const hasMinimalPrivileges = feature.subFeatures.length > 0;
    const effectivePrivilege = primaryFeaturePrivileges.find(pfp => {
      const isPrimaryGranted = assignedPrivileges.grantsPrivilege(pfp);
      if (!isPrimaryGranted && hasMinimalPrivileges) {
        const correspondingMinimal = minimalPrimaryFeaturePrivileges.find(mpfp => mpfp.id === pfp.getMinimalPrivilegeId());
        return assignedPrivileges.grantsPrivilege(correspondingMinimal);
      }
      return isPrimaryGranted;
    });
    return effectivePrivilege;
  }
  collectAssignedPrivileges(entry) {
    if ((0, _securityRoleManagementModel.isGlobalPrivilegeDefinition)(entry)) {
      return this.kibanaPrivileges.createCollectionFromRoleKibanaPrivileges([entry]);
    }
    const globalPrivilege = this.locateGlobalPrivilege(this.role);
    return this.kibanaPrivileges.createCollectionFromRoleKibanaPrivileges(globalPrivilege ? [globalPrivilege, entry] : [entry]);
  }
  locateGlobalPrivilege(role) {
    return role.kibana.find(entry => (0, _securityRoleManagementModel.isGlobalPrivilegeDefinition)(entry));
  }
}
exports.PrivilegeSummaryCalculator = PrivilegeSummaryCalculator;