"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ParamsSchema = exports.CoreQueryParamsSchemaProperties = exports.CoreQueryParamsSchema = void 0;
exports.validateCoreQueryBody = validateCoreQueryBody;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _constants = require("../common/constants");
var _utils = require("../common/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const CoreQueryParamsSchemaProperties = exports.CoreQueryParamsSchemaProperties = {
  index: _configSchema.schema.oneOf([_configSchema.schema.string({
    minLength: 1
  }), _configSchema.schema.arrayOf(_configSchema.schema.string({
    minLength: 1
  }), {
    minSize: 1
  })], {
    meta: {
      description: 'The indices to query.'
    }
  }),
  timeField: _configSchema.schema.string({
    minLength: 1,
    meta: {
      description: 'The field that is used to calculate the time window.'
    }
  }),
  aggType: _configSchema.schema.string({
    validate: _utils.validateAggType,
    defaultValue: 'count',
    meta: {
      description: 'The type of aggregation to perform.'
    }
  }),
  aggField: _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1,
    meta: {
      description: 'The name of the numeric field that is used in the aggregation. This property is required when `aggType` is `avg`, `max`, `min` or `sum`.'
    }
  })),
  groupBy: _configSchema.schema.string({
    validate: _utils.validateGroupBy,
    defaultValue: 'all',
    meta: {
      description: 'Indicates whether the aggregation is applied over all documents (`all`) or split into groups (`top`) using a grouping field (`termField`). If grouping is used, an alert will be created for each group when it exceeds the threshold; only the top groups (up to `termSize` number of groups) are checked.'
    }
  }),
  /**
   *
   *
   */
  termField: _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1,
    meta: {
      description: 'The names of up to four fields that are used for grouping the aggregation. This property is required when `groupBy` is `top`.'
    }
  })),
  filterKuery: _configSchema.schema.maybe(_configSchema.schema.string({
    validate: _utils.validateKuery,
    meta: {
      description: 'A Kibana Query Language (KQL) expression thats limits the scope of alerts.'
    }
  })),
  termSize: _configSchema.schema.maybe(_configSchema.schema.number({
    min: 1,
    meta: {
      description: 'This property is required when `groupBy` is `top`. It specifies the number of groups to check against the threshold and therefore limits the number of alerts on high cardinality fields.'
    }
  })),
  timeWindowSize: _configSchema.schema.number({
    min: 1,
    meta: {
      description: 'The size of the time window (in `timeWindowUnit` units), which determines how far back to search for documents. Generally it should be a value higher than the rule check interval to avoid gaps in detection.'
    }
  }),
  timeWindowUnit: _configSchema.schema.string({
    validate: _utils.validateTimeWindowUnits,
    meta: {
      description: 'The type of units for the time window. For example: seconds, minutes, hours, or days.'
    }
  })
};
const CoreQueryParamsSchema = exports.CoreQueryParamsSchema = _configSchema.schema.object(CoreQueryParamsSchemaProperties);
const ParamsSchema = exports.ParamsSchema = _configSchema.schema.object({
  ...CoreQueryParamsSchemaProperties,
  // the comparison function to use to determine if the threshold as been met
  thresholdComparator: (0, _utils.getComparatorSchemaType)(_utils.validateComparator),
  // the values to use as the threshold; `between` and `notBetween` require
  // two values, the others require one.
  threshold: _configSchema.schema.arrayOf(_configSchema.schema.number(), {
    minSize: 1,
    maxSize: 2
  })
}, {
  validate: validateParams
});

// using direct type not allowed, circular reference, so body is typed to any
function validateParams(anyParams) {
  // validate core query parts, return if it fails validation (returning string)
  const coreQueryValidated = validateCoreQueryBody(anyParams);
  if (coreQueryValidated) return coreQueryValidated;
  const {
    thresholdComparator,
    threshold
  } = anyParams;
  if (_utils.betweenComparators.has(thresholdComparator) && threshold.length === 1) {
    return _i18n.i18n.translate('xpack.responseOps.ruleParams.indexThreshold.invalidThreshold2ErrorMessage', {
      defaultMessage: '[threshold]: must have two elements for the "{thresholdComparator}" comparator',
      values: {
        thresholdComparator
      }
    });
  }
}
function validateCoreQueryBody(anyParams) {
  const {
    aggType,
    aggField,
    groupBy,
    termField,
    termSize
  } = anyParams;
  if (aggType !== 'count' && !aggField) {
    return _i18n.i18n.translate('xpack.responseOps.ruleParams.coreQueryParams.aggTypeRequiredErrorMessage', {
      defaultMessage: '[aggField]: must have a value when [aggType] is "{aggType}"',
      values: {
        aggType
      }
    });
  }

  // check grouping
  if (groupBy === 'top') {
    if (termField == null) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.coreQueryParams.termFieldRequiredErrorMessage', {
        defaultMessage: '[termField]: termField required when [groupBy] is top'
      });
    }
    if (termSize == null) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.coreQueryParams.termSizeRequiredErrorMessage', {
        defaultMessage: '[termSize]: termSize required when [groupBy] is top'
      });
    }
    if (termSize > _constants.MAX_GROUPS) {
      return _i18n.i18n.translate('xpack.responseOps.ruleParams.coreQueryParams.invalidTermSizeMaximumErrorMessage', {
        defaultMessage: '[termSize]: must be less than or equal to {maxGroups}',
        values: {
          maxGroups: _constants.MAX_GROUPS
        }
      });
    }
  }
}