"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSetupRoute = registerSetupRoute;
var _common = require("@kbn/spaces-plugin/common");
var _common2 = require("../../../common");
var _get_has_setup_privileges = require("../../lib/setup/get_has_setup_privileges");
var _handle_route_error_handler = require("../../utils/handle_route_error_handler");
var _compat = require("../compat");
var _get_cloud_setup_instructions = require("./get_cloud_setup_instructions");
var _get_self_managed_instructions = require("./get_self_managed_instructions");
var _setup_cloud = require("./setup_cloud");
var _setup_self_managed = require("./setup_self_managed");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerSetupRoute({
  router,
  logger,
  services: {
    createProfilingEsClient
  },
  dependencies
}) {
  const paths = (0, _common2.getRoutePaths)();
  router.get({
    path: paths.HasSetupESResources,
    security: {
      authz: {
        requiredPrivileges: ['profiling']
      }
    },
    validate: false
  }, async (context, request, response) => {
    try {
      var _dependencies$setup$s, _dependencies$setup$s2;
      const hasRequiredRole = dependencies.start.security ? await (0, _get_has_setup_privileges.getHasSetupPrivileges)({
        securityPluginStart: dependencies.start.security,
        request
      }) : true;
      const core = await context.core;
      const profilingStatus = await dependencies.start.profilingDataAccess.services.getStatus({
        esClient: core.elasticsearch.client,
        soClient: core.savedObjects.client,
        spaceId: (_dependencies$setup$s = dependencies.setup.spaces) === null || _dependencies$setup$s === void 0 ? void 0 : (_dependencies$setup$s2 = _dependencies$setup$s.spacesService) === null || _dependencies$setup$s2 === void 0 ? void 0 : _dependencies$setup$s2.getSpaceId(request)
      });
      return response.ok({
        body: {
          ...profilingStatus,
          has_required_role: hasRequiredRole
        }
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while checking plugin setup'
      });
    }
  });
  router.post({
    path: paths.HasSetupESResources,
    security: {
      authz: {
        requiredPrivileges: ['profiling']
      }
    },
    validate: false
  }, async (context, request, response) => {
    try {
      var _dependencies$setup$s3, _dependencies$setup$s4, _dependencies$setup$s5, _dependencies$setup$s6, _dependencies$setup$s7, _dependencies$setup$s8, _dependencies$setup$c;
      const esClient = await (0, _compat.getClient)(context);
      const core = await context.core;
      const clientWithDefaultAuth = createProfilingEsClient({
        esClient,
        request,
        useDefaultAuth: true
      });
      const clientWithProfilingAuth = createProfilingEsClient({
        esClient,
        request,
        useDefaultAuth: false
      });
      const commonSetupParams = {
        client: clientWithDefaultAuth,
        clientWithProfilingAuth,
        logger,
        soClient: core.savedObjects.client,
        spaceId: (_dependencies$setup$s3 = (_dependencies$setup$s4 = dependencies.setup.spaces) === null || _dependencies$setup$s4 === void 0 ? void 0 : (_dependencies$setup$s5 = _dependencies$setup$s4.spacesService) === null || _dependencies$setup$s5 === void 0 ? void 0 : _dependencies$setup$s5.getSpaceId(request)) !== null && _dependencies$setup$s3 !== void 0 ? _dependencies$setup$s3 : _common.DEFAULT_SPACE_ID
      };
      const scopedESClient = (await context.core).elasticsearch.client;
      const {
        type,
        setupState
      } = await dependencies.start.profilingDataAccess.services.getSetupState({
        esClient: scopedESClient,
        soClient: core.savedObjects.client,
        spaceId: (_dependencies$setup$s6 = (_dependencies$setup$s7 = dependencies.setup.spaces) === null || _dependencies$setup$s7 === void 0 ? void 0 : (_dependencies$setup$s8 = _dependencies$setup$s7.spacesService) === null || _dependencies$setup$s8 === void 0 ? void 0 : _dependencies$setup$s8.getSpaceId(request)) !== null && _dependencies$setup$s6 !== void 0 ? _dependencies$setup$s6 : _common.DEFAULT_SPACE_ID
      });
      const isCloudEnabled = (_dependencies$setup$c = dependencies.setup.cloud) === null || _dependencies$setup$c === void 0 ? void 0 : _dependencies$setup$c.isCloudEnabled;
      if (isCloudEnabled && type === 'cloud') {
        if (!dependencies.start.fleet) {
          const msg = `Elastic Fleet is required to set up Universal Profiling on Cloud`;
          logger.error(msg);
          return response.custom({
            statusCode: 500,
            body: {
              message: msg
            }
          });
        }
        logger.debug('Setting up Universal Profiling on Cloud');
        await (0, _setup_cloud.setupCloud)({
          setupState,
          setupParams: {
            ...commonSetupParams,
            packagePolicyClient: dependencies.start.fleet.packagePolicyService,
            isCloudEnabled,
            config: dependencies.config
          }
        });
        logger.debug('[DONE] Setting up Universal Profiling on Cloud');
      } else {
        logger.debug('Setting up self-managed Universal Profiling');
        await (0, _setup_self_managed.setupSelfManaged)({
          setupState,
          setupParams: commonSetupParams
        });
        logger.debug('[DONE] Setting up self-managed Universal Profiling');
      }

      // Wait until Profiling ES plugin creates all resources
      await clientWithDefaultAuth.profilingStatus({
        waitForResourcesCreated: true
      });
      if (dependencies.telemetryUsageCounter) {
        dependencies.telemetryUsageCounter.incrementCounter({
          counterName: `POST ${paths.HasSetupESResources}`,
          counterType: 'success'
        });
      }

      // We return a status code of 202 instead of 200 because enabling
      // resource management in Elasticsearch is an asynchronous action
      // and is not guaranteed to complete before Kibana sends a response.
      return response.accepted();
    } catch (error) {
      if (dependencies.telemetryUsageCounter) {
        dependencies.telemetryUsageCounter.incrementCounter({
          counterName: `POST ${paths.HasSetupESResources}`,
          counterType: 'error'
        });
      }
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while setting up Universal Profiling'
      });
    }
  });
  // Show users the instructions on how to set up Universal Profiling agents
  router.get({
    path: paths.SetupDataCollectionInstructions,
    security: {
      authz: {
        requiredPrivileges: ['profiling']
      }
    },
    validate: false
  }, async (context, request, response) => {
    try {
      var _dependencies$setup$c2;
      const stackVersion = dependencies.stackVersion;
      const isCloudEnabled = (_dependencies$setup$c2 = dependencies.setup.cloud) === null || _dependencies$setup$c2 === void 0 ? void 0 : _dependencies$setup$c2.isCloudEnabled;
      if (isCloudEnabled) {
        var _dependencies$setup$c3, _dependencies$setup$c4, _dependencies$start$f;
        if (!dependencies.start.fleet) {
          const msg = `Elastic Fleet is required to set up Universal Profiling on Cloud`;
          logger.error(msg);
          return response.custom({
            statusCode: 500,
            body: {
              message: msg
            }
          });
        }
        const apmServerHost = (_dependencies$setup$c3 = dependencies.setup.cloud) === null || _dependencies$setup$c3 === void 0 ? void 0 : (_dependencies$setup$c4 = _dependencies$setup$c3.apm) === null || _dependencies$setup$c4 === void 0 ? void 0 : _dependencies$setup$c4.url;
        const setupInstructions = await (0, _get_cloud_setup_instructions.getCloudSetupInstructions)({
          packagePolicyClient: (_dependencies$start$f = dependencies.start.fleet) === null || _dependencies$start$f === void 0 ? void 0 : _dependencies$start$f.packagePolicyService,
          soClient: (await context.core).savedObjects.client,
          apmServerHost,
          stackVersion
        });
        return response.ok({
          body: setupInstructions
        });
      }
      return response.ok({
        body: (0, _get_self_managed_instructions.getSelfManagedInstructions)({
          stackVersion
        })
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while fetching Universal Profiling instructions'
      });
    }
  });
}