"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateEsqlTask = void 0;
var _rxjs = require("rxjs");
var _inferenceCommon = require("@kbn/inference-common");
var _common = require("../../../../common");
var _constants = require("../../../../common/tasks/nl_to_esql/constants");
var _shared = require("./shared");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CALLS = 8;
const generateEsqlTask = ({
  chatCompleteApi,
  connectorId,
  systemMessage,
  messages,
  toolOptions: {
    tools,
    toolChoice
  },
  docBase,
  functionCalling,
  maxRetries,
  retryConfiguration,
  logger,
  system,
  metadata,
  maxCallsAllowed = MAX_CALLS
}) => {
  return function askLlmToRespond({
    documentationRequest: {
      commands,
      functions
    },
    callCount = 0
  }) {
    const functionLimitReached = callCount >= maxCallsAllowed;
    const availableTools = Object.keys(tools !== null && tools !== void 0 ? tools : {});
    const hasTools = !functionLimitReached && availableTools.length > 0;
    const keywords = [...(commands !== null && commands !== void 0 ? commands : []), ...(functions !== null && functions !== void 0 ? functions : [])];
    const requestedDocumentation = docBase.getDocumentation(keywords);
    const fakeRequestDocsToolCall = createFakeTooCall(commands, functions);
    const hasToolBlock = hasTools ? `**IMPORTANT**: If there is a tool suitable for answering the user's question, use that tool,
preferably with a natural language reply included. DO NOT attempt to call any other tools
that are not explicitly listed as available. Only use the following available tools: ${availableTools.join(', ')}` : '**IMPORTANT**: There are no tools available to use. Do not attempt to call any tools.';
    return (0, _rxjs.merge)((0, _rxjs.of)({
      type: _inferenceCommon.OutputEventType.OutputComplete,
      id: 'request_documentation',
      output: {
        keywords,
        requestedDocumentation
      },
      content: ''
    }), chatCompleteApi({
      connectorId,
      functionCalling,
      maxRetries,
      retryConfiguration,
      metadata,
      stream: true,
      system: `${systemMessage}

          # Current task

          Your current task is to respond to the user's question. If there is a tool
          suitable for answering the user's question, use that tool, preferably
          with a natural language reply included.

          ${hasToolBlock}

          Format any ES|QL query as follows:
          \`\`\`esql
          <query>
          \`\`\`

          When generating ES|QL, it is VERY important that you only use commands and functions present in the
          requested documentation, and follow the syntax as described in the documentation and its examples.
          Assume that ONLY the set of capabilities described in the provided ES|QL documentation is valid, and
          do not try to guess parameters or syntax based on other query languages.

          If what the user is asking for is not technically achievable with ES|QL's capabilities, just inform
          the user. DO NOT invent capabilities not described in the documentation just to provide
          a positive answer to the user. E.g. Pagination is not supported by the language, do not try to invent
          workarounds based on other languages.

          When converting queries from one language to ES|QL, make sure that the functions are available
          and documented in ES|QL. E.g., for SPL's LEN, use LENGTH. For IF, use CASE.
          ${system ? `## Additional instructions\n\n${system}` : ''}`,
      messages: [...messages, {
        role: _inferenceCommon.MessageRole.Assistant,
        content: null,
        toolCalls: [fakeRequestDocsToolCall]
      }, {
        name: fakeRequestDocsToolCall.function.name,
        role: _inferenceCommon.MessageRole.Tool,
        response: {
          documentation: requestedDocumentation
        },
        toolCallId: fakeRequestDocsToolCall.toolCallId
      }],
      toolChoice: !functionLimitReached ? toolChoice : _inferenceCommon.ToolChoiceType.none,
      tools: functionLimitReached ? {} : {
        ...tools,
        request_documentation: {
          description: 'Request additional ES|QL documentation if needed',
          schema: _shared.requestDocumentationSchema
        }
      }
    }).pipe((0, _inferenceCommon.withoutTokenCountEvents)(), (0, _rxjs.map)(generateEvent => {
      if ((0, _inferenceCommon.isChatCompletionMessageEvent)(generateEvent)) {
        return {
          ...generateEvent,
          content: generateEvent.content ? correctEsqlMistakes({
            content: generateEvent.content,
            logger
          }) : generateEvent.content
        };
      }
      return generateEvent;
    }), (0, _rxjs.switchMap)(generateEvent => {
      if ((0, _inferenceCommon.isChatCompletionMessageEvent)(generateEvent)) {
        const toolCalls = generateEvent.toolCalls;
        const onlyToolCall = toolCalls.length === 1 ? toolCalls[0] : undefined;
        if (onlyToolCall && onlyToolCall.function.name === 'request_documentation') {
          var _args$commands, _args$functions;
          if (functionLimitReached) {
            return (0, _rxjs.of)({
              ...generateEvent,
              content: `You have reached the maximum number of documentation requests. Do not try to request documentation again for commands ${commands === null || commands === void 0 ? void 0 : commands.join(', ')} and functions ${functions === null || functions === void 0 ? void 0 : functions.join(', ')}. Try to answer the user's question using currently available information.`
            });
          }
          const args = 'arguments' in onlyToolCall.function ? onlyToolCall.function.arguments : undefined;
          if (args && ((_args$commands = args.commands) !== null && _args$commands !== void 0 && _args$commands.length || (_args$functions = args.functions) !== null && _args$functions !== void 0 && _args$functions.length)) {
            var _args$commands2, _args$functions2;
            return askLlmToRespond({
              documentationRequest: {
                commands: (_args$commands2 = args.commands) !== null && _args$commands2 !== void 0 ? _args$commands2 : [],
                functions: (_args$functions2 = args.functions) !== null && _args$functions2 !== void 0 ? _args$functions2 : []
              },
              callCount: callCount + 1
            });
          }
        }
      }
      return (0, _rxjs.of)(generateEvent);
    })));
  };
};
exports.generateEsqlTask = generateEsqlTask;
const correctEsqlMistakes = ({
  content,
  logger
}) => {
  return content.replaceAll(_constants.INLINE_ESQL_QUERY_REGEX, (_match, query) => {
    const correction = (0, _common.correctCommonEsqlMistakes)(query);
    if (correction.isCorrection) {
      logger.debug(`Corrected query, from: \n${correction.input}\nto:\n${correction.output}`);
    }
    return '```esql\n' + correction.output + '\n```';
  });
};
const createFakeTooCall = (commands, functions) => {
  return {
    function: {
      name: 'request_documentation',
      arguments: {
        commands,
        functions
      }
    },
    toolCallId: (0, _common.generateFakeToolCallId)()
  };
};