"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSecrets = createSecrets;
exports.deleteDownloadSourceSecrets = deleteDownloadSourceSecrets;
exports.deleteFleetServerHostsSecrets = deleteFleetServerHostsSecrets;
exports.deleteOutputSecrets = deleteOutputSecrets;
exports.deleteSOSecrets = deleteSOSecrets;
exports.deleteSecrets = deleteSecrets;
exports.deleteSecretsIfNotReferenced = deleteSecretsIfNotReferenced;
exports.diffSOSecretPaths = diffSOSecretPaths;
exports.diffSecretPaths = diffSecretPaths;
exports.extractAndUpdateDownloadSourceSecrets = extractAndUpdateDownloadSourceSecrets;
exports.extractAndUpdateFleetServerHostsSecrets = extractAndUpdateFleetServerHostsSecrets;
exports.extractAndUpdateOutputSecrets = extractAndUpdateOutputSecrets;
exports.extractAndUpdateSecrets = extractAndUpdateSecrets;
exports.extractAndWriteDownloadSourcesSecrets = extractAndWriteDownloadSourcesSecrets;
exports.extractAndWriteFleetServerHostsSecrets = extractAndWriteFleetServerHostsSecrets;
exports.extractAndWriteOutputSecrets = extractAndWriteOutputSecrets;
exports.extractAndWriteSecrets = extractAndWriteSecrets;
exports.findPackagePoliciesUsingSecrets = findPackagePoliciesUsingSecrets;
exports.getDownloadSourceSecretReferences = getDownloadSourceSecretReferences;
exports.getFleetServerHostsSecretReferences = getFleetServerHostsSecretReferences;
exports.getOutputSecretReferences = getOutputSecretReferences;
exports.getPolicySecretPaths = getPolicySecretPaths;
exports.isOutputSecretStorageEnabled = isOutputSecretStorageEnabled;
exports.isSecretStorageEnabled = isSecretStorageEnabled;
exports.toCompiledSecretRef = toCompiledSecretRef;
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _policy_template = require("../../common/services/policy_template");
var _common = require("../../common");
var _services = require("../../common/services");
var _errors = require("../errors");
var _constants = require("../constants");
var _retry = require("./epm/elasticsearch/retry");
var _audit_logging = require("./audit_logging");
var _app_context = require("./app_context");
var _package_policy = require("./package_policy");
var _ = require(".");
var _fleet_server = require("./fleet_server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function createSecrets(opts) {
  const {
    esClient,
    values
  } = opts;
  const logger = _app_context.appContextService.getLogger();
  const sendESRequest = value => {
    return (0, _retry.retryTransientEsErrors)(() => esClient.transport.request({
      method: 'POST',
      path: _constants.SECRETS_ENDPOINT_PATH,
      body: {
        value
      }
    }), {
      logger
    });
  };
  const secretsResponse = await Promise.all(values.map(async value => {
    try {
      if (Array.isArray(value)) {
        return await Promise.all(value.map(sendESRequest));
      } else {
        return await sendESRequest(value);
      }
    } catch (err) {
      const msg = `Error creating secrets: ${err}`;
      logger.error(msg);
      throw new _errors.FleetError(msg);
    }
  }));
  const writeLog = item => {
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `secret created: ${item.id}`,
      event: {
        action: 'secret_create',
        category: ['database'],
        type: ['access'],
        outcome: 'success'
      }
    });
  };
  secretsResponse.forEach(item => {
    if (Array.isArray(item)) {
      item.forEach(writeLog);
    } else {
      writeLog(item);
    }
  });
  return secretsResponse;
}
async function deleteSecretsIfNotReferenced(opts) {
  const {
    esClient,
    soClient,
    ids
  } = opts;
  const logger = _app_context.appContextService.getLogger();
  const packagePoliciesUsingSecrets = await findPackagePoliciesUsingSecrets({
    soClient,
    ids
  });
  if (packagePoliciesUsingSecrets.length) {
    packagePoliciesUsingSecrets.forEach(({
      id,
      policyIds
    }) => {
      logger.debug(`Not deleting secret with id ${id} is still in use by package policies: ${policyIds.join(', ')}`);
    });
  }
  const secretsToDelete = ids.filter(id => {
    return !packagePoliciesUsingSecrets.some(packagePolicy => packagePolicy.id === id);
  });
  if (!secretsToDelete.length) {
    return;
  }
  try {
    await deleteSecrets({
      esClient,
      ids: secretsToDelete
    });
  } catch (e) {
    logger.warn(`Error cleaning up secrets ${ids.join(', ')}: ${e}`);
  }
}
async function findPackagePoliciesUsingSecrets(opts) {
  const {
    soClient,
    ids
  } = opts;
  const packagePolicies = await _package_policy.packagePolicyService.list(soClient, {
    kuery: `ingest-package-policies.secret_references.id: (${ids.join(' or ')})`,
    perPage: _common.SO_SEARCH_LIMIT,
    page: 1
  });
  if (!packagePolicies.total) {
    return [];
  }

  // create a map of secret_references.id to package policy id
  const packagePoliciesBySecretId = packagePolicies.items.reduce((acc, packagePolicy) => {
    var _packagePolicy$secret;
    packagePolicy === null || packagePolicy === void 0 ? void 0 : (_packagePolicy$secret = packagePolicy.secret_references) === null || _packagePolicy$secret === void 0 ? void 0 : _packagePolicy$secret.forEach(secretReference => {
      if (Array.isArray(secretReference)) {
        secretReference.forEach(({
          id
        }) => {
          if (!acc[id]) {
            acc[id] = [];
          }
          acc[id].push(packagePolicy.id);
        });
      } else {
        if (!acc[secretReference.id]) {
          acc[secretReference.id] = [];
        }
        acc[secretReference.id].push(packagePolicy.id);
      }
    });
    return acc;
  }, {});
  const res = [];
  for (const id of ids) {
    if (packagePoliciesBySecretId[id]) {
      res.push({
        id,
        policyIds: packagePoliciesBySecretId[id]
      });
    }
  }
  return res;
}
async function deleteSecrets(opts) {
  const {
    esClient,
    ids
  } = opts;
  const logger = _app_context.appContextService.getLogger();
  const deletedRes = await Promise.all(ids.map(async id => {
    try {
      const getDeleteRes = await (0, _retry.retryTransientEsErrors)(() => esClient.transport.request({
        method: 'DELETE',
        path: `${_constants.SECRETS_ENDPOINT_PATH}/${id}`
      }), {
        logger
      });
      return {
        ...getDeleteRes,
        id
      };
    } catch (err) {
      const msg = `Error deleting secrets: ${err}`;
      logger.error(msg);
      throw new _errors.FleetError(msg);
    }
  }));
  deletedRes.forEach(item => {
    if (item.deleted === true) {
      _audit_logging.auditLoggingService.writeCustomAuditLog({
        message: `secret deleted: ${item.id}`,
        event: {
          action: 'secret_delete',
          category: ['database'],
          type: ['access'],
          outcome: 'success'
        }
      });
    }
  });
}
async function extractAndWriteSecrets(opts) {
  const {
    packagePolicy,
    packageInfo,
    esClient
  } = opts;
  const secretPaths = getPolicySecretPaths(packagePolicy, packageInfo);
  if (!secretPaths.length) {
    return {
      packagePolicy,
      secretReferences: []
    };
  }
  const secretsToCreate = secretPaths.filter(secretPath => !!secretPath.value.value);
  const secrets = await createSecrets({
    esClient,
    values: secretsToCreate.map(secretPath => secretPath.value.value)
  });
  const policyWithSecretRefs = getPolicyWithSecretReferences(secretsToCreate, secrets, packagePolicy);
  return {
    packagePolicy: policyWithSecretRefs,
    secretReferences: secrets.reduce((acc, secret) => {
      if (Array.isArray(secret)) {
        return [...acc, ...secret.map(({
          id
        }) => ({
          id
        }))];
      }
      return [...acc, {
        id: secret.id
      }];
    }, [])
  };
}
async function extractAndUpdateSecrets(opts) {
  const {
    oldPackagePolicy,
    packagePolicyUpdate,
    packageInfo,
    esClient
  } = opts;
  const oldSecretPaths = getPolicySecretPaths(oldPackagePolicy, packageInfo);
  const updatedSecretPaths = getPolicySecretPaths(packagePolicyUpdate, packageInfo);
  if (!oldSecretPaths.length && !updatedSecretPaths.length) {
    return {
      packagePolicyUpdate,
      secretReferences: [],
      secretsToDelete: []
    };
  }
  const {
    toCreate,
    toDelete,
    noChange
  } = diffSecretPaths(oldSecretPaths, updatedSecretPaths);
  const secretsToCreate = toCreate.filter(secretPath => !!secretPath.value.value);
  const createdSecrets = await createSecrets({
    esClient,
    values: secretsToCreate.map(secretPath => secretPath.value.value)
  });
  const policyWithSecretRefs = getPolicyWithSecretReferences(secretsToCreate, createdSecrets, packagePolicyUpdate);
  const secretReferences = [...noChange.reduce((acc, secretPath) => {
    if (secretPath.value.value.ids) {
      return [...acc, ...secretPath.value.value.ids.map(id => ({
        id
      }))];
    }
    return [...acc, {
      id: secretPath.value.value.id
    }];
  }, []), ...createdSecrets.reduce((acc, secret) => {
    if (Array.isArray(secret)) {
      return [...acc, ...secret.map(({
        id
      }) => ({
        id
      }))];
    }
    return [...acc, {
      id: secret.id
    }];
  }, [])];
  const secretsToDelete = [];
  toDelete.forEach(secretPath => {
    var _secretPath$value$val;
    // check if the previous secret is actually a secret refrerence
    // it may be that secrets were not enabled at the time of creation
    // in which case they are just stored as plain text
    if ((_secretPath$value$val = secretPath.value.value) !== null && _secretPath$value$val !== void 0 && _secretPath$value$val.isSecretRef) {
      if (secretPath.value.value.ids) {
        secretPath.value.value.ids.forEach(id => {
          secretsToDelete.push({
            id
          });
        });
      } else {
        secretsToDelete.push({
          id: secretPath.value.value.id
        });
      }
    }
  });
  return {
    packagePolicyUpdate: policyWithSecretRefs,
    secretReferences,
    secretsToDelete
  };
}
function isSecretVar(varDef) {
  return varDef.secret === true;
}
function containsSecretVar(vars) {
  return vars === null || vars === void 0 ? void 0 : vars.some(isSecretVar);
}

// this is how secrets are stored on the package policy
function toVarSecretRef(secret) {
  if (Array.isArray(secret)) {
    return {
      ids: secret.map(({
        id
      }) => id),
      isSecretRef: true
    };
  }
  return {
    id: secret.id,
    isSecretRef: true
  };
}

// this is how IDs are inserted into compiled templates
function toCompiledSecretRef(id) {
  return `$co.elastic.secret{${id}}`;
}
function diffSecretPaths(oldPaths, newPaths) {
  const toCreate = [];
  const toDelete = [];
  const noChange = [];
  const newPathsByPath = (0, _lodash.keyBy)(newPaths, x => x.path.join('.'));
  for (const oldPath of oldPaths) {
    if (!newPathsByPath[oldPath.path.join('.')]) {
      toDelete.push(oldPath);
    }
    const newPath = newPathsByPath[oldPath.path.join('.')];
    if (newPath && newPath.value.value) {
      var _newPath$value;
      const newValue = (_newPath$value = newPath.value) === null || _newPath$value === void 0 ? void 0 : _newPath$value.value;
      if (!(newValue !== null && newValue !== void 0 && newValue.isSecretRef)) {
        toCreate.push(newPath);
        toDelete.push(oldPath);
      } else {
        noChange.push(newPath);
      }
      delete newPathsByPath[oldPath.path.join('.')];
    }
  }
  const remainingNewPaths = Object.values(newPathsByPath);
  return {
    toCreate: [...toCreate, ...remainingNewPaths],
    toDelete,
    noChange
  };
}

// Given a package policy and a package,
// returns an array of lodash style paths to all secrets and their current values
function getPolicySecretPaths(packagePolicy, packageInfo) {
  var _packageInfo$policy_t;
  const packageLevelVarPaths = _getPackageLevelSecretPaths(packagePolicy, packageInfo);
  if (!(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t = packageInfo.policy_templates) !== null && _packageInfo$policy_t !== void 0 && _packageInfo$policy_t.length) || (0, _policy_template.packageHasNoPolicyTemplates)(packageInfo)) {
    return packageLevelVarPaths;
  }
  const inputSecretPaths = _getInputSecretPaths(packagePolicy, packageInfo);
  return [...packageLevelVarPaths, ...inputSecretPaths];
}
async function isSecretStorageEnabled(esClient, soClient) {
  var _appContextService$ge, _appContextService$ge2, _appContextService$ge3;
  const logger = _app_context.appContextService.getLogger();

  // if serverless then secrets will always be supported
  const isFleetServerStandalone = (_appContextService$ge = (_appContextService$ge2 = _app_context.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : (_appContextService$ge3 = _appContextService$ge2.internal) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.fleetServerStandalone) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : false;
  if (isFleetServerStandalone) {
    logger.trace('Secrets storage is enabled as fleet server is standalone');
    return true;
  }

  // now check the flag in settings to see if the fleet server requirement has already been met
  // once the requirement has been met, secrets are always on
  const settings = await _.settingsService.getSettingsOrUndefined(soClient);
  if (settings && settings.secret_storage_requirements_met) {
    logger.debug('Secrets storage requirements already met, turned on in settings');
    return true;
  }
  const areAllFleetServersOnProperVersion = await (0, _fleet_server.checkFleetServerVersionsForSecretsStorage)(esClient, soClient, _constants.SECRETS_MINIMUM_FLEET_SERVER_VERSION);

  // otherwise check if we have the minimum fleet server version and enable secrets if so
  if (areAllFleetServersOnProperVersion) {
    logger.debug('Enabling secrets storage as minimum fleet server version has been met');
    try {
      await _.settingsService.saveSettings(soClient, {
        secret_storage_requirements_met: true
      });
    } catch (err) {
      // we can suppress this error as it will be retried on the next function call
      logger.warn(`Failed to save settings after enabling secrets storage: ${err.message}`);
    }
    return true;
  }
  logger.info('Secrets storage is disabled as minimum fleet server version has not been met');
  return false;
}
async function isOutputSecretStorageEnabled(esClient, soClient) {
  var _appContextService$ge4, _appContextService$ge5, _appContextService$ge6;
  const logger = _app_context.appContextService.getLogger();

  // if serverless then output secrets will always be supported
  const isFleetServerStandalone = (_appContextService$ge4 = (_appContextService$ge5 = _app_context.appContextService.getConfig()) === null || _appContextService$ge5 === void 0 ? void 0 : (_appContextService$ge6 = _appContextService$ge5.internal) === null || _appContextService$ge6 === void 0 ? void 0 : _appContextService$ge6.fleetServerStandalone) !== null && _appContextService$ge4 !== void 0 ? _appContextService$ge4 : false;
  if (isFleetServerStandalone) {
    logger.trace('Output secrets storage is enabled as fleet server is standalone');
    return true;
  }

  // now check the flag in settings to see if the fleet server requirement has already been met
  // once the requirement has been met, output secrets are always on
  const settings = await _.settingsService.getSettingsOrUndefined(soClient);
  if (settings && settings.output_secret_storage_requirements_met) {
    logger.debug('Output secrets storage requirements already met, turned on in settings');
    return true;
  }

  // otherwise check if we have the minimum fleet server version and enable secrets if so
  if (await (0, _fleet_server.checkFleetServerVersionsForSecretsStorage)(esClient, soClient, _constants.OUTPUT_SECRETS_MINIMUM_FLEET_SERVER_VERSION)) {
    logger.debug('Enabling output secrets storage as minimum fleet server version has been met');
    try {
      await _.settingsService.saveSettings(soClient, {
        output_secret_storage_requirements_met: true
      });
    } catch (err) {
      // we can suppress this error as it will be retried on the next function call
      logger.warn(`Failed to save settings after enabling output secrets storage: ${err.message}`);
    }
    return true;
  }
  logger.info('Secrets storage is disabled as minimum fleet server version has not been met');
  return false;
}
function _getPackageLevelSecretPaths(packagePolicy, packageInfo) {
  var _packageInfo$vars;
  const packageSecretVars = ((_packageInfo$vars = packageInfo.vars) === null || _packageInfo$vars === void 0 ? void 0 : _packageInfo$vars.filter(isSecretVar)) || [];
  const packageSecretVarsByName = (0, _lodash.keyBy)(packageSecretVars, 'name');
  const packageVars = Object.entries(packagePolicy.vars || {});
  return packageVars.reduce((vars, [name, configEntry], i) => {
    if (packageSecretVarsByName[name]) {
      vars.push({
        value: configEntry,
        path: ['vars', name]
      });
    }
    return vars;
  }, []);
}
function _getInputSecretPaths(packagePolicy, packageInfo) {
  var _packageInfo$policy_t2;
  if (!(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t2 = packageInfo.policy_templates) !== null && _packageInfo$policy_t2 !== void 0 && _packageInfo$policy_t2.length)) return [];
  const inputSecretVarDefsByPolicyTemplateAndType = _getInputSecretVarDefsByPolicyTemplateAndType(packageInfo);
  const streamSecretVarDefsByDatasetAndInput = _getStreamSecretVarDefsByDatasetAndInput(packageInfo);
  return packagePolicy.inputs.flatMap((input, inputIndex) => {
    if (!input.vars && !input.streams) {
      return [];
    }
    const currentInputVarPaths = [];
    const inputKey = (0, _services.doesPackageHaveIntegrations)(packageInfo) ? `${input.policy_template}-${input.type}` : input.type;
    const inputVars = Object.entries(input.vars || {});
    if (inputVars.length) {
      inputVars.forEach(([name, configEntry]) => {
        var _inputSecretVarDefsBy;
        if ((_inputSecretVarDefsBy = inputSecretVarDefsByPolicyTemplateAndType[inputKey]) !== null && _inputSecretVarDefsBy !== void 0 && _inputSecretVarDefsBy[name]) {
          currentInputVarPaths.push({
            path: ['inputs', inputIndex.toString(), 'vars', name],
            value: configEntry
          });
        }
      });
    }
    if (input.streams.length) {
      input.streams.forEach((stream, streamIndex) => {
        const streamVarDefs = streamSecretVarDefsByDatasetAndInput[`${stream.data_stream.dataset}-${input.type}`];
        if (streamVarDefs && Object.keys(streamVarDefs).length) {
          Object.entries(stream.vars || {}).forEach(([name, configEntry]) => {
            if (streamVarDefs[name]) {
              currentInputVarPaths.push({
                path: ['inputs', inputIndex.toString(), 'streams', streamIndex.toString(), 'vars', name],
                value: configEntry
              });
            }
          });
        }
      });
    }
    return currentInputVarPaths;
  });
}

// a map of all secret vars for each dataset and input combo
function _getStreamSecretVarDefsByDatasetAndInput(packageInfo) {
  const dataStreams = (0, _services.getNormalizedDataStreams)(packageInfo);
  const streamsByDatasetAndInput = dataStreams.reduce((streams, dataStream) => {
    var _dataStream$streams;
    (_dataStream$streams = dataStream.streams) === null || _dataStream$streams === void 0 ? void 0 : _dataStream$streams.forEach(stream => {
      streams[`${dataStream.dataset}-${stream.input}`] = stream;
    });
    return streams;
  }, {});
  return Object.entries(streamsByDatasetAndInput).reduce((varDefs, [path, stream]) => {
    if (stream.vars && containsSecretVar(stream.vars)) {
      const secretVars = stream.vars.filter(isSecretVar);
      varDefs[path] = (0, _lodash.keyBy)(secretVars, 'name');
    }
    return varDefs;
  }, {});
}

// a map of all secret vars for each policyTemplate and input type combo
function _getInputSecretVarDefsByPolicyTemplateAndType(packageInfo) {
  var _packageInfo$policy_t3;
  if (!(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t3 = packageInfo.policy_templates) !== null && _packageInfo$policy_t3 !== void 0 && _packageInfo$policy_t3.length)) return {};
  const hasIntegrations = (0, _services.doesPackageHaveIntegrations)(packageInfo);
  return packageInfo.policy_templates.reduce((varDefs, policyTemplate) => {
    const inputs = (0, _services.getNormalizedInputs)(policyTemplate);
    inputs.forEach(input => {
      var _input$vars;
      const varDefKey = hasIntegrations ? `${policyTemplate.name}-${input.type}` : input.type;
      const secretVars = input === null || input === void 0 ? void 0 : (_input$vars = input.vars) === null || _input$vars === void 0 ? void 0 : _input$vars.filter(isSecretVar);
      if (secretVars !== null && secretVars !== void 0 && secretVars.length) {
        varDefs[varDefKey] = (0, _lodash.keyBy)(secretVars, 'name');
      }
    });
    return varDefs;
  }, {});
}

/**
 * Given an array of secret paths, existing secrets, and a package policy, generates a
 * new package policy object that includes resolved secret reference values at each
 * provided path.
 */
function getPolicyWithSecretReferences(secretPaths, secrets, packagePolicy) {
  const result = JSON.parse(JSON.stringify(packagePolicy));
  secretPaths.forEach((secretPath, secretPathIndex) => {
    secretPath.path.reduce((acc, val, secretPathComponentIndex) => {
      if (!acc[val]) {
        acc[val] = {};
      }
      const isLast = secretPathComponentIndex === secretPath.path.length - 1;
      if (isLast) {
        acc[val].value = toVarSecretRef(secrets[secretPathIndex]);
      }
      return acc[val];
    }, result);
  });
  return result;
}

/**
 * Common functions for SO objects
 * Currently used for outputs and fleet server hosts
 */

/**
 * diffSOSecretPaths
 * Makes the diff betwwen old and new secrets paths
 */
function diffSOSecretPaths(oldPaths, newPaths) {
  const toCreate = [];
  const toDelete = [];
  const noChange = [];
  const newPathsByPath = (0, _lodash.keyBy)(newPaths, 'path');
  for (const oldPath of oldPaths) {
    if (!newPathsByPath[oldPath.path]) {
      toDelete.push(oldPath);
    }
    const newPath = newPathsByPath[oldPath.path];
    if (newPath && newPath.value) {
      const newValue = newPath.value;
      if (typeof newValue === 'string') {
        toCreate.push(newPath);
        toDelete.push(oldPath);
      } else {
        noChange.push(newPath);
      }
    }
    delete newPathsByPath[oldPath.path];
  }
  const remainingNewPaths = Object.values(newPathsByPath);
  return {
    toCreate: [...toCreate, ...remainingNewPaths],
    toDelete,
    noChange
  };
}

/**
 * deleteSOSecrets
 * Given an array of secret paths, deletes the corresponding secrets
 */
async function deleteSOSecrets(esClient, secretPaths) {
  if (secretPaths.length === 0) {
    return Promise.resolve();
  }
  const secretIds = secretPaths.map(({
    value
  }) => value.id);
  try {
    return deleteSecrets({
      esClient,
      ids: secretIds
    });
  } catch (err) {
    _app_context.appContextService.getLogger().warn(`Error deleting secrets: ${err}`);
  }
}

/**
 * extractAndWriteSOSecrets
 * Takes a generic object T and its secret paths
 * Creates new secrets and returns the references
 */
async function extractAndWriteSOSecrets(opts) {
  const {
    soObject,
    esClient,
    secretPaths,
    secretHashes = {}
  } = opts;
  if (secretPaths.length === 0) {
    return {
      soObjectWithSecrets: soObject,
      secretReferences: []
    };
  }
  const secrets = await createSecrets({
    esClient,
    values: secretPaths.map(({
      value
    }) => value)
  });
  const objectWithSecretRefs = JSON.parse(JSON.stringify(soObject));
  secretPaths.forEach((secretPath, i) => {
    const pathWithoutPrefix = secretPath.path.replace('secrets.', '');
    const maybeHash = (0, _lodash.get)(secretHashes, pathWithoutPrefix);
    const currentSecret = secrets[i];
    (0, _saferLodashSet.set)(objectWithSecretRefs, secretPath.path, {
      ...(Array.isArray(currentSecret) ? {
        ids: currentSecret.map(({
          id
        }) => id)
      } : {
        id: currentSecret.id
      }),
      ...(typeof maybeHash === 'string' && {
        hash: maybeHash
      })
    });
  });
  return {
    soObjectWithSecrets: objectWithSecretRefs,
    secretReferences: secrets.reduce((acc, secret) => {
      if (Array.isArray(secret)) {
        return [...acc, ...secret.map(({
          id
        }) => ({
          id
        }))];
      }
      return [...acc, {
        id: secret.id
      }];
    }, [])
  };
}

/**
 * extractAndUpdateSOSecrets
 * Takes a generic object T to update and its old and new secret paths
 * Updates secrets and returns the references
 */
async function extractAndUpdateSOSecrets(opts) {
  const {
    updatedSoObject,
    oldSecretPaths,
    updatedSecretPaths,
    esClient,
    secretHashes
  } = opts;
  if (!oldSecretPaths.length && !updatedSecretPaths.length) {
    return {
      updatedSoObject,
      secretReferences: [],
      secretsToDelete: []
    };
  }
  const {
    toCreate,
    toDelete,
    noChange
  } = diffSOSecretPaths(oldSecretPaths, updatedSecretPaths);
  const createdSecrets = await createSecrets({
    esClient,
    values: toCreate.map(secretPath => secretPath.value)
  });
  const soObjectWithSecretRefs = JSON.parse(JSON.stringify(updatedSoObject));
  toCreate.forEach((secretPath, i) => {
    const pathWithoutPrefix = secretPath.path.replace('secrets.', '');
    const maybeHash = (0, _lodash.get)(secretHashes, pathWithoutPrefix);
    const currentSecret = createdSecrets[i];
    (0, _saferLodashSet.set)(soObjectWithSecretRefs, secretPath.path, {
      ...(Array.isArray(currentSecret) ? {
        ids: currentSecret.map(({
          id
        }) => id)
      } : {
        id: currentSecret.id
      }),
      ...(typeof maybeHash === 'string' && {
        hash: maybeHash
      })
    });
  });
  const secretReferences = [...noChange.reduce((acc, secretPath) => {
    const currentValue = secretPath.value;
    if ('ids' in currentValue) {
      return [...acc, ...currentValue.ids.map(id => ({
        id
      }))];
    } else {
      return [...acc, {
        id: currentValue.id
      }];
    }
  }, []), ...createdSecrets.reduce((acc, secret) => {
    if (Array.isArray(secret)) {
      return [...acc, ...secret.map(({
        id
      }) => ({
        id
      }))];
    }
    return [...acc, {
      id: secret.id
    }];
  }, [])];
  return {
    updatedSoObject: soObjectWithSecretRefs,
    secretReferences,
    secretsToDelete: toDelete.map(secretPath => ({
      id: secretPath.value.id
    }))
  };
}

// Outputs functions
async function extractAndWriteOutputSecrets(opts) {
  const {
    output,
    esClient,
    secretHashes = {}
  } = opts;
  const secretPaths = getOutputSecretPaths(output.type, output).filter(path => typeof path.value === 'string');
  const secretRes = await extractAndWriteSOSecrets({
    soObject: output,
    secretPaths,
    esClient,
    secretHashes
  });
  return {
    output: secretRes.soObjectWithSecrets,
    secretReferences: secretRes.secretReferences
  };
}
async function extractAndUpdateOutputSecrets(opts) {
  const {
    oldOutput,
    outputUpdate,
    esClient,
    secretHashes
  } = opts;
  const outputType = outputUpdate.type || oldOutput.type;
  const oldSecretPaths = getOutputSecretPaths(oldOutput.type, oldOutput);
  const updatedSecretPaths = getOutputSecretPaths(outputType, outputUpdate);
  const secretRes = await extractAndUpdateSOSecrets({
    updatedSoObject: outputUpdate,
    oldSecretPaths,
    updatedSecretPaths,
    esClient,
    secretHashes: outputUpdate.is_preconfigured ? secretHashes : undefined
  });
  return {
    outputUpdate: secretRes.updatedSoObject,
    secretReferences: secretRes.secretReferences,
    secretsToDelete: secretRes.secretsToDelete
  };
}
function getOutputSecretPaths(outputType, output) {
  var _output$secrets, _output$secrets$ssl;
  const outputSecretPaths = [];
  if (outputType === 'kafka') {
    var _kafkaOutput$secrets;
    const kafkaOutput = output;
    if (kafkaOutput !== null && kafkaOutput !== void 0 && (_kafkaOutput$secrets = kafkaOutput.secrets) !== null && _kafkaOutput$secrets !== void 0 && _kafkaOutput$secrets.password) {
      outputSecretPaths.push({
        path: 'secrets.password',
        value: kafkaOutput.secrets.password
      });
    }
  }
  if (outputType === 'remote_elasticsearch') {
    var _remoteESOutput$secre;
    const remoteESOutput = output;
    if ((_remoteESOutput$secre = remoteESOutput.secrets) !== null && _remoteESOutput$secre !== void 0 && _remoteESOutput$secre.service_token) {
      outputSecretPaths.push({
        path: 'secrets.service_token',
        value: remoteESOutput.secrets.service_token
      });
    }
  }

  // common to all outputs
  if (output !== null && output !== void 0 && (_output$secrets = output.secrets) !== null && _output$secrets !== void 0 && (_output$secrets$ssl = _output$secrets.ssl) !== null && _output$secrets$ssl !== void 0 && _output$secrets$ssl.key) {
    outputSecretPaths.push({
      path: 'secrets.ssl.key',
      value: output.secrets.ssl.key
    });
  }
  return outputSecretPaths;
}
async function deleteOutputSecrets(opts) {
  const {
    output,
    esClient
  } = opts;
  const outputType = output.type;
  const outputSecretPaths = getOutputSecretPaths(outputType, output);
  await deleteSOSecrets(esClient, outputSecretPaths);
}
function getOutputSecretReferences(output) {
  var _output$secrets2, _output$secrets2$ssl, _output$secrets3;
  const outputSecretPaths = [];
  if (typeof ((_output$secrets2 = output.secrets) === null || _output$secrets2 === void 0 ? void 0 : (_output$secrets2$ssl = _output$secrets2.ssl) === null || _output$secrets2$ssl === void 0 ? void 0 : _output$secrets2$ssl.key) === 'object') {
    outputSecretPaths.push({
      id: output.secrets.ssl.key.id
    });
  }
  if (output.type === 'kafka' && typeof (output === null || output === void 0 ? void 0 : (_output$secrets3 = output.secrets) === null || _output$secrets3 === void 0 ? void 0 : _output$secrets3.password) === 'object') {
    outputSecretPaths.push({
      id: output.secrets.password.id
    });
  }
  if (output.type === 'remote_elasticsearch') {
    var _output$secrets4;
    if (typeof (output === null || output === void 0 ? void 0 : (_output$secrets4 = output.secrets) === null || _output$secrets4 === void 0 ? void 0 : _output$secrets4.service_token) === 'object') {
      outputSecretPaths.push({
        id: output.secrets.service_token.id
      });
    }
  }
  return outputSecretPaths;
}
// Fleet server hosts functions
function getFleetServerHostsSecretPaths(fleetServerHost) {
  var _fleetServerHost$secr, _fleetServerHost$secr2, _fleetServerHost$secr3, _fleetServerHost$secr4;
  const secretPaths = [];
  if (fleetServerHost !== null && fleetServerHost !== void 0 && (_fleetServerHost$secr = fleetServerHost.secrets) !== null && _fleetServerHost$secr !== void 0 && (_fleetServerHost$secr2 = _fleetServerHost$secr.ssl) !== null && _fleetServerHost$secr2 !== void 0 && _fleetServerHost$secr2.key) {
    secretPaths.push({
      path: 'secrets.ssl.key',
      value: fleetServerHost.secrets.ssl.key
    });
  }
  if (fleetServerHost !== null && fleetServerHost !== void 0 && (_fleetServerHost$secr3 = fleetServerHost.secrets) !== null && _fleetServerHost$secr3 !== void 0 && (_fleetServerHost$secr4 = _fleetServerHost$secr3.ssl) !== null && _fleetServerHost$secr4 !== void 0 && _fleetServerHost$secr4.es_key) {
    secretPaths.push({
      path: 'secrets.ssl.es_key',
      value: fleetServerHost.secrets.ssl.es_key
    });
  }
  return secretPaths;
}
async function extractAndWriteFleetServerHostsSecrets(opts) {
  const {
    fleetServerHost,
    esClient,
    secretHashes = {}
  } = opts;
  const secretPaths = getFleetServerHostsSecretPaths(fleetServerHost);
  const secretRes = await extractAndWriteSOSecrets({
    soObject: fleetServerHost,
    secretPaths,
    esClient,
    secretHashes
  });
  return {
    fleetServerHost: secretRes.soObjectWithSecrets,
    secretReferences: secretRes.secretReferences
  };
}
async function extractAndUpdateFleetServerHostsSecrets(opts) {
  const {
    oldFleetServerHost,
    fleetServerHostUpdate,
    esClient,
    secretHashes
  } = opts;
  const oldSecretPaths = getFleetServerHostsSecretPaths(oldFleetServerHost);
  const updatedSecretPaths = getFleetServerHostsSecretPaths(fleetServerHostUpdate);
  const secretsRes = await extractAndUpdateSOSecrets({
    updatedSoObject: fleetServerHostUpdate,
    oldSecretPaths,
    updatedSecretPaths,
    esClient,
    secretHashes
  });
  return {
    fleetServerHostUpdate: secretsRes.updatedSoObject,
    secretReferences: secretsRes.secretReferences,
    secretsToDelete: secretsRes.secretsToDelete
  };
}
async function deleteFleetServerHostsSecrets(opts) {
  const {
    fleetServerHost,
    esClient
  } = opts;
  const secretPaths = getFleetServerHostsSecretPaths(fleetServerHost);
  await deleteSOSecrets(esClient, secretPaths);
}
function getFleetServerHostsSecretReferences(fleetServerHost) {
  var _fleetServerHost$secr5, _fleetServerHost$secr6, _fleetServerHost$secr7, _fleetServerHost$secr8;
  const secretPaths = [];
  if (typeof ((_fleetServerHost$secr5 = fleetServerHost.secrets) === null || _fleetServerHost$secr5 === void 0 ? void 0 : (_fleetServerHost$secr6 = _fleetServerHost$secr5.ssl) === null || _fleetServerHost$secr6 === void 0 ? void 0 : _fleetServerHost$secr6.key) === 'object') {
    secretPaths.push({
      id: fleetServerHost.secrets.ssl.key.id
    });
  }
  if (typeof ((_fleetServerHost$secr7 = fleetServerHost.secrets) === null || _fleetServerHost$secr7 === void 0 ? void 0 : (_fleetServerHost$secr8 = _fleetServerHost$secr7.ssl) === null || _fleetServerHost$secr8 === void 0 ? void 0 : _fleetServerHost$secr8.es_key) === 'object') {
    secretPaths.push({
      id: fleetServerHost.secrets.ssl.es_key.id
    });
  }
  return secretPaths;
}

// Download sources functions
function getDownloadSourcesSecretPaths(downloadSource) {
  var _downloadSource$secre, _downloadSource$secre2;
  const secretPaths = [];
  if (downloadSource !== null && downloadSource !== void 0 && (_downloadSource$secre = downloadSource.secrets) !== null && _downloadSource$secre !== void 0 && (_downloadSource$secre2 = _downloadSource$secre.ssl) !== null && _downloadSource$secre2 !== void 0 && _downloadSource$secre2.key) {
    secretPaths.push({
      path: 'secrets.ssl.key',
      value: downloadSource.secrets.ssl.key
    });
  }
  return secretPaths;
}
async function extractAndWriteDownloadSourcesSecrets(opts) {
  const {
    downloadSource,
    esClient,
    secretHashes = {}
  } = opts;
  const secretPaths = getFleetServerHostsSecretPaths(downloadSource).filter(path => typeof path.value === 'string');
  const secretRes = await extractAndWriteSOSecrets({
    soObject: downloadSource,
    secretPaths,
    esClient,
    secretHashes
  });
  return {
    downloadSource: secretRes.soObjectWithSecrets,
    secretReferences: secretRes.secretReferences
  };
}
async function extractAndUpdateDownloadSourceSecrets(opts) {
  const {
    oldDownloadSource,
    downloadSourceUpdate,
    esClient,
    secretHashes
  } = opts;
  const oldSecretPaths = getDownloadSourcesSecretPaths(oldDownloadSource);
  const updatedSecretPaths = getDownloadSourcesSecretPaths(downloadSourceUpdate);
  const secretsRes = await extractAndUpdateSOSecrets({
    updatedSoObject: downloadSourceUpdate,
    oldSecretPaths,
    updatedSecretPaths,
    esClient,
    secretHashes
  });
  return {
    downloadSourceUpdate: secretsRes.updatedSoObject,
    secretReferences: secretsRes.secretReferences,
    secretsToDelete: secretsRes.secretsToDelete
  };
}
async function deleteDownloadSourceSecrets(opts) {
  const {
    downloadSource,
    esClient
  } = opts;
  const secretPaths = getDownloadSourcesSecretPaths(downloadSource);
  await deleteSOSecrets(esClient, secretPaths);
}
function getDownloadSourceSecretReferences(downloadSource) {
  var _downloadSource$secre3, _downloadSource$secre4;
  const secretPaths = [];
  if (typeof ((_downloadSource$secre3 = downloadSource.secrets) === null || _downloadSource$secre3 === void 0 ? void 0 : (_downloadSource$secre4 = _downloadSource$secre3.ssl) === null || _downloadSource$secre4 === void 0 ? void 0 : _downloadSource$secre4.key) === 'object') {
    secretPaths.push({
      id: downloadSource.secrets.ssl.key.id
    });
  }
  return secretPaths;
}