"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.evaluateDefendInsights = void 0;
var _std = require("@kbn/std");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _server = require("@kbn/langchain/server");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _run_evaluations = require("./run_evaluations");
var _constants = require("../../attack_discovery/evaluation/constants");
var _utils = require("../../../routes/utils");
var _utils2 = require("../../../routes/evaluate/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const evaluateDefendInsights = async ({
  actionsClient,
  defendInsightsGraphs,
  anonymizationFields = _constants.DEFAULT_EVAL_ANONYMIZATION_FIELDS,
  // determines which fields are included in the alerts
  connectors,
  connectorTimeout,
  datasetName,
  esClient,
  esClientInternalUser,
  evaluationId,
  evaluatorConnectorId,
  langSmithApiKey,
  langSmithProject,
  logger,
  runName,
  size
}) => {
  await (0, _std.asyncForEach)(defendInsightsGraphs, async ({
    getDefaultDefendInsightsGraph
  }) => {
    // create a graph for every connector:
    const graphs = connectors.map(connector => {
      var _connector$config;
      const llmType = (0, _utils.getLlmType)(connector.actionTypeId);
      const traceOptions = {
        projectName: langSmithProject,
        tracers: [...(0, _langsmith.getLangSmithTracer)({
          apiKey: langSmithApiKey,
          projectName: langSmithProject,
          logger
        })]
      };
      const llm = new _server.ActionsClientLlm({
        actionsClient,
        connectorId: connector.id,
        llmType,
        logger,
        temperature: 0,
        // zero temperature for defend insights, because we want structured JSON output
        timeout: connectorTimeout,
        traceOptions,
        model: (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.defaultModel
      });
      const graph = getDefaultDefendInsightsGraph({
        insightType: _elasticAssistantCommon.DefendInsightType.Enum.incompatible_antivirus,
        // TODO: parameterize
        endpointIds: [],
        // Empty endpointIds, because we are seeding the graph with the dataset
        esClient,
        llm,
        logger,
        size,
        anonymizationFields,
        prompts: connector.prompts
      });
      return {
        connector,
        graph,
        llmType,
        name: `${runName} - ${connector.name} - ${evaluationId} - Defend Insights`,
        traceOptions
      };
    });

    // run the evaluations for each graph:
    await (0, _run_evaluations.runDefendInsightsEvaluations)({
      insightType: _elasticAssistantCommon.DefendInsightType.Enum.incompatible_antivirus,
      evaluatorConnectorId,
      datasetName,
      graphs,
      langSmithApiKey,
      logger
    });
  });
  await (0, _utils2.createOrUpdateEvaluationResults)({
    evaluationResults: [{
      id: evaluationId,
      status: _utils2.EvaluationStatus.COMPLETE
    }],
    esClientInternalUser,
    logger
  });
};
exports.evaluateDefendInsights = evaluateDefendInsights;