/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.geometry.R1Interval;
import com.google.common.geometry.S1Angle;
import com.google.common.geometry.S1Interval;
import com.google.common.geometry.S2;
import com.google.common.geometry.S2AreaCentroid;
import com.google.common.geometry.S2Cap;
import com.google.common.geometry.S2Cell;
import com.google.common.geometry.S2EdgeIndex;
import com.google.common.geometry.S2EdgeUtil;
import com.google.common.geometry.S2LatLng;
import com.google.common.geometry.S2LatLngRect;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2Region;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

public strictfp final class S2Loop
implements S2Region,
Comparable<S2Loop> {
    private static final Logger log = Logger.getLogger(S2Loop.class.getCanonicalName());
    public static final double MAX_INTERSECTION_ERROR = 1.0E-15;
    private S2EdgeIndex index;
    private Map<S2Point, Integer> vertexToIndex;
    private final S2Point[] vertices;
    private final int numVertices;
    private int firstLogicalVertex;
    private S2LatLngRect bound;
    private boolean originInside;
    private int depth;

    public S2Loop(List<S2Point> vertices) {
        this.numVertices = vertices.size();
        this.vertices = new S2Point[this.numVertices];
        this.bound = S2LatLngRect.full();
        this.depth = 0;
        vertices.toArray(this.vertices);
        this.initOrigin();
        this.initBound();
        this.initFirstLogicalVertex();
    }

    public S2Loop(S2Cell cell) {
        this(cell, cell.getRectBound());
    }

    public S2Loop(S2Cell cell, S2LatLngRect bound) {
        this.bound = bound;
        this.numVertices = 4;
        this.vertices = new S2Point[this.numVertices];
        this.vertexToIndex = null;
        this.index = null;
        this.depth = 0;
        for (int i = 0; i < 4; ++i) {
            this.vertices[i] = cell.getVertex(i);
        }
        this.initOrigin();
        this.initFirstLogicalVertex();
    }

    public S2Loop(S2Loop src) {
        this.numVertices = src.numVertices();
        this.vertices = (S2Point[])src.vertices.clone();
        this.vertexToIndex = src.vertexToIndex;
        this.index = src.index;
        this.firstLogicalVertex = src.firstLogicalVertex;
        this.bound = src.getRectBound();
        this.originInside = src.originInside;
        this.depth = src.depth();
    }

    public int depth() {
        return this.depth;
    }

    public void setDepth(int depth) {
        this.depth = depth;
    }

    public boolean isHole() {
        return (this.depth & 1) != 0;
    }

    public int sign() {
        return this.isHole() ? -1 : 1;
    }

    public int numVertices() {
        return this.numVertices;
    }

    public S2Point vertex(int i) {
        try {
            return this.vertices[i >= this.vertices.length ? i - this.vertices.length : i];
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new IllegalStateException("Invalid vertex index");
        }
    }

    @Override
    public int compareTo(S2Loop other) {
        if (this.numVertices() != other.numVertices()) {
            return this.numVertices() - other.numVertices();
        }
        int maxVertices = this.numVertices();
        int iThis = this.firstLogicalVertex;
        int iOther = other.firstLogicalVertex;
        int i = 0;
        while (i < maxVertices) {
            int compare = this.vertex(iThis).compareTo(other.vertex(iOther));
            if (compare != 0) {
                return compare;
            }
            ++i;
            ++iThis;
            ++iOther;
        }
        return 0;
    }

    private void initFirstLogicalVertex() {
        int first = 0;
        for (int i = 1; i < this.numVertices; ++i) {
            if (this.vertex(i).compareTo(this.vertex(first)) >= 0) continue;
            first = i;
        }
        this.firstLogicalVertex = first;
    }

    public boolean isNormalized() {
        return this.getArea() <= 6.283185307179596;
    }

    public void normalize() {
        if (!this.isNormalized()) {
            this.invert();
        }
    }

    public void invert() {
        int last = this.numVertices() - 1;
        for (int i = (last - 1) / 2; i >= 0; --i) {
            S2Point t = this.vertices[i];
            this.vertices[i] = this.vertices[last - i];
            this.vertices[last - i] = t;
        }
        this.vertexToIndex = null;
        this.index = null;
        this.originInside ^= true;
        if (this.bound.lat().lo() > -1.5707963267948966 && this.bound.lat().hi() < 1.5707963267948966) {
            this.bound = S2LatLngRect.full();
        } else {
            this.initBound();
        }
        this.initFirstLogicalVertex();
    }

    private S2AreaCentroid getAreaCentroid(boolean doCentroid) {
        S2Point centroid = null;
        if (this.numVertices() < 3) {
            return new S2AreaCentroid(0.0, centroid);
        }
        S2Point origin = this.vertex(0);
        int axis = (origin.largestAbsComponent() + 1) % 3;
        double slightlyDisplaced = origin.get(axis) + 2.718281828459045E-10;
        origin = new S2Point(axis == 0 ? slightlyDisplaced : origin.x, axis == 1 ? slightlyDisplaced : origin.y, axis == 2 ? slightlyDisplaced : origin.z);
        origin = S2Point.normalize(origin);
        double areaSum = 0.0;
        S2Point centroidSum = new S2Point(0.0, 0.0, 0.0);
        for (int i = 1; i <= this.numVertices(); ++i) {
            areaSum += S2.signedArea(origin, this.vertex(i - 1), this.vertex(i));
            if (!doCentroid) continue;
            S2Point trueCentroid = S2.trueCentroid(origin, this.vertex(i - 1), this.vertex(i));
            centroidSum = S2Point.add(centroidSum, trueCentroid);
        }
        if (areaSum < 0.0) {
            areaSum += Math.PI * 4;
        }
        if (doCentroid) {
            centroid = centroidSum;
        }
        return new S2AreaCentroid(areaSum, centroid);
    }

    public S2AreaCentroid getAreaAndCentroid() {
        return this.getAreaCentroid(true);
    }

    public double getArea() {
        return this.getAreaCentroid(false).getArea();
    }

    public S2Point getCentroid() {
        return this.getAreaCentroid(true).getCentroid();
    }

    public boolean contains(S2Loop b) {
        if (!this.bound.contains(b.getRectBound())) {
            return false;
        }
        if (!this.contains(b.vertex(0)) && this.findVertex(b.vertex(0)) < 0) {
            return false;
        }
        if (this.checkEdgeCrossings(b, new S2EdgeUtil.WedgeContains()) <= 0) {
            return false;
        }
        return !this.bound.union(b.getRectBound()).isFull() || !b.contains(this.vertex(0)) || b.findVertex(this.vertex(0)) >= 0;
    }

    public boolean intersects(S2Loop b) {
        if (!this.bound.intersects(b.getRectBound())) {
            return false;
        }
        if (b.getRectBound().lng().getLength() > this.bound.lng().getLength()) {
            return b.intersects(this);
        }
        if (this.contains(b.vertex(0)) && this.findVertex(b.vertex(0)) < 0) {
            return true;
        }
        if (this.checkEdgeCrossings(b, new S2EdgeUtil.WedgeIntersects()) < 0) {
            return true;
        }
        return b.getRectBound().contains(this.bound) && b.contains(this.vertex(0)) && b.findVertex(this.vertex(0)) < 0;
    }

    public boolean containsNested(S2Loop b) {
        if (!this.bound.contains(b.getRectBound())) {
            return false;
        }
        int m = this.findVertex(b.vertex(1));
        if (m < 0) {
            return this.contains(b.vertex(1));
        }
        return new S2EdgeUtil.WedgeContains().test(this.vertex(m - 1), this.vertex(m), this.vertex(m + 1), b.vertex(0), b.vertex(2)) > 0;
    }

    public int containsOrCrosses(S2Loop b) {
        if (!this.bound.intersects(b.getRectBound())) {
            return 0;
        }
        int result = this.checkEdgeCrossings(b, new S2EdgeUtil.WedgeContainsOrCrosses());
        if (result <= 0) {
            return result;
        }
        if (!this.bound.contains(b.getRectBound())) {
            return 0;
        }
        if (!this.contains(b.vertex(0)) && this.findVertex(b.vertex(0)) < 0) {
            return 0;
        }
        return 1;
    }

    boolean boundaryApproxEquals(S2Loop b, double maxError) {
        if (this.numVertices() != b.numVertices()) {
            return false;
        }
        int maxVertices = this.numVertices();
        int iThis = this.firstLogicalVertex;
        int iOther = b.firstLogicalVertex;
        int i = 0;
        while (i < maxVertices) {
            if (!S2.approxEquals(this.vertex(iThis), b.vertex(iOther), maxError)) {
                return false;
            }
            ++i;
            ++iThis;
            ++iOther;
        }
        return true;
    }

    @Override
    public S2Cap getCapBound() {
        return this.bound.getCapBound();
    }

    @Override
    public S2LatLngRect getRectBound() {
        return this.bound;
    }

    @Override
    public boolean contains(S2Cell cell) {
        S2LatLngRect cellBound = cell.getRectBound();
        if (!this.bound.contains(cellBound)) {
            return false;
        }
        S2Loop cellLoop = new S2Loop(cell, cellBound);
        return this.contains(cellLoop);
    }

    @Override
    public boolean mayIntersect(S2Cell cell) {
        S2LatLngRect cellBound = cell.getRectBound();
        if (!this.bound.intersects(cellBound)) {
            return false;
        }
        return new S2Loop(cell, cellBound).intersects(this);
    }

    public boolean contains(S2Point p) {
        if (!this.bound.contains(p)) {
            return false;
        }
        boolean inside = this.originInside;
        S2Point origin = S2.origin();
        S2EdgeUtil.EdgeCrosser crosser = new S2EdgeUtil.EdgeCrosser(origin, p, this.vertices[this.numVertices - 1]);
        if (this.numVertices < 2000) {
            for (int i = 0; i < this.numVertices; ++i) {
                inside ^= crosser.edgeOrVertexCrossing(this.vertices[i]);
            }
        } else {
            S2EdgeIndex.DataEdgeIterator it = this.getEdgeIterator(this.numVertices);
            int previousIndex = -2;
            it.getCandidates(origin, p);
            while (it.hasNext()) {
                int ai = it.index();
                if (previousIndex != ai - 1) {
                    crosser.restartAt(this.vertices[ai]);
                }
                previousIndex = ai;
                inside ^= crosser.edgeOrVertexCrossing(this.vertex(ai + 1));
                it.next();
            }
        }
        return inside;
    }

    public S1Angle getDistance(S2Point p) {
        S2Point normalized = S2Point.normalize(p);
        S1Angle minDistance = S1Angle.radians(Math.PI);
        for (int i = 0; i < this.numVertices(); ++i) {
            minDistance = S1Angle.min(minDistance, S2EdgeUtil.getDistance(normalized, this.vertex(i), this.vertex(i + 1)));
        }
        return minDistance;
    }

    private final S2EdgeIndex.DataEdgeIterator getEdgeIterator(int expectedQueries) {
        if (this.index == null) {
            this.index = new S2EdgeIndex(){

                @Override
                protected int getNumEdges() {
                    return S2Loop.this.numVertices;
                }

                @Override
                protected S2Point edgeFrom(int index) {
                    return S2Loop.this.vertex(index);
                }

                @Override
                protected S2Point edgeTo(int index) {
                    return S2Loop.this.vertex(index + 1);
                }
            };
        }
        this.index.predictAdditionalCalls(expectedQueries);
        return new S2EdgeIndex.DataEdgeIterator(this.index);
    }

    public boolean isValid() {
        if (this.numVertices < 3) {
            log.info("Degenerate loop");
            return false;
        }
        for (int i = 0; i < this.numVertices; ++i) {
            if (S2.isUnitLength(this.vertex(i))) continue;
            log.info("Vertex " + i + " is not unit length");
            return false;
        }
        HashMap vmap = Maps.newHashMap();
        for (int i = 0; i < this.numVertices; ++i) {
            Integer previousVertexIndex = vmap.put(this.vertex(i), i);
            if (previousVertexIndex == null) continue;
            log.info("Duplicate vertices: " + previousVertexIndex + " and " + i);
            return false;
        }
        boolean crosses = false;
        S2EdgeIndex.DataEdgeIterator it = this.getEdgeIterator(this.numVertices);
        for (int a1 = 0; a1 < this.numVertices; ++a1) {
            int a2 = (a1 + 1) % this.numVertices;
            S2EdgeUtil.EdgeCrosser crosser = new S2EdgeUtil.EdgeCrosser(this.vertex(a1), this.vertex(a2), this.vertex(0));
            int previousIndex = -2;
            it.getCandidates(this.vertex(a1), this.vertex(a2));
            while (it.hasNext()) {
                int b1 = it.index();
                int b2 = (b1 + 1) % this.numVertices;
                if (a1 != b2 && a2 != b1 && a1 != b1) {
                    boolean abdNearlyLinear;
                    double abc = S2.angle(this.vertex(a1), this.vertex(a2), this.vertex(b1));
                    boolean abcNearlyLinear = S2.approxEquals(abc, 0.0, 1.0E-15) || S2.approxEquals(abc, Math.PI, 1.0E-15);
                    double abd = S2.angle(this.vertex(a1), this.vertex(a2), this.vertex(b2));
                    boolean bl = abdNearlyLinear = S2.approxEquals(abd, 0.0, 1.0E-15) || S2.approxEquals(abd, Math.PI, 1.0E-15);
                    if (!abcNearlyLinear || !abdNearlyLinear) {
                        if (previousIndex != b1) {
                            crosser.restartAt(this.vertex(b1));
                        }
                        crosses = crosser.robustCrossing(this.vertex(b2)) > 0;
                        previousIndex = b2;
                        if (crosses) {
                            log.info("Edges " + a1 + " and " + b1 + " cross");
                            log.info(String.format("Edge locations in degrees: %s-%s and %s-%s", new S2LatLng(this.vertex(a1)).toStringDegrees(), new S2LatLng(this.vertex(a2)).toStringDegrees(), new S2LatLng(this.vertex(b1)).toStringDegrees(), new S2LatLng(this.vertex(b2)).toStringDegrees()));
                            return false;
                        }
                    }
                }
                it.next();
            }
        }
        return true;
    }

    public static boolean isValid(List<S2Point> vertices) {
        return new S2Loop(vertices).isValid();
    }

    public String toString() {
        StringBuilder builder = new StringBuilder("S2Loop, ");
        builder.append(this.vertices.length).append(" points. [");
        for (S2Point v : this.vertices) {
            builder.append(v.toString()).append(" ");
        }
        builder.append("]");
        return builder.toString();
    }

    private void initOrigin() {
        Preconditions.checkState((boolean)this.bound.contains(this.vertex(1)));
        this.originInside = false;
        boolean v1Inside = S2.orderedCCW(S2.ortho(this.vertex(1)), this.vertex(0), this.vertex(2), this.vertex(1));
        if (v1Inside != this.contains(this.vertex(1))) {
            this.originInside = true;
        }
    }

    private void initBound() {
        S2EdgeUtil.RectBounder bounder = new S2EdgeUtil.RectBounder();
        for (int i = 0; i <= this.numVertices(); ++i) {
            bounder.addPoint(this.vertex(i));
        }
        S2LatLngRect b = bounder.getBound();
        this.bound = S2LatLngRect.full();
        if (this.contains(new S2Point(0.0, 0.0, 1.0))) {
            b = new S2LatLngRect(new R1Interval(b.lat().lo(), 1.5707963267948966), S1Interval.full());
        }
        if (b.lng().isFull() && this.contains(new S2Point(0.0, 0.0, -1.0))) {
            b = new S2LatLngRect(new R1Interval(-1.5707963267948966, b.lat().hi()), b.lng());
        }
        this.bound = b;
    }

    private int findVertex(S2Point p) {
        Integer index;
        if (this.vertexToIndex == null) {
            this.vertexToIndex = new HashMap<S2Point, Integer>();
            for (int i = 1; i <= this.numVertices; ++i) {
                this.vertexToIndex.put(this.vertex(i), i);
            }
        }
        if ((index = this.vertexToIndex.get(p)) == null) {
            return -1;
        }
        return index;
    }

    private int checkEdgeCrossings(S2Loop b, S2EdgeUtil.WedgeRelation relation) {
        S2EdgeIndex.DataEdgeIterator it = this.getEdgeIterator(b.numVertices);
        int result = 1;
        for (int j = 0; j < b.numVertices(); ++j) {
            S2EdgeUtil.EdgeCrosser crosser = new S2EdgeUtil.EdgeCrosser(b.vertex(j), b.vertex(j + 1), this.vertex(0));
            int previousIndex = -2;
            it.getCandidates(b.vertex(j), b.vertex(j + 1));
            while (it.hasNext()) {
                int i = it.index();
                if (previousIndex != i - 1) {
                    crosser.restartAt(this.vertex(i));
                }
                previousIndex = i;
                int crossing = crosser.robustCrossing(this.vertex(i + 1));
                if (crossing >= 0) {
                    if (crossing > 0) {
                        return -1;
                    }
                    if (this.vertex(i + 1).equals(b.vertex(j + 1)) && (result = Math.min(result, relation.test(this.vertex(i), this.vertex(i + 1), this.vertex(i + 2), b.vertex(j), b.vertex(j + 2)))) < 0) {
                        return result;
                    }
                }
                it.next();
            }
        }
        return result;
    }
}

