"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Timefilter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _rxjs = require("rxjs");
var _moment = _interopRequireDefault(require("moment"));
var _diff_time_picker_vals = require("./lib/diff_time_picker_vals");
var _common = require("../../../common");
var _auto_refresh_loop = require("./lib/auto_refresh_loop");
var _use_timefilter = require("./use_timefilter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class Timefilter {
  constructor(config, timeHistory, nowProvider) {
    // Fired when isTimeRangeSelectorEnabled \ isAutoRefreshSelectorEnabled are toggled
    (0, _defineProperty2.default)(this, "enabledUpdated$", new _rxjs.BehaviorSubject(false));
    // Fired when a user changes the timerange
    (0, _defineProperty2.default)(this, "timeUpdate$", new _rxjs.Subject());
    // Fired when a user changes the the autorefresh settings
    (0, _defineProperty2.default)(this, "refreshIntervalUpdate$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "fetch$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "_time", void 0);
    // Denotes whether setTime has been called, can be used to determine if the constructor defaults are being used.
    (0, _defineProperty2.default)(this, "_isTimeTouched", false);
    (0, _defineProperty2.default)(this, "_refreshInterval", void 0);
    (0, _defineProperty2.default)(this, "_minRefreshInterval", void 0);
    // Denotes whether the refresh interval defaults were overriden.
    (0, _defineProperty2.default)(this, "_isRefreshIntervalTouched", false);
    (0, _defineProperty2.default)(this, "_history", void 0);
    (0, _defineProperty2.default)(this, "_isTimeRangeSelectorEnabled", false);
    (0, _defineProperty2.default)(this, "_isAutoRefreshSelectorEnabled", false);
    (0, _defineProperty2.default)(this, "timeDefaults", void 0);
    (0, _defineProperty2.default)(this, "refreshIntervalDefaults", void 0);
    (0, _defineProperty2.default)(this, "useTimefilter", void 0);
    // Used when an auto refresh is triggered
    (0, _defineProperty2.default)(this, "autoRefreshLoop", (0, _auto_refresh_loop.createAutoRefreshLoop)());
    (0, _defineProperty2.default)(this, "getEnabledUpdated$", () => {
      return this.enabledUpdated$.asObservable();
    });
    (0, _defineProperty2.default)(this, "getTimeUpdate$", () => {
      return this.timeUpdate$.asObservable();
    });
    (0, _defineProperty2.default)(this, "getRefreshIntervalUpdate$", () => {
      return this.refreshIntervalUpdate$.asObservable();
    });
    /**
     * Get an observable that emits when it is time to refetch data due to refresh interval
     * Each subscription to this observable resets internal interval
     * Emitted value is a callback {@link AutoRefreshDoneFn} that must be called to restart refresh interval loop
     * Apps should use this callback to start next auto refresh loop when view finished updating
     */
    (0, _defineProperty2.default)(this, "getAutoRefreshFetch$", () => this.autoRefreshLoop.loop$);
    (0, _defineProperty2.default)(this, "triggerFetch", () => {
      this.fetch$.next();
    });
    (0, _defineProperty2.default)(this, "getFetch$", () => {
      return this.fetch$.asObservable();
    });
    (0, _defineProperty2.default)(this, "getTime", () => {
      const {
        from,
        to
      } = this._time;
      return {
        ...this._time,
        from: _moment.default.isMoment(from) ? from.toISOString() : from,
        to: _moment.default.isMoment(to) ? to.toISOString() : to
      };
    });
    /**
     * Updates timefilter time.
     * Emits 'timeUpdate' and 'fetch' events when time changes
     * @param {Object} time
     * @property {string|moment} time.from
     * @property {string|moment} time.to
     */
    (0, _defineProperty2.default)(this, "setTime", time => {
      // Object.assign used for partially composed updates
      const newTime = Object.assign(this.getTime(), time);
      if ((0, _diff_time_picker_vals.areTimeRangesDifferent)(this.getTime(), newTime)) {
        this._time = {
          from: newTime.from,
          to: newTime.to
        };
        this._isTimeTouched = true;
        this._history.add(this._time);
        this.timeUpdate$.next();
        this.fetch$.next();
      }
    });
    (0, _defineProperty2.default)(this, "getRefreshInterval", () => {
      return _lodash.default.clone(this._refreshInterval);
    });
    (0, _defineProperty2.default)(this, "getMinRefreshInterval", () => {
      return this._minRefreshInterval;
    });
    /**
     * Set timefilter refresh interval.
     * @param {Object} refreshInterval
     * @property {number} time.value Refresh interval in milliseconds. Positive integer
     * @property {boolean} time.pause
     */
    (0, _defineProperty2.default)(this, "setRefreshInterval", refreshInterval => {
      const prevRefreshInterval = this.getRefreshInterval();
      const newRefreshInterval = {
        ...prevRefreshInterval,
        ...refreshInterval
      };
      if (newRefreshInterval.value === 0) {
        // override only when explicitly set to 0
        newRefreshInterval.pause = true;
      }
      if (newRefreshInterval.value < this._minRefreshInterval) {
        newRefreshInterval.value = this._minRefreshInterval;
      }
      let shouldUnpauseRefreshLoop = newRefreshInterval.pause === false && prevRefreshInterval != null;
      if ((prevRefreshInterval === null || prevRefreshInterval === void 0 ? void 0 : prevRefreshInterval.value) > 0 && newRefreshInterval.value <= 0) {
        shouldUnpauseRefreshLoop = false;
      }
      this._isRefreshIntervalTouched = this._isRefreshIntervalTouched || (0, _diff_time_picker_vals.areRefreshIntervalsDifferent)(this.refreshIntervalDefaults, newRefreshInterval);

      // If the refresh interval is <= 0 handle that as a paused refresh
      // unless the user has un-paused the refresh loop and the value is not going from > 0 to 0
      if (newRefreshInterval.value <= 0) {
        newRefreshInterval.value = 0;
        newRefreshInterval.pause = shouldUnpauseRefreshLoop ? false : true;
      }
      this._refreshInterval = {
        value: newRefreshInterval.value,
        pause: newRefreshInterval.pause
      };
      // Only send out an event if we already had a previous refresh interval (not for the initial set)
      // and the old and new refresh interval are actually different.
      if (prevRefreshInterval && (0, _diff_time_picker_vals.areRefreshIntervalsDifferent)(prevRefreshInterval, newRefreshInterval)) {
        this.refreshIntervalUpdate$.next();
        if (!newRefreshInterval.pause && newRefreshInterval.value !== 0) {
          this.fetch$.next();
        }
      }
      this.autoRefreshLoop.stop();
      if (!newRefreshInterval.pause && newRefreshInterval.value !== 0) {
        this.autoRefreshLoop.start(newRefreshInterval.value);
      }
    });
    /**
     * Create a time filter that coerces all time values to absolute time.
     *
     * This is useful for creating a filter that ensures all ES queries will fetch the exact same data
     * and leverages ES query cache for performance improvement.
     *
     * One use case is keeping different elements embedded in the same UI in sync.
     */
    (0, _defineProperty2.default)(this, "createFilter", (indexPattern, timeRange) => {
      return (0, _common.getTime)(indexPattern, timeRange ? timeRange : this._time, {
        forceNow: this.nowProvider.get()
      });
    });
    /**
     * Create a time filter that converts only absolute time to ISO strings, it leaves relative time
     * values unchanged (e.g. "now-1").
     *
     * This is useful for sending datemath values to ES endpoints to generate reports over time.
     *
     * @note Consumers of this function need to ensure that the ES endpoint supports datemath.
     */
    (0, _defineProperty2.default)(this, "createRelativeFilter", (indexPattern, timeRange) => {
      return (0, _common.getRelativeTime)(indexPattern, timeRange ? timeRange : this._time, {
        forceNow: this.nowProvider.get()
      });
    });
    /**
     * Show the time bounds selector part of the time filter
     */
    (0, _defineProperty2.default)(this, "enableTimeRangeSelector", () => {
      this._isTimeRangeSelectorEnabled = true;
      this.enabledUpdated$.next(true);
    });
    /**
     * Hide the time bounds selector part of the time filter
     */
    (0, _defineProperty2.default)(this, "disableTimeRangeSelector", () => {
      this._isTimeRangeSelectorEnabled = false;
      this.enabledUpdated$.next(false);
    });
    /**
     * Show the auto refresh part of the time filter
     */
    (0, _defineProperty2.default)(this, "enableAutoRefreshSelector", () => {
      this._isAutoRefreshSelectorEnabled = true;
      this.enabledUpdated$.next(true);
    });
    /**
     * Hide the auto refresh part of the time filter
     */
    (0, _defineProperty2.default)(this, "disableAutoRefreshSelector", () => {
      this._isAutoRefreshSelectorEnabled = false;
      this.enabledUpdated$.next(false);
    });
    this.nowProvider = nowProvider;
    this._history = timeHistory;
    this.timeDefaults = config.timeDefaults;

    // Initialize 0ms intervals with pause set to true and min value
    this.refreshIntervalDefaults = {
      pause: config.refreshIntervalDefaults.value === 0 ? true : config.refreshIntervalDefaults.pause,
      value: Math.max(config.refreshIntervalDefaults.value, config.minRefreshIntervalDefault)
    };
    this._minRefreshInterval = config.minRefreshIntervalDefault;
    this._time = config.timeDefaults;
    this.setRefreshInterval(config.refreshIntervalDefaults);
    this.useTimefilter = (0, _use_timefilter.createUseTimefilterHook)(this, nowProvider);
  }
  isTimeRangeSelectorEnabled() {
    return this._isTimeRangeSelectorEnabled;
  }
  isAutoRefreshSelectorEnabled() {
    return this._isAutoRefreshSelectorEnabled;
  }
  isTimeTouched() {
    return this._isTimeTouched;
  }
  isRefreshIntervalTouched() {
    return this._isRefreshIntervalTouched;
  }
  /**
   * Same as {@link getTime}, but also converts relative time range to absolute time range
   */
  getAbsoluteTime() {
    return (0, _common.getAbsoluteTimeRange)(this._time, {
      forceNow: this.nowProvider.get()
    });
  }
  getBounds() {
    return this.calculateBounds(this._time);
  }
  calculateBounds(timeRange) {
    return (0, _common.calculateBounds)(timeRange, {
      forceNow: this.nowProvider.get()
    });
  }
  getActiveBounds() {
    if (this.isTimeRangeSelectorEnabled()) {
      return this.getBounds();
    }
  }
  getTimeDefaults() {
    return _lodash.default.cloneDeep(this.timeDefaults);
  }
  getRefreshIntervalDefaults() {
    return _lodash.default.cloneDeep(this.refreshIntervalDefaults);
  }
}
exports.Timefilter = Timefilter;