"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dashboardAttributesOut = dashboardAttributesOut;
exports.itemAttrsToSavedObjectWithTags = exports.itemAttrsToSavedObject = exports.getResultV3ToV2 = void 0;
exports.savedObjectToItem = savedObjectToItem;
var _lodash = require("lodash");
var _transforms = require("./transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function dashboardAttributesOut(attributes, references, getTagNamesFromReferences) {
  const {
    controlGroupInput,
    description,
    kibanaSavedObjectMeta,
    optionsJSON,
    panelsJSON,
    sections,
    refreshInterval,
    timeFrom,
    timeRestore,
    timeTo,
    title,
    version
  } = attributes;
  // Inject any tag names from references into the attributes
  let tags;
  if (getTagNamesFromReferences && references && references.length) {
    tags = getTagNamesFromReferences(references);
  }

  // try to maintain a consistent (alphabetical) order of keys
  return {
    ...(controlGroupInput && {
      controlGroupInput: (0, _transforms.transformControlGroupOut)(controlGroupInput)
    }),
    ...(description && {
      description
    }),
    ...(kibanaSavedObjectMeta && {
      kibanaSavedObjectMeta: (0, _transforms.transformSearchSourceOut)(kibanaSavedObjectMeta)
    }),
    ...(optionsJSON && {
      options: (0, _transforms.transformOptionsOut)(optionsJSON)
    }),
    ...((panelsJSON || sections) && {
      panels: (0, _transforms.transformPanelsOut)(panelsJSON, sections, references)
    }),
    ...(refreshInterval && {
      refreshInterval: {
        pause: refreshInterval.pause,
        value: refreshInterval.value
      }
    }),
    ...(tags && tags.length && {
      tags
    }),
    ...(timeFrom && {
      timeFrom
    }),
    timeRestore: timeRestore !== null && timeRestore !== void 0 ? timeRestore : false,
    ...(timeTo && {
      timeTo
    }),
    title,
    ...(version && {
      version
    })
  };
}
const getResultV3ToV2 = result => {
  const {
    meta,
    item
  } = result;
  const {
    attributes,
    ...rest
  } = item;
  const {
    controlGroupInput,
    description,
    kibanaSavedObjectMeta,
    options,
    panels,
    refreshInterval,
    timeFrom,
    timeRestore,
    timeTo,
    title,
    version
  } = attributes;
  const v2Attributes = {
    ...(controlGroupInput && {
      controlGroupInput: (0, _transforms.transformControlGroupIn)(controlGroupInput)
    }),
    description,
    ...(kibanaSavedObjectMeta && {
      kibanaSavedObjectMeta: (0, _transforms.transformSearchSourceIn)(kibanaSavedObjectMeta)
    }),
    ...(options && {
      optionsJSON: JSON.stringify(options)
    }),
    panelsJSON: panels ? (0, _transforms.transformPanelsIn)(panels, true).panelsJSON : '[]',
    refreshInterval,
    ...(timeFrom && {
      timeFrom
    }),
    timeRestore,
    ...(timeTo && {
      timeTo
    }),
    title,
    ...(version && {
      version
    })
  };
  return {
    meta,
    item: {
      ...rest,
      attributes: v2Attributes
    }
  };
};
exports.getResultV3ToV2 = getResultV3ToV2;
const itemAttrsToSavedObject = ({
  attributes,
  incomingReferences = []
}) => {
  try {
    const {
      controlGroupInput,
      kibanaSavedObjectMeta,
      options,
      panels,
      tags,
      ...rest
    } = attributes;
    const {
      panelsJSON,
      sections
    } = (0, _transforms.transformPanelsIn)(panels);
    const soAttributes = {
      ...rest,
      ...(controlGroupInput && {
        controlGroupInput: (0, _transforms.transformControlGroupIn)(controlGroupInput)
      }),
      ...(options && {
        optionsJSON: JSON.stringify(options)
      }),
      ...(panels && {
        panelsJSON
      }),
      ...((sections === null || sections === void 0 ? void 0 : sections.length) && {
        sections
      }),
      ...(kibanaSavedObjectMeta && {
        kibanaSavedObjectMeta: (0, _transforms.transformSearchSourceIn)(kibanaSavedObjectMeta)
      })
    };
    return {
      attributes: soAttributes,
      references: incomingReferences,
      error: null
    };
  } catch (e) {
    return {
      attributes: null,
      references: null,
      error: e
    };
  }
};
exports.itemAttrsToSavedObject = itemAttrsToSavedObject;
const itemAttrsToSavedObjectWithTags = async ({
  attributes,
  replaceTagReferencesByName,
  incomingReferences = []
}) => {
  const {
    tags,
    ...restAttributes
  } = attributes;
  // Tags can be specified as an attribute or in the incomingReferences.
  const soReferences = replaceTagReferencesByName && tags && tags.length ? await replaceTagReferencesByName({
    references: incomingReferences,
    newTagNames: tags
  }) : incomingReferences;
  return itemAttrsToSavedObject({
    attributes: restAttributes,
    incomingReferences: soReferences
  });
};
exports.itemAttrsToSavedObjectWithTags = itemAttrsToSavedObjectWithTags;
function savedObjectToItem(savedObject, partial, {
  allowedAttributes,
  allowedReferences,
  getTagNamesFromReferences
} = {}) {
  const {
    id,
    type,
    updated_at: updatedAt,
    updated_by: updatedBy,
    created_at: createdAt,
    created_by: createdBy,
    attributes,
    error,
    namespaces,
    references,
    version,
    managed
  } = savedObject;
  try {
    const attributesOut = allowedAttributes ? (0, _lodash.pick)(dashboardAttributesOut(attributes, references, getTagNamesFromReferences), allowedAttributes) : dashboardAttributesOut(attributes, references, getTagNamesFromReferences);

    // if includeReferences is provided, only include references of those types
    const referencesOut = allowedReferences ? references === null || references === void 0 ? void 0 : references.filter(reference => allowedReferences.includes(reference.type)) : references;
    return {
      item: {
        id,
        type,
        updatedAt,
        updatedBy,
        createdAt,
        createdBy,
        attributes: attributesOut,
        error,
        namespaces,
        references: referencesOut,
        version,
        managed
      },
      error: null
    };
  } catch (e) {
    return {
      item: null,
      error: e
    };
  }
}