"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLatestFindings = exports.getFindingsQuery = void 0;
var _reactQuery = require("@kbn/react-query");
var _ioTs = require("io-ts");
var _rxjs = require("rxjs");
var _discoverUtils = require("@kbn/discover-utils");
var _cloudSecurityPosture = require("@kbn/cloud-security-posture");
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _use_get_benchmark_rules_state_api = require("@kbn/cloud-security-posture/src/hooks/use_get_benchmark_rules_state_api");
var _constants = require("../../../common/constants");
var _use_kibana = require("../../../common/hooks/use_kibana");
var _utils = require("../utils/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getRuntimeMappingsFromSort = sort => {
  return sort.filter(([field]) => _constants.CDR_MISCONFIGURATION_DATA_TABLE_RUNTIME_MAPPING_FIELDS.includes(field)).reduce((acc, [field]) => {
    const type = 'keyword';
    return {
      ...acc,
      [field]: {
        type
      }
    };
  }, {});
};
const getFindingsQuery = ({
  query,
  sort
}, rulesStates, pageParam) => {
  var _query$bool$filter, _query$bool, _query$bool$must_not, _query$bool2;
  const mutedRulesFilterQuery = (0, _cloudSecurityPostureCommon.buildMutedRulesFilter)(rulesStates);
  return {
    index: _cloudSecurityPostureCommon.CDR_MISCONFIGURATIONS_INDEX_PATTERN,
    sort: getMultiFieldsSort(sort),
    runtime_mappings: getRuntimeMappingsFromSort(sort),
    size: _cloudSecurityPostureCommon.MAX_FINDINGS_TO_LOAD,
    aggs: (0, _utils.getFindingsCountAggQuery)(),
    ignore_unavailable: true,
    query: {
      ...query,
      bool: {
        ...(query === null || query === void 0 ? void 0 : query.bool),
        filter: [...((_query$bool$filter = query === null || query === void 0 ? void 0 : (_query$bool = query.bool) === null || _query$bool === void 0 ? void 0 : _query$bool.filter) !== null && _query$bool$filter !== void 0 ? _query$bool$filter : []), {
          range: {
            '@timestamp': {
              gte: `now-${_cloudSecurityPostureCommon.LATEST_FINDINGS_RETENTION_POLICY}`,
              lte: 'now'
            }
          }
        }],
        must_not: [...((_query$bool$must_not = query === null || query === void 0 ? void 0 : (_query$bool2 = query.bool) === null || _query$bool2 === void 0 ? void 0 : _query$bool2.must_not) !== null && _query$bool$must_not !== void 0 ? _query$bool$must_not : []), ...mutedRulesFilterQuery]
      }
    },
    ...(pageParam ? {
      from: pageParam
    } : {})
  };
};
exports.getFindingsQuery = getFindingsQuery;
const getMultiFieldsSort = sort => {
  return sort.map(([id, direction]) => {
    return {
      ...getSortField({
        field: id,
        direction
      })
    };
  });
};

/**
 * By default, ES will sort keyword fields in case-sensitive format, the
 * following fields are required to have a case-insensitive sorting.
 */
const fieldsRequiredSortingByPainlessScript = ['rule.section', 'resource.name', 'resource.sub_type'];

/**
 * Generates Painless sorting if the given field is matched or returns default sorting
 * This painless script will sort the field in case-insensitive manner
 */
const getSortField = ({
  field,
  direction
}) => {
  if (fieldsRequiredSortingByPainlessScript.includes(field)) {
    return {
      _script: {
        type: 'string',
        order: direction,
        script: {
          source: `doc["${field}"].value.toLowerCase()`,
          lang: 'painless'
        }
      }
    };
  }
  return {
    [field]: direction
  };
};
const useLatestFindings = options => {
  const {
    data,
    notifications: {
      toasts
    }
  } = (0, _use_kibana.useKibana)().services;
  const {
    data: rulesStates
  } = (0, _use_get_benchmark_rules_state_api.useGetCspBenchmarkRulesStatesApi)();

  /**
   * We're using useInfiniteQuery in this case to allow the user to fetch more data (if available and up to 10k)
   * useInfiniteQuery differs from useQuery because it accumulates and caches a chunk of data from the previous fetches into an array
   * it uses the getNextPageParam to know if there are more pages to load and retrieve the position of
   * the last loaded record to be used as a from parameter to fetch the next chunk of data.
   */
  return (0, _reactQuery.useInfiniteQuery)(['csp_findings', {
    params: options
  }, rulesStates], async ({
    pageParam
  }) => {
    const {
      rawResponse: {
        hits,
        aggregations
      }
    } = await (0, _rxjs.lastValueFrom)(data.search.search({
      params: getFindingsQuery(options, rulesStates, pageParam) // ruleStates always exists since it under the `enabled` dependency.
    }));
    if (!aggregations) throw new Error('expected aggregations to be an defined');
    if (!Array.isArray(aggregations.count.buckets)) throw new Error('expected buckets to be an array');
    return {
      page: hits.hits.map(hit => (0, _discoverUtils.buildDataTableRecord)(hit)),
      total: _ioTs.number.is(hits.total) ? hits.total : 0,
      count: (0, _utils.getAggregationCount)(aggregations.count.buckets)
    };
  }, {
    enabled: options.enabled && !!rulesStates,
    keepPreviousData: true,
    onError: err => (0, _cloudSecurityPosture.showErrorToast)(toasts, err),
    getNextPageParam: (lastPage, allPages) => {
      if (lastPage.page.length < options.pageSize) {
        return undefined;
      }
      return allPages.length * options.pageSize;
    }
  });
};
exports.useLatestFindings = useLatestFindings;